#! /bin/bash
##
## common functions used by boot-floppies *.sh scripts
##

# env. vars you can set, and what they mean
#   tmpdir     -- where to build
#   arch       -- architecture, otherwise, autosense
#   verbose    -- be more verbose in output
#   debug      -- the above, and set -x
#   ignorefail -- ignore failures in subshells

# you can set tmpdir
tmpdir=${tmpdir:-/var/tmp}

# architecture is set in top-level Makefile's config; use that if we have it
arch=${architecture-$(dpkg --print-architecture)}

# sensible default -- if customizing, set this in 'config' please
MIRRORS=${MIRRORS:-http://http.us.debian.org/debian}
export MIRRORS

# what distro we use for base -- should be "woody" or "sid"
# we use "sid" for now until woody base is frozen
export SUITE="woody"

# abort on any non-zero exit value
set -e


##
## trap exit to make errors more obvious
##
settrapzero () {
    trap "if [ \$? -ne 0 ]; then echo E: $0 abort >&2; fi" 0
}

settrapzero

##
## you can turn on debugging with 'debug'
##
if [ ${debug} ]; then
    set -x
    set -v
fi

set_file_perms () {
    local file="$1"
    chmod 0644 $file
    chown root.root $file
}


first () {
    echo $1
}

thelast () {
    eval 'echo $'$#
}

##
## given two packages, pick the last one based on which one is newest
##
last_by_version () {
    local a="$1"
    local b="$2"

    if [ -z "$b" ]; then
	echo $a
	exit 0
    fi

    shift 2

    if dpkg --compare-versions `dpkg -f $a Version` gt `dpkg -f $b Version`; then
	last_by_version $a $@
    else
	last_by_version $b $@
    fi
}

##
## Given a list of package names, give us the absolute paths to the downloaded files.
## Always favor files in the 'updates' dir; otherwise, if multiple
## versions exist, pick based on what has the highest version.
##
make_paths() {
    local PATHS=""
    local exit=false

    for pkg in $*; do
	local f=""
	local p
	local j
        # package name might  have '=' or /', so get rid of that
        pkg=${pkg%%[/=]*}
        # force files in the 'updates' dir to alway override
        debug "searching for updates/${pkg}_*.deb"
	for j in `pwd`/updates/${pkg}_*.deb; do
            if [ -f "$j" ]; then
		f="$f $j"
            fi
	done
        p=`last_by_version $f`
        if [ -f "$p" ]; then
            info " found $pkg in updates dir, '$p'" 1>& 2
        else
            # nothing in updates, try the archive area
            debug "searching for pkg glob $archive/cache/archives/${pkg}_*_{${arch},all}.deb"
            for j in $archive/var/cache/apt/archives/${pkg}_*_{${arch},all}.deb; do
                if [ -f "$j" ]; then
                    f="$f $j"
                fi
            done
            p=`last_by_version $f`
            if [ ! -f "$p" ]; then
                echo "can't find package, or no such package, '$pkg'" 1>& 2
                exit=true
            fi
        fi
        PATHS="$PATHS $p"
    done
    if $exit; then
	exit -1
    fi
    echo $PATHS
}

##
## sync and sleep, avoiding weird filesystem flush failures, generally
##
relax () {
    sync
    sleep 1
    sync
    sleep 1
}

##
## ensure that we are root
##
testroot () {
    [ `id -u` = 0 ] || error "must be run as root"
}

##
## make a temporary directory
##  FIXME: why not use mktmp or whatever?
##
make_tmpdir() {
    test -d $1 || mkdir $1
}

##
## subshells failure doesn't cause script failure, so we use this 
## after using subshells
##
testexit () {
    local status=$?
    if [ $status -ne 0 ]; then
	if [ ${ignorefail} ]; then
            warn "ignoring failure ($status)"
	else
            warn "abort ($status)"
            exit $status
	fi
    fi
}

##
## different internal logging facilities
##

warn () {
    echo "W:" $* 1>&2
}

error () {
    echo "E:" $* 1>&2
    exit -1
}

info () {
    echo "I:" $* 1>&2
}

debug () {
    # either debug or the special verbose var can turn this on
    [ ${debug}${verbose} ] && echo "D: " $* 1>&2 || true
}

##
## Pad the last floppy image with zeroes
## pad_last_image basename size
##
pad_last_image () {
    last=`ls ${1}*|tac|head -1`

    mv ${last} ${last}.tmp

    dd if=/dev/zero of=${last} bs=1k count=$2
    dd if=${last}.tmp of=${last} bs=1k count=$2 conv=notrunc

    rm -f ${last}.tmp
}

##
## download a package into $archive using debootstrap
##
downloadpkg () {
    if [ ! "${MIRRORS}" ]; then
        error "MIRRORS variable is not set"
    fi
    if [ -z ${archive} ]; then
        error "archive variable is not set"
    fi
    if [ ! -d ${archive} ]; then
        mkdir -p ${archive}
    fi

    . /usr/lib/debootstrap/functions

    # provide some variables debootstrap wants
    export LANG=C
    DOWNLOAD_ONLY=yes
    export TARGET=${archive}
    export ARCH=${arch}
    export PKGDETAILS=/usr/lib/debootstrap/pkgdetails

    mirror_style release
    download_style apt

    if grep '^offline_mode[[:space:]]*:\?=[[:space:]]*true' config >/dev/null; then
        info " not downloading, offline_mode is on" 1>& 2
        return
    fi

    local LIST=""
    for i in $*; do
        local_file=`make_paths $i 2>/dev/null` || true
        if [ -z "$local_file" ] ; then
            LIST="$LIST $i"
        else
            debug " not downloading, have file '$local_file'"
        fi
    done

    if [ ! "$LIST" ]; then
        debug " nothing to download, we must have them all"
    else
        debug " downloading package(s) $LIST"
        download $LIST 1>&2
    fi
}

##
## just a convenience function, since commonly, after we download, we
## want the file location
##
grab_paths () {
    downloadpkg $*
    make_paths $*
}

if [ ! -d ${tmpdir}/boot-floppies ]
then
    mkdir -m 700 ${tmpdir}/boot-floppies
fi
