#!/bin/bash
#
# Bruce Perens, July-September 1995
# Enrique Zanardi, January 1998
#
# This is free software under the GNU General Public License.

. ./common.sh
umask 022

# print a usage message and exit if the argument count is wrong
if [ $# -ne 4 ]; then
    echo "usage: $0 revision-extension modules-archive modcont-file archive" 1>&2
    cat 1>&2 << EOF

	revision-extension: the revision extension of the kernel.
	modules-archive: tgz containing the modules
	modcont-file: text file containing the list of modules in each subdir
		(generated by kernel.sh)
	archive: the directory containing the Debian binary packages.
EOF
    exit -1
fi

revext="$1"

modules=$2

modcont=$3

archive=$4

# whether this arch supports PCMCIA
if [ "$arch" = i386 -o "$revext" = pmac ]; then
    HAS_PCMCIA=true
else
    HAS_PCMCIA=false
fi

# if we're not root, bad things happen (modules owned by wrong user)
testroot

# make sure the required files are there.
if [ ! -f $modules ]; then
    error "can't find $modules"
fi
if [ ! -f $modcont ]; then
    error "can't find $modcont"
fi


# where we construct the disk breaking up
tartmp=${tmpdir}/boot-floppies/drivers-$$
rm -f -r $tartmp
make_tmpdir $tartmp

# temporary directory
E=${tmpdir}/boot-floppies/extract-tmp-$$

exit=false

if [ "$USE_LANGUAGE_CHOOSER" = true ] ; then
   whiptailpkg="whiptail-utf8"
else
   whiptailpkg="whiptail"
fi

info "downloading packages modconf, modutils, $whiptailpkg"
PACKAGE_PATH=`grab_paths modconf modutils $whiptailpkg`

if [ "$HAS_PCMCIA" = true ]; then
    info "this arch/subarch supports PCMCIA, downloading pcmcia-cs"
    PACKAGE_PATH="$PACKAGE_PATH `grab_paths pcmcia-cs`"
else
    info "no support for PCMCIA"
fi

if $exit; then
    exit -1
fi

if [ -d $E ]; then
    info "removing old driver extract area ($E)"
    mv -f $E $E.old
    rm -fr $E.old
fi

make_tmpdir $E

info "extracting packages into driver extract area $E"
for i in $PACKAGE_PATH; do
    debug extracting $i
    dpkg-deb --extract $i $E
done

if [ "$HAS_PCMCIA" = true ]; then
    info "tarring pcmcia materials into incorporated tarball, $tartmp/pcmcia.tgz"
    tar -cf $E/pcmcia.tar -C $E \
		etc/apm \
		etc/apm/event.d/pcmcia \
		etc/init.d/pcmcia \
		etc/pcmcia \
		sbin/ifport \
		sbin/ifuser \
		sbin/cardmgr
    gzip -9f $E/pcmcia.tar
    cp $E/pcmcia.tar.gz $tartmp/pcmcia.tgz
fi

info "removing unneeded message catalogs from modconf"
if [ "$USE_LANGUAGE_CHOOSER" = true ] ; then
   # remove non-UTF8 message catalogs
   rm $E/usr/share/modconf/eval_?? $E/usr/share/modconf/eval_??_??
else
   # removing .utf8 messages. Drop foreign messages too
   rm -f $E/usr/share/modconf/eval_*.utf-8
   if [ -f $E/usr/share/modconf/eval_$LINGUA ]; then
      ls $E/usr/share/modconf/eval_??* | grep -v eval_$LINGUA | xargs rm
   fi
fi

# this is a temporar fix until someone fixed modconf or/and whiptail
grep "unset.*LC_MESSAGES" $E/usr/share/modconf/dialog > /dev/null 2>&1 || cat scripts/drivers/dialog.diff | (cd $E/usr/share/modconf ; sudo patch -p0)
 
info "patching modconf to work with LANG_CHOOSER"

info "moving from extra area into staging area for driver tarball, $tartmp"
tar -cf $E/modconf.tar -C $E \
	usr/bin/whiptail \
	sbin/update-modules \
	etc/modutils \
	usr/sbin/modconf \
	usr/share/modconf
gzip -9f $E/modconf.tar
cp $E/modconf.tar.gz $tartmp/modconf.tgz

cp $modules $tartmp/modules.tgz
cp $modcont $tartmp/modcont

rm -fr $E

info "finishing up $tartmp"
cp scripts/drivers/install.sh $tartmp/install.sh
chmod 755 $tartmp/install.sh
echo drivers > $tartmp/type.txt

outfile=drivers${revext}.tgz
info "moving $tartmp into $outfile"
(cd $tartmp; tar clf - .) | gzip --no-name --best > $outfile
rm -fr $tartmp

info "splitting drivers into floppy-sized chunks"
case $arch in
    i386)
	if [ -z "$revext" ] ; then
            ./utilities/floppy_split $outfile drv12$revext 1200
            pad_last_image drv12$revext 1200
	fi
	./utilities/floppy_split $outfile drv14$revext 1440
	pad_last_image drv14$revext 1440
	;;
    m68k)
	./utilities/floppy_split $outfile drv14$revext 1440
	pad_last_image drv14$revext 1440
	;;
    powerpc)
	./utilities/floppy_split $outfile drv14$revext 1440
	pad_last_image drv14$revext 1440
	;;
    *)
	./utilities/floppy_split $outfile drv14$revext 1440
	pad_last_image drv14$revext 1440
	;;
esac

exit 0
