package buoy.widget;

import buoy.event.*;
import buoy.xml.*;
import buoy.xml.delegate.*;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A BButton is a pushbutton Widget.  Its appearance can be customized by setting the text and/or image which
 * appears on it.
 * <p>
 * In addition to the event types generated by all Widgets, BButtons generate the following event types:
 * <ul>
 * <li>{@link buoy.event.CommandEvent CommandEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class BButton extends Widget
{
  public static final Position CENTER = new Position(0);
  public static final Position NORTH = new Position(1);
  public static final Position SOUTH = new Position(2);
  public static final Position WEST = new Position(4);
  public static final Position EAST = new Position(8);
  public static final Position NORTHEAST = new Position(NORTH.value+EAST.value);
  public static final Position SOUTHEAST = new Position(SOUTH.value+EAST.value);
  public static final Position NORTHWEST = new Position(NORTH.value+WEST.value);
  public static final Position SOUTHWEST = new Position(SOUTH.value+WEST.value);

  static
  {
    WidgetEncoder.setPersistenceDelegate(Position.class, new StaticFieldDelegate(BButton.class));
  }

  /**
   * Create a new BButton with no set text or image.
   */
  
  public BButton()
  {
    component = createComponent();
    getComponent().addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent ev)
      {
        dispatchEvent(new CommandEvent(BButton.this, ev.getWhen(), ev.getModifiers(), ((JButton) component).getActionCommand()));
      }
    });
  }
  
  /**
   * Create a new BButton.
   *
   * @param text      the text to display on the button
   */
  
  public BButton(String text)
  {
    this();
    getComponent().setText(text);
  }

  /**
   * Create a new BButton.
   *
   * @param icon      the image to display on the button
   */
  
  public BButton(Icon icon)
  {
    this();
    getComponent().setIcon(icon);
  }

  /**
   * Create a new BButton.
   *
   * @param text      the text to display on the button
   * @param icon      the image to display on the button
   */
  
  public BButton(String text, Icon icon)
  {
    this();
    getComponent().setText(text);
    getComponent().setIcon(icon);
  }
  
  /**
   * Create the JButton which serves as this Widget's Component.  This method is protected so that
   * subclasses can override it.
   */
  
  protected JButton createComponent()
  {
    return new JButton();
  }

  public JButton getComponent()
  {
    return (JButton) component;
  }

  /**
   * Get the text being displayed on the button.  This may be null.
   */
  
  public String getText()
  {
    return getComponent().getText();
  }

  /**
   * Set the text being displayed on the button.  This may be null.
   */
  
  public void setText(String text)
  {
    getComponent().setText(text);
    invalidateSize();
  }

  /**
   * Get the image being displayed on the button.  This may be null.
   */
  
  public Icon getIcon()
  {
    return getComponent().getIcon();
  }

  /**
   * Set the image being displayed on the button.  This may be null.
   */
  
  public void setIcon(Icon icon)
  {
    getComponent().setIcon(icon);
    invalidateSize();
  }

  /**
   * Get the "action command" which will be sent in a CommandEvent when this button is pressed.
   */
  
  public String getActionCommand()
  {
    return getComponent().getActionCommand();
  }

  /**
   * Set the "action command" which will be sent in a CommandEvent when this button is pressed.
   */
  
  public void setActionCommand(String command)
  {
    getComponent().setActionCommand(command);
  }

  /**
   * Get the largest size at which this Widget can reasonably be drawn.  When a WidgetContainer lays out
   * its contents, it will attempt never to make this Widget larger than its maximum size.
   */
  
  public Dimension getMaximumSize()
  {
    return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
  }

  /**
   * Get the position of the text relative to the image.  This will be one of the alignment constants
   * defined by this class: CENTER, NORTH, NORTHEAST, EAST, etc.
   */

  public Position getTextPosition()
  {
    int hpos = getComponent().getHorizontalTextPosition();
    int vpos = getComponent().getVerticalTextPosition();
    return Position.get(hpos, vpos);
  }

  /**
   * Set the position of the text relative to the image.  This should be one of the alignment constants
   * defined by this class: CENTER, NORTH, NORTHEAST, EAST, etc.  If this BButton does not display both
   * text and an image, this method will have no effect.
   */

  public void setTextPosition(Position position)
  {
    int pos = position.value;
    JButton jb = getComponent();
    if ((pos&NORTH.value) != 0)
      jb.setVerticalTextPosition(SwingConstants.TOP);
    else if ((pos&SOUTH.value) != 0)
      jb.setVerticalTextPosition(SwingConstants.BOTTOM);
    else
      jb.setVerticalTextPosition(SwingConstants.CENTER);
    if ((pos&EAST.value) != 0)
      jb.setHorizontalTextPosition(SwingConstants.RIGHT);
    else if ((pos&WEST.value) != 0)
      jb.setHorizontalTextPosition(SwingConstants.LEFT);
    else
      jb.setHorizontalTextPosition(SwingConstants.CENTER);
    invalidateSize();
  }

  /**
   * This inner class represents a value for the alignment or text position.
   */

  public static class Position
  {
    protected int value;

    private Position(int value)
    {
      this.value = value;
    }

    private static Position get(int hpos, int vpos)
    {
      switch (hpos)
      {
        case SwingConstants.LEFT:
          switch (vpos)
          {
            case SwingConstants.TOP:
              return NORTHWEST;
            case SwingConstants.BOTTOM:
              return SOUTHWEST;
            default:
              return WEST;
          }
        case SwingConstants.RIGHT:
          switch (vpos)
          {
            case SwingConstants.TOP:
              return NORTHEAST;
            case SwingConstants.BOTTOM:
              return SOUTHEAST;
            default:
              return EAST;
          }
        default:
          switch (vpos)
          {
            case SwingConstants.TOP:
              return NORTH;
            case SwingConstants.BOTTOM:
              return SOUTH;
            default:
              return CENTER;
          }
      }
    }
  }
}