#!/usr/bin/python

import os
import shutil
import pwd
import grp
import plistlib
import copy
import StringIO
from uuid import UUID, uuid5


def main():

    # Init params
    defaultConfigFile = "/etc/caldavd/caldavd.plist.upgrade.nss.tmp"
    defaultConfig = {
        "DirectoryService": {
            "type": "twistedcaldav.directory.xmlfile.XMLDirectoryService",
        },
        "DocumentRoot": "/Library/CalendarServer/Documents",
    }
    serviceDefaultParams = {
        "twistedcaldav.directory.nss.NssDirectoryService": {
            "realmName": "Test Realm",
            "groupPrefix": "caldavd-",
            "firstValidUid": 1000,
            "lastValidUid": 65533,
            "firstValidGid": 1000,
            "lastValidGid": 65533,
        }
    }
    recordType_users = "users"
    recordType_groups = "groups"
    nssServiceBaseGUID = "8EFFFAF1-5221-4813-B971-58506B963573"

    # Function to generate uuid from name
    def uuidFromName(namespace, name):
        """
        Generate a version 5 (SHA-1) UUID from a namespace UUID and a name.
        See http://www.ietf.org/rfc/rfc4122.txt, section 4.3.
        @param namespace: a UUID denoting the namespace of the generated UUID.
        @param name: a byte string to generate the UUID from.
        """
        # We don't want Unicode here; convert to UTF-8
        if type(name) is unicode:
            name = name.encode("utf-8")

        return str(uuid5(UUID(namespace), name))

    # Function to update config
    def _update_config(configElement, defaultConfigElement):
        for key, value in defaultConfigElement.iteritems():
            if key not in configElement:
                configElement[key] = copy.deepcopy(value)
            elif isinstance(configElement[key], dict) and \
                 isinstance(value, dict):
                _update_config(configElement[key], value)


    # Function to move individual user/group directory
    def _move_directory(nssServiceGUID, documentRoot, recordType, shortName):
        guid = uuidFromName(nssServiceGUID, "%s:%s" % (recordType, shortName))
        olddir = os.path.join(documentRoot, "calendars", "__uids__", shortName)
        newdir = os.path.join(documentRoot, "calendars", "__uids__", guid)
        print "Moving directory for %s '%s' (GUID: %s) ..." % \
              (recordType[:-1], shortName, guid)
        if (not os.path.isdir(olddir)):
            print "Old data directory '%s' does not exist. Nothing to move." % \
                  (olddir)
            return
        if (os.path.isdir(newdir)):
            print "New data directory '%s' already exists. Aborting move." % \
                  (newdir)
            return
        shutil.move(olddir, newdir)
        print "Data directory successfully moved from '%s' to '%s'." % \
              (olddir, newdir)

    # Begin
    print ("Moving caldavd directories of NSS users and groups as the "
           "directory names in calendarserver 2.x are based on UUID "
           "rather than username/groupname as in calendarserver 1.x "
           "(See Debian Bug#610124 for more info) ...\n")

    # Check if config file exists.
    if not os.path.isfile(defaultConfigFile):
        print "Caldavd config file %s does not exist. Exiting." % \
              (defaultConfigFile)
        return

    # Load configuration file
    configPlist = plistlib.readPlist(defaultConfigFile)

    # Update default configuration
    s = StringIO.StringIO()
    plistlib.writePlist(defaultConfig, s)
    defaultConfigPlist = plistlib.readPlistFromString(s.getvalue())
    _update_config(configPlist, defaultConfigPlist)

    # If NSS is not the default directory service, exit
    if configPlist.DirectoryService.type != \
                "twistedcaldav.directory.nss.NssDirectoryService":
        print "Directory service NssDirectoryService is not configured in " \
              "%s. Exiting." % (defaultConfigFile)
        return

    # Update with service default params
    s = StringIO.StringIO()
    plistlib.writePlist(
       {"params": serviceDefaultParams[
                     "twistedcaldav.directory.nss.NssDirectoryService"]},
       s
    )
    nssServiceDefaultParamsPlist = plistlib.readPlistFromString(s.getvalue())
    _update_config(configPlist.DirectoryService, nssServiceDefaultParamsPlist)

    # Generate serviceGUID
    nssServiceGUID = uuidFromName(
                        nssServiceBaseGUID,
                        configPlist.DirectoryService.params.realmName
                     )
    print "NSS Directory Service GUID - %s\n" % (nssServiceGUID)

    # Move users' data directories
    print "Moving data directories of users ..."
    users = pwd.getpwall()
    for user in users:
        if user[2] >= configPlist.DirectoryService.params.firstValidUid and \
           user[2] <= configPlist.DirectoryService.params.lastValidUid:
            _move_directory(
                nssServiceGUID,
                configPlist.DocumentRoot,
                recordType_users,
                user[0]
            )
            print

    # Move groups' data directories
    print "Moving data directories of groups ..."
    groups = grp.getgrall()
    for group in groups:
        if group[2] >= configPlist.DirectoryService.params.firstValidGid and \
           group[2] <= configPlist.DirectoryService.params.lastValidGid and \
           group[0].startswith(configPlist.DirectoryService.params.groupPrefix):
            _move_directory(
                nssServiceGUID,
                configPlist.DocumentRoot,
                recordType_groups, 
                group[0].replace(
                    configPlist.DirectoryService.params.groupPrefix,'',1
                )
            )
            print


if __name__ == "__main__":
    main()

