/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * Copyright  1994-2001 Roxen Internet Software
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_auth_basic.pike,v 1.20.2.3.2.3 2001/09/26 00:27:31 kiwi Exp $
 */

//
//! module: CAMAS: Basic Auth
//!  Basic Auth Module for CAMAS.<br />
//!  This module will give the interface between standard
//!  flat file (eg. /etc/passwd) to get names, login etc...
//! inherits: module
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_auth_basic.pike,v 1.20.2.3.2.3 2001/09/26 00:27:31 kiwi Exp $
// Faster Debugging
//#define CAMAS_DEBUG
#ifdef CAMAS_DEBUG
# define WERR(X) if(QUERY(debug)) werror("CAMAS AUTH: "+X+"\n");
#else
# define WERR(X)
#endif

#include <module.h>
inherit "module";

constant cvs_version="$Id: camas_auth_basic.pike,v 1.20.2.3.2.3 2001/09/26 00:27:31 kiwi Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Basic Auth";
constant module_doc  = "Basic Auth Module for CAMAS.<br>"
                       "This module will give the interface between standard "
		       "flat file (eg. /etc/passwd) to get names, login, etc..";
constant module_unique = 1;
constant thread_safe=1;

// Global variables pour this module
mapping userdatabase;	// The contents of namedb in array
			// in the same form than the file
			// format
			
// Functions to hide parameters when they are not need
int hide_file() {
  return (QUERY (usernamemethod) != "file");
}

int hide_dns() {
  return (QUERY (usernamemethod) == "none");
}

// Variables
void create()
{
#ifdef CAMAS_DEBUG
 defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
        "error log ?");
#endif
 defvar("usernamemethod","file",
	"User names method", TYPE_STRING_LIST, "Chooses the method used "
	"to set the user names.", ({ "file", "none" }));

 defvar("namedb","/etc/passwd",
	"User names file", TYPE_FILE, "If method is set to file, this is "
	"the file that will be searched for the users real name."
	"<br />Format is composed of lines like: "
	"'login:surname:name:domain:email'.", 0, hide_file);

 defvar("dnsname","","The sitewide DNS domain name",TYPE_STRING,
        "The DNS domain name used for CAMAS webmail", 0, hide_dns);

 defvar("emaillogin",0,"Login using email address",TYPE_FLAG,
        "Use the email address to login instead of the imap login", 0, hide_file);

 defvar("showdns",0,"Set the dns domain on login", TYPE_FLAG,
        "Set the right part of the DNS domain on login. This make less "
	"to type for the users, but in the other hand avoid users to "
	"login under another email if the imap server can handle lots of "
	"different domains...", 0, hide_dns);
}

array|int readfile(string file)
{
  object o=Stdio.File();
  if(!o->open(file,"r"))
    return 0;
  array result=( (o->read()-"\r") / "\n");
  o->close();

  return result - ({ "" });
}

void start(int cnt, object conf)
{
  if (QUERY(usernamemethod)=="file")		// We use flat file database
  {
    array|int tempdb=readfile(QUERY(namedb));
    if (intp(tempdb))
    {
      WERR(" Cannot read "+QUERY(namedb)+" file !\n");
    }
    else
     {
       WERR("Data base :" + sprintf("%O", tempdb) +"\n");
       sort(tempdb);			// sort it
       WERR("Sorted Data base :" + sprintf("%O",tempdb) +"\n");
       userdatabase = ([ ]);		// initialize it :)
       foreach( tempdb, string foo) {
         array entry = foo / ":";
	 if (sizeof(entry) > 6) {
	   userdatabase += ([ entry[0] : ({ entry[0], // Login
					    entry[4], // Gecos
					    entry[5], // Home directory
					    entry[6]  // Shell
					   }) ]);
	 }

       }
       // userdatabase += ({ foo / ":" }); // populate
     }
    WERR(sprintf("UserDatabase:%O\n",mkmapping(indices(userdatabase),values(userdatabase))));
  }
  else
  {
    userdatabase = 0;	// Garbage collector :)
  }
}

string status()
{
  string out;
  out = "Basic Authentication module.";
  if (QUERY(usernamemethod)=="file")
    out += sprintf("<br />Status :<br />"
		   "<ul><li>Method : file</li>"
		   "<li>Entries in %s : %d</li>"
		   "</ul>",QUERY(namedb), sizeof(userdatabase));
  return out;
}

string query_provides()
{
 return("camas_auth");
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide v2, we
//!  *MUST* be backward compatible with v1.
//
int version()
{
  WERR("version()");
  return 1;
}

//
//! method: string|int getlogin(string login)
//!  Return the imap login to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a string : the imap login to be used.<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard code or for unknown error type)<br />
//!  1 : Bad password<br />
//!  2 : Access denied<br />
//!  3 : Account is locked<br />
//!  4 : Change your password<br />
//
string|int getlogin(string login)
{
  string|int out;
  if (QUERY(usernamemethod)=="file") {
   if( userdatabase[login] ) out = userdatabase[login][0];
   else out = 0;
   WERR("getlogin("+login+") = " + out );
   return out;
  }
  return login;
}

//
//! method: string|int getdnsname(void)
//!  Gets the sitewide email domain name that can be used
//!  for email login or to complete unqualified emails
//! returns:
//!  a string : the dns domain name<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard code or unknown error type)<br />
//!  1 : There is no domain name set, use whole email or imap login
//
string|int getdnsname()
{
  WERR("getdnsname()");
  if(QUERY(usernamemethod)!="none") {
    if(QUERY(dnsname)=="")
      return 1;
    else
      return QUERY(dnsname);
  }
  else
   return 1;
}

//
//! method: int getemaillogin(void)
//!  Do the email is used instead of the imap login as
//!  login method ?
//! returns:
//!  a int : 1 = Yes, 0 = No
//
int getemaillogin()
{
  WERR("getemaillogin()");
  if (QUERY(usernamemethod)=="file")
    return (int)QUERY(emaillogin);
  else
    return 0;
}

//
//! method: array|int getfullnames(string login)
//!  Get the name and the surname of the specifier imap login
//! arg: string login
//!  The imap login
//! returns:
//!  a array : the name and surname in the form :
//!  ({ "surname", "name", "email@domain.com", "imaplogin" })<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard error code or unknown error type)<br />
//!  1 : Empty set
//
array|int getfullnames(string login)
{
  WERR("getfullnames("+login+")");
  array|int out;
  if (QUERY(usernamemethod)=="file") {
   if( userdatabase[login] ) { 
     out = (userdatabase[login][1]/",")[0]/" ";
     if (sizeof(out)==1) out += ({ "" });
     if (QUERY(emaillogin) && (QUERY(dnsname)!= "")) {
       out += ({ login + "@" + QUERY(dnsname) });
     } else {
       if (QUERY(dnsname)!="") 
        out += ({ login + "@" + QUERY(dnsname) });
       else 
        out += ({ login + "@unknown.domain" });
     }
     out += ({ login });
   }
   else out = 0;
   WERR("getfullnames("+login+") = " + sprintf("%O",out));
   return out;
  }
  return 1;
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: usernamemethod
//! Chooses the method used to set the user names.
//!  type: TYPE_STRING_LIST
//!  name: User names method
//
//! defvar: namedb
//! If method is set to file, this is the file that will be searched for the users real name.<br />Format is composed of lines like: 'login:surname:name:domain:email'.
//!  type: TYPE_FILE
//!  name: User names file
//
//! defvar: dnsname
//! The DNS domain name used for CAMAS webmail
//!  type: TYPE_STRING
//!  name: The sitewide DNS domain name
//
//! defvar: emaillogin
//! Use the email address to login instead of the imap login
//!  type: TYPE_FLAG
//!  name: Login using email address
//
//! defvar: showdns
//! Set the right part of the DNS domain on login. This make less to type for the users, but in the other hand avoid users to login under another email if the imap server can handle lots of different domains...
//!  type: TYPE_FLAG
//!  name: Set the dns domain on login
//
