/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_images.pike,v 1.5.2.7.2.1 2001/09/24 11:10:57 kiwi Exp $
 */
#include <module.h>
inherit "module";
inherit "caudiumlib";

//
//! module: CAMAS: Images Module
//!  This module is used to create the various images used in CAMAS.<br />
//!  You can specify there differents icons that the autogenerated ones used
//!  in Mailindex pages (New mail icon, Replyed mail icon,..).<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER|MODULE_PROVIDER
//! cvs_version: $Id: camas_images.pike,v 1.5.2.7.2.1 2001/09/24 11:10:57 kiwi Exp $
//

constant cvs_version   = "$Id: camas_images.pike,v 1.5.2.7.2.1 2001/09/24 11:10:57 kiwi Exp $";
constant module_type   = MODULE_PARSER | MODULE_PROVIDER;
constant module_name   = "CAMAS: Images Module";
constant module_doc    = "This module is used to create the various images "
                         "used in CAMAS.<br /> You can specify there differents "
                         "icons that the autogenerated ones used in Mailindex "
                         "pages (New mail icon, Replyed mail icon,...).<br /> "
                         "<b>This module is automatically selected if you "
                         "select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;

// For debug
//#define CAMAS_DEBUG
#ifdef CAMAS_DEBUG
# define DEBUG(X) if(QUERY(debug)) perror("CAMAS IMAGES: "+X+"\n");
#else
# define DEBUG(X)
#endif

// Cache images
caudium.ImageCache image_cache;

// Function used to hide parameters when no needed
int hide_icons() {
  return !QUERY(miicons);
}

void create()
{
#ifdef CAMAS_DEBUG
  // Debug
  defvar("debug", 0, "Debug", TYPE_FLAG,"Debug the calls / errors into "
         "Caudium error log ?");
#endif
  // Mailindex icons
  defvar("miicons",0,"External Image icons for Mailindex", TYPE_FLAG,
         "Use external image icons for mailindex instead of images "
         "generated by this module");
  defvar("checkimg","","Check image URL",TYPE_STRING,
         "Check image url. Leave empty if has allready specified "
         "reply and new mails icons or if you'd like the internal generated "
         "image instead.",0,hide_icons);
  defvar("ncheckimg","","New mail image URL", TYPE_STRING,
         "New mail icon used when a new mail is in mail index.", 0, hide_icons);
  defvar("rcheckimg","","Replyed mail image URL",TYPE_STRING,
         "Replyed mail icon used when a mail has been replyed in the "
         "mailindex.", 0, hide_icons);
}

void start (int cnt, object conf) {
  image_cache = caudium.ImageCache ("camas_image", draw_image);
}

string query_provides () { 
  return "camas_images";
}

mapping query_tag_callers () {
  return ([ 
    "imho_image": tag_imho_image,
    "imho_image_url": tag_imho_image
  ]);
}

mapping find_internal (string f, object id) {
  return image_cache->http_file_answer (f, id);
}

mixed  draw_image (mapping args, string text, object id) {
  object image;

  array fg = Colors.parse_color (args->fg || "white");
  array bg = Colors.parse_color (args->bg || "black");
  int size = (int)args->size || 20;

  switch (args->image) {

  case "check":
  case "ncheck":
  case "rcheck":
    image = Image.image (size, size, @bg);
    image = image->setcolor (@fg)->polyfill (Array.map ( ({ 10,50,50,90,90,10,70,20,50,80,20,40 }),
							 lambda (int i, int s) { return (int)(i * s / 100); }, size));
    break;

  case "arrowup":
  case "arrowdown":
    image = Image.image (size, size, @bg);
    image = image->setcolor (@fg)->polyfill (Array.map ( ({ 20,80,80,80,50,20 }),
							 lambda (int i, int s) { return (int)(i * s / 100); }, size));
    if (args->image == "arrowdown")
      image = image->mirrory ();
    break;
    
  case "arrownone":
    image = Image.image (size, size, @bg);
    image = image->setcolor (@fg)->polyfill (Array.map ( ({ 20,60,80,60,80,40,20,40 }),
							 lambda (int i, int s) { return (int)(i * s / 100); }, size));
    break;
    
  case "bannerleft":
  case "bannerright":
    array clouds = Colors.parse_color (args->clouds || "white");
    string text = (args->image == "bannerright") ? 0 : (args->text || "CAMAS");
    image = Image.image (text ? 500 : 1, 32, @bg);
  
    if (text) {
      image = image->turbulence ( ({ 0, bg, 1, clouds }), 3, 0.1, 0, 0, 0.7);
      image->tuned_box (150, 0, 499, 32, ({ bg + ({ 255 }), bg + ({ 0 }), bg + ({ 255 }), bg + ({ 0 }) }) );
    }

    image->setcolor (0, 0, 0);
    for (int i = 1; i < image->ysize (); i += 2) {
      image->line (0, i, image->xsize () - 1, i);
    }

    if (text) {
      object fnt = get_font ("lucida", 32, 0, 0, "left", 0.0, 0.0);
      object textim = fnt->write (text);
      textim = textim->scale (0.8);
      image->paste_alpha_color (textim->apply_matrix( ({({1,2,1}),
							({2,5,2}),
							({1,2,1})}) ), 0, 0, 0, 8, 3);
      image->paste_alpha_color (textim, @fg, 5, 1);
    }
    break;

  case "newmail":
  case "nonewmail":
    int mail = 0;
    if (args->image == "newmail")
      mail = 1;
    image = Image.image (160, 128, @bg);
  
    array envelopecolor = Colors.parse_color (args->envelopecolor || "white");
    array insideenvcolor = Colors.parse_color (args->insideenvcolor || "#808080");
    array mailcolor = Colors.parse_color (args->mailcolor || "#ffc030");
    array maillinecolor = Colors.parse_color (args->maillinecolor || "black");

    image->tuned_box (16, 40, 140, 120,
		      ({ envelopecolor + ({ 0 }), envelopecolor + ({ 0 }),
			 envelopecolor + ({ 0 }), insideenvcolor + ({ 0 }) }));
    image->tuned_box (136, 40, 140, 120,
		      ({ insideenvcolor + ({ 128 }), insideenvcolor + ({ 128 }),
			 insideenvcolor + ({ 128 }), insideenvcolor + ({ 128 }) }));
    image->tuned_box (16, 116, 140, 120,
		      ({ insideenvcolor + ({ 128 }), insideenvcolor + ({ 128 }),
			 insideenvcolor + ({ 128 }), insideenvcolor + ({ 128 }) }));
    
    if (!mail) {
      image->setcolor (@insideenvcolor);
      image->line (16,40,80,84);
      image->line (80,84,140,40);
    }
    else {
      image->setcolor (@envelopecolor);
      image->polyfill (({ 16,40, 80,84, 136,40, 80,8 }));
      image->setcolor (@insideenvcolor);
      image->polyfill (({ 20,40, 80,80, 132,40, 80,16 }));
      
      int h = 4;
      image->setcolor (@mailcolor);
      image->polyfill (({ 40,4*h, 40,52, 80,80, 116,52, 116,4*h }));
      image->setcolor (@maillinecolor);
      image->line (40,4*h,40,52);
      image->line (116,4*h,116,52);
      image->line (40,4*h,112,4*h);
      image->line (52,4*h+12,100,4*h+12);
      image->line (52,4*h+24,100,4*h+24);
    }
    
    break;

  default:
    image = Image.image (size, size, @bg);
  }

  return image;
}

constant imho_img_args = ({ "align", "border", "hspace", "vspace", "class", "alt", "width", "height" });

//! tag: imho_image
//!  Create an image or an icon for imho. E.g. used for the banner and so on.
//! see also: imho_image_url
//!  Create an image and return the url.
//! attribute: src
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: width
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: height
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: alt
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: align
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: border
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: hspace
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: vspace
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: class
//!  Same attribute like standard html &lt;img &gt; tag
//! attribute: image
//!  Create a CAMAS image, correct args are : arrowup, arrowdown
//!  arrownone, bannerleft, bannerright, newmail, nonewmail, check, 
//!  ncheck (new mail icon), rcheck (replyed mail icon).
//! attribute: text
//!  When image attribute is bannerleft the text to write on the image
string tag_imho_image (string tag_name, mapping args, object id, object file) {
  mapping img_args = ([ ]);

  foreach (imho_img_args, string arg) {
    if (args[arg]) {
      img_args[arg] = args[arg];
      m_delete (args, arg);
    }
  }

  if (args->src && (args->src != ""))
    return make_tag ("img", img_args + ([ "src": args->src ]));
  m_delete (args, "src");

  if (args->image && QUERY(miicons)) {
  DEBUG("image called : "+args->image );
   switch (args->image) 
   {
     case "check": if(sizeof(QUERY(checkimg)))
                     return make_tag ("img", img_args + ([ "src": QUERY(checkimg) ]));
                   break;
     case "ncheck": if(sizeof(QUERY(ncheckimg)))
                      return make_tag ("img", img_args + ([ "src": QUERY(ncheckimg) ]));
                   break;
     case "rcheck": if(sizeof(QUERY(rcheckimg)))
                      return make_tag ("img", img_args + ([ "src": QUERY(rcheckimg) ]));
                   break;
   }
  }

  string num = image_cache->store (args, id);
  img_args->src = query_internal_location () + num;

  if (tag_name == "imho_image_url")
    return img_args->src;

  mapping size = image_cache->metadata (num, id, 1);
  if (size) {
    if (!img_args->width)
      img_args->width = (string)size->xsize;
    if (!img_args->height)
      img_args->height = (string)size->ysize;
  }

  if (!img_args->alt)
    img_args->alt = "";

  return make_tag ("img", img_args);
}


/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the calls / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: miicons
//! Use external image icons for mailindex instead of images generated by this module
//!  type: TYPE_FLAG
//!  name: External Image icons for Mailindex
//
//! defvar: checkimg
//! Check image url. Leave empty if has allready specified reply and new mails icons or if you'd like the internal generated image instead.
//!  type: TYPE_STRING
//!  name: Check image URL
//
//! defvar: ncheckimg
//! New mail icon used when a new mail is in mail index.
//!  type: TYPE_STRING
//!  name: New mail image URL
//
//! defvar: rcheckimg
//! Replyed mail icon used when a mail has been replyed in the mailindex.
//!  type: TYPE_STRING
//!  name: Replyed mail image URL
//
