#
#   Copyright (C) 1998, 1999, 2000, 2001 Loic Dachary
#
#   This program is free software; you can redistribute it and/or modify it
#   under the terms of the GNU General Public License as published by the
#   Free Software Foundation; either version 2, or (at your option) any
#   later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software 
#   Foundation, Inc. 59 Temple Place, Suite 330, Boston, 
#   MA  02111-1307  USA.
#
package Catalog::tools::hook_mifluz;
use vars qw(@ISA);
use strict;

use POSIX;

use Catalog::tools::mifluz;
use Catalog::tools::cgi;
use Catalog::tools::tools;
use Search::Mifluz;

@ISA = qw(Catalog::tools::mifluz);

sub initialize {
    my($self) = @_;

    $self->Catalog::tools::mifluz::initialize();

    my($config) = config_load("hook_mifluz.conf");
    error("missing hook_mifluz.conf") if(!defined($config));
    %$self = ( %$self , %$config );
}

sub mysql {
    my($self, $mysql) = @_;
    $self->{'mysql'} = $mysql;
}

sub hook_delete {
    my($self, $table, $primary_values) = @_;
    my($mysql) = $self->{'mysql'};

    my($spec) = $self->{$mysql->{'base'}};
    return if(!exists($spec->{'tables'}->{$table}));
    my($info) = $mysql->info_table($table);
    my($primary_key) = $info->{'_primary_'};

    my($primary_value);
    foreach $primary_value (@$primary_values) {
	$self->hook_modify($table, $primary_value, "delete");
    }
}

#
# Since mifluz reindexes everything each time we change
# a field in a row, don't bother to make an update. Just 
# delete and insert again.
#
sub hook_before_update {
    my($self, $table, $primary_values) = @_;

    $self->hook_delete($table, $primary_values);
}

sub hook_update {
    my($self, $table, $primary_values) = @_;
    my($mysql) = $self->{'mysql'};

    my($spec) = $self->{$mysql->{'base'}};
    return if(!exists($spec->{'tables'}->{$table}));

    my($primary_value);
    foreach $primary_value (@$primary_values) {
	$self->hook_insert($table, $primary_value);
    }
}

#
# Extract the fields values from $table for record matching $primary_value.
# Only the fields specified in the configuration file hook_mifluz.conf are
# extracted and type conversion occurs for date/time fields (hardwired).
# A row is returned (hashref).
#
sub hook_modify_prepare {
    my($self, $table, $primary_value, $fields_limit) = @_;
    my($mysql) = $self->{'mysql'};

    #
    # Extract the indexing specifications for this table.
    #
    my($spec) = $self->{$mysql->{'base'}};
    return if(!exists($spec->{'tables'}->{$table}));

    #
    # General information about the MySQL table
    #
    my($info) = $mysql->info_table($table);
    my($primary_key) = $info->{'_primary_'};
    my($spec_params) = $spec->{'tables'}->{$table}->{'params'};
    #
    # Extract row
    #
    my($where) = $spec_params->{'where'};
    if(defined($where)) {
	$where = "and ( $where )";
    } else {
	$where = '';
    }
    my($sql) = "select $primary_key from $table where $primary_key = $primary_value $where";
    my($row) = $mysql->exec_select_one($sql);
    #
    # If the row does not match the constraint, do nothing
    #
    dbg($sql, "hook_mifluz");
    return if(!defined($row));
    #
    # Build the list of fields with appropriate format conversion
    #
    my($spec_fields) = $spec->{'tables'}->{$table}->{'fields'};
    my(@fields);
    my($field, $spec_field);
    while(($field, $spec_field) = each(%$spec_fields)) {
	next if(defined($fields_limit) && !exists($fields_limit->{$field}));
	my($type) = $info->{$field}->{'type'};
	if($type eq 'time' || $type eq 'date') {
	    push(@fields, "date_format($field, 'Ymd') as $field");
	} elsif($type eq 'set' && defined($info->{$field}->{'dict'})) {
	    #
	    # Do nothing, will be added magically by sexec_select
	    #
	} else {
	    push(@fields, "$field");
	}
    }
    error("unexpected empty field list for $table") if(!@fields);
    #
    # Extract data
    #
    my($fields) = join(',', @fields);
    dbg("extracting $fields", "hook_mifluz");
    $row = $mysql->sexec_select_one($table, "select $primary_key,$fields from $table where $table.$primary_key = $primary_value $where");

    return $row;
}

sub hook_index {
    my($self, $file) = @_;

    open(INDEX, $file) or error("cannot open $file : $!");
    my($mysql) = $self->{'mysql'};
    #
    # Extract the indexing specifications for this table.
    #
    my($spec) = $self->{$mysql->{'base'}};

#    my($env_config) = $ENV{'MIFLUZ_CONFIG'};
#    my($mifluz_config) = config_path("mifluz.conf");
#    $ENV{'MIFLUZ_CONFIG'} = $mifluz_config if(defined($mifluz_config));
#
#    my($mifluz) = new Search::Mifluz();
#    my($words) = $mifluz->List();
#    $words->Open($self->file(), O_RDWR);
#    $words->BatchStart();
#    my($wordref) = $mifluz->Word();
#    my($key) = $wordref->Key();

    my($table);
    foreach $table (keys(%{$spec->{'tables'}})) {
	
	#
	# General information about the MySQL table
	#
	my($info) = $mysql->info_table($table);
	my($primary_key) = $info->{'_primary_'};

	my($table_code) = $spec->{'tables'}->{$table}->{'params'}->{'code'};
	my($spec_fields) = $spec->{'tables'}->{$table}->{'fields'};
	my($fields) = join(',', keys(%$spec_fields));
	my($spec_params) = $spec->{'tables'}->{$table}->{'params'};
	#
	# Extract row
	#
	my($where) = $spec_params->{'where'};
	if(defined($where)) {
	    $where = "where ( $where )";
	} else {
	    $where = '';
	}
	my($sql) = "select * from $table $where";
	my($rows);
	my($bottom) = 0;
	do {
	    ($rows) = $mysql->select($sql, $bottom, 10000);
	    if(@$rows > 0) {
		my($row);
		foreach $row (@$rows) {
		    my($primary_value) = $row->{$primary_key};
		    my($key_template) = "%s <DEF> $table_code %d $primary_value %d";
		    my($key_template1) = "%s <DEF> " . ($table_code + 4) . " 100 $primary_value %d";
		    my($field, $spec_field);
		    my($location1) = 0;
		    while(($field, $spec_field) = each(%$spec_fields)) {
			my($text) = lc(unaccent_8859($row->{$field}));
			$text =~ s/_+/ /g;
			$text =~ s/^\W+//;
			my($location) = 0;
			my($mifluz_field) = $spec_field->{'field'};
			my($word);
			foreach $word (split(/\W+/, $text)) {
			    $location++;
			    print INDEX sprintf($key_template, $word, $mifluz_field, $location) . "\n";
#			    $key->SetAscii(sprintf($key_template, $word, $mifluz_field, $location));
#			    $words->Override($wordref);
#			    $location1++;
#			    $key->SetAscii(sprintf($key_template1, $word, $location1));
#			    $words->Override($wordref);
			}
		    }

#		    $self->hook_insert($table, $row->{$primary_key});
#
#		    my($keys) = $self->hook_record2keys($row, $spec_fields, $table_code, $primary_value);
#		    if(@$keys > 0) {
#			print INDEX join("\n", @$keys) . "\n";
#		    }
		}
	    }
	    $bottom += scalar(@$rows);
	} while(scalar(@$rows));
    }

#    if(defined($env_config) && defined($mifluz_config)) {
#	$ENV{'MIFLUZ_CONFIG'} = $env_config;
#    }

    close(INDEX);
}

sub hook_modify {
    my($self, $table, $primary_value, $action) = @_;
    my($mysql) = $self->{'mysql'};
    #
    # Extract the indexing specifications for this table.
    #
    my($spec) = $self->{$mysql->{'base'}};
    return if(!exists($spec->{'tables'}->{$table}));

    #
    # General information about the MySQL table
    #
    my($info) = $mysql->info_table($table);
    my($primary_key) = $info->{'_primary_'};
    my($params) = $spec->{'params'};

    my($row) = $self->hook_modify_prepare($table, $primary_value);
    return if(!defined($row));

    my($table_code) = $spec->{'tables'}->{$table}->{'params'}->{'code'};
    error("no table code (or 0) for $table") if(!$table_code);
    my($spec_fields) = $spec->{'tables'}->{$table}->{'fields'};
    error("no fields spec for $table") if(!$spec_fields);
    #
    # Build a $action order for mifluz from Mysql values (see
    # mifluz.conf wordlist_wordkey_description for the meaning of
    # $key.
    #
    my($keys) = $self->hook_record2keys($row, $spec_fields, $table_code, $primary_value);

    if(@$keys > 0) {
	dbg("$action " . join(";", @$keys), "hook_mifluz");
	$self->exec("$action " . join(";", @$keys));
    }
}

sub hook_record2keys {
    my($self, $row, $spec_fields, $table_code, $primary_value) = @_;

    my($keys) = [];
    my($key) = "%s <DEF> $table_code %d $primary_value %d";

    my($field, $spec_field);
    while(($field, $spec_field) = each(%$spec_fields)) {
	dbg("call hook_keys for $field", "hook_mifluz");
	my($mifluz_field) = $spec_field->{'field'};
	hook_keys($keys, $row->{$field}, $mifluz_field, $key);
    }

    return $keys;
}

sub hook_insert {
    my($self, $table, $primary_value) = @_;
    my($mysql) = $self->{'mysql'};

    my($spec) = $self->{$mysql->{'base'}};
    return if(!exists($spec->{'tables'}->{$table}));
    my($info) = $mysql->info_table($table);
    my($primary_key) = $info->{'_primary_'};

    $self->hook_modify($table, $primary_value, "insert");
}

#
# Push in $keys a list of ascii key descriptions made with
# the $key template by filling $word, $field, $location. Each
# $word is extracted from $words. The location is the order number
# of the word. The $words string is splitted in words (\w) after
# removing accented chars.
#
sub hook_keys {
    my($keys, $words, $field, $key) = @_;

    return if(!defined($words));

    $words = lc(unaccent_8859($words));
    $words =~ s/_+/ /g;
    $words =~ s/^\W+//;
    my($location) = 0;
    my($word);
    foreach $word (split(/\W+/, $words)) {
	$location++;
	push(@$keys, sprintf($key, $word, $field, $location));
    }
}

sub hook_select {
    my($self, $relevance, $where, $order, $sql, $index, $length) = @_;

#    my($rows, $rows_total) = $self->select("$sql [low=<UNDEF> <DEF> 3 0 0 <UNDEF>]", $index, $length);
    my($rows, $rows_total) = $self->select("$sql", $index, $length);

    return (undef, 0) if(!@$rows);

    my($mysql) = $self->{'mysql'};

    my($spec) = $self->{$mysql->{'base'}};
    my($query_params) = $spec->{'query'}->{'params'};

    #
    # Build lookup map to find the table name from the code (the
    # spec file contains the map to find the code from the table name).
    #
    my($spec_tables) = $spec->{'tables'};
    my(%code2table) = map { $spec_tables->{$_}->{'params'}->{'code'} => $_ } keys(%$spec_tables);

    #
    # See mifluz.conf
    #
    my($rowid_pos) = 3;
    my($table_pos) = 1;
    
    #
    # Result is a list of rows with one entry named r_<table>
    # associated with the rowid found. The caller only expects this
    # information. 
    #
    my(@result) = map {
#	print STDERR $_->{'match'}->GetAscii() . "\n";
	my($code) = $_->{'match'}->Get($table_pos);
#	$code -= 4;
	my($table) = $code2table{$code};
	error("no table found for code $code") if(!$table);
	
	{ "r_$table" => $_->{'match'}->Get($rowid_pos) }
    } @$rows;

    return (\@result, $rows_total);
}

sub query2sql {
    my($self, $params) = @_;
    my($mysql) = $self->{'mysql'};

    my($spec) = $self->{$mysql->{'base'}};
    my($query_params) = $spec->{'query'}->{'params'};
    my($sql);
    my($stop) = $self->{'stop'};
    my($questions);
    if(defined($params->{'text'}) && $params->{'text'} !~ /^\s*$/o) {
	$questions = lc(unaccent_8859($params->{'text'}));
	$questions =~ s/[^\s\w:+\"-]//g;
    }

    my($scope_map) = '';
    if(exists($query_params->{'scope_map'})) {
	$scope_map = "[scope_map={ $query_params->{'scope_map'} }]";
    }

    my($method) = '';
    if(exists($params->{'method'}) && $params->{'method'} eq 'Advanced') {
	$method = '[method=Advanced]';
    }
    
    $sql = "select $questions $scope_map $method";

    dbg("query = $sql, questions = $questions", "hook_mifluz");
    return ($sql, '', $questions, '');
}

1;
# Local Variables: ***
# mode: perl ***
# End: ***
