/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _GNU_SOURCE /* asprintf() */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdbool.h>

#include "cdw_string.h"
#include "cdw_ext_tools.h"
#include "cdw_which.h"
#include "cdw_debug.h"
#include "cdw_sys.h"


static bool cdw_which_get(cdw_dll_t **list, const char *tool_name, char *PATH);




/**
   \brief Get a list of instances of given tool

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   Function fills \p list with list of full paths to instances
   (implementations) of a tool specified by \p tool_name.

   Function behaves similarly to "which -a" command. It was created because
   some systems (some versions of shell?) don't support "-a" option.

   \p list may be NULL, then the function will allocate the first element
   of the list (if there will be any full path found). \p list may also
   be non-NULL, then the function will continue appending new full paths.


   Function collects no more than CDW_EXT_TOOLS_N_INSTANCES_MAX instances
   of a tool.

   Function returns false on errors, and true otherwise. The fact
   that there may be no tool instances on user's system is not an error.
   Function may return false if PATH environment variable is not
   defined on user's system.

   \param tool_name - name of a tool to be found
   \param list - pointer to list, in which to store results

   \return false on errors
   \return true otherwise
*/
bool cdw_which(const char *tool_name, cdw_dll_t **list)
{
	cdw_assert (tool_name, "ERROR: tool name is NULL\n");
	cdw_assert (strlen(tool_name), "ERROR: passed empty tool name to the function (strlen = 0)\n");

	char *PATH = cdw_sys_env_path();
	if (!PATH) {
		cdw_vdm ("ERROR: failed to get PATH\n");
		return false;
	} else {

		bool success = cdw_which_get(list, tool_name, PATH);

		free(PATH);
		PATH = (char *) NULL;

		return success;
	}
}





/**
   \brief Collect paths to a tool

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   Go through all directories defined in \p PATH, search in them for
   executable called \p tool_name. If an executable is found, concatenate
   directory name with executable name, and append the full path to \p list.

   Function makes sure that there are no duplicate full paths on \p list.

   Function collects no more than CDW_EXT_TOOLS_N_INSTANCES_MAX instances
   of a tool.

   \p list can be either pointer to empty list, or to non-empty list.
   \p PATH should be obtained with cdw_sys_env_path(). PATH argument is
   modified by this function.

   \param list - pointer to list, in which to store full paths
   \param tool_name - name of searched executable
   \param PATH - string representing PATH environment variable

   \return false on errors
   \return true otherwise
*/
bool cdw_which_get(cdw_dll_item_t **list, const char *tool_name, char *PATH)
{
	cdw_vdm ("INFO: searching for tool \"%s\"\n", tool_name);
	size_t n_instances = 0;
	const char *delim = ":";
	char *dirpath = strtok(PATH, delim);
	bool success = true;
	while (dirpath) {

		size_t len = strlen(dirpath);
		if (len) {
			cdw_vdm ("INFO:    inspecting dirpath \"%s\"\n", dirpath);
			char *fullpath = (char *) NULL;
			int a = 0;
			if (dirpath[len - 1] == '/') {
				/* unlikely, unless user manually adds path
				   that ends with slash */
				a = asprintf(&fullpath, "%s%s", dirpath, tool_name);
			} else {
				a = asprintf(&fullpath, "%s/%s", dirpath, tool_name);
			}
			if (a == -1) {
				cdw_vdm ("ERROR: asprintf() fails\n");
				success = false;
				break;
			}

			if (!access(fullpath, X_OK)) {
				/* if there will be zero executable files then no
				   file will be put into table, even though the
				   table will be allocated */
				cdw_vdm ("INFO:      found: \"%s\"\n", fullpath);
				cdw_dll_append(list, (void *) fullpath, cdw_string_equal);
				n_instances++;
			} else {
				/* remember that cdw_dll_append() doesn't
				   duplicate appended data only puts pointer
				   on a list; so you can't free fullpath if
				   it is valid; here fullpath is invalid
				   (inaccessible), so we can free it */
				free(fullpath);
				fullpath = (char *) NULL;
			}
		} else { /* strlen(dirpath) == 0 - malformed PATH? */
			;
		}

		dirpath = strtok((char *) NULL, delim);
		if (n_instances == CDW_EXT_TOOLS_N_INSTANCES_MAX && dirpath) {
			cdw_vdm ("WARNING: not all paths checked, but number of instances now equals limit: %zd\n",
				 n_instances);
			break;
		}
	}

	return success;
}
