(include "test-infrastructure")

;; test chapter 4 out of the R5RS handbook
(define test-r5rs-ch4-expressions
	(lambda ()
		;; used in Simple Variable Reference
		(define test-var 42)

		;; used in Procedures
		(define reverse-subtract
			(lambda (x y) (- y x)))

		(define add4
			(let ((x 4))
				(lambda (y) (+ x y))))

		;; used in Assignments
		(define x 2)

		;; used in Sequencing
		(define y 0)

		;; a test-package holds test-cases grouped by purpose
		(test-package "Primitive Expression Types" pd pe

			(test-case "Simple Variable Reference" d e
				(expect-equal "Check (define ...)" 
					42 test-var))

			(test-case "Literal Expressions" d e
				(expect-equal "Check (quote ...) with symbols" 
					'a 'a)
				(expect-equal "Check (quote ...) with arrays" 
					'#(a b c) '#(a b c))
				(expect-equal "Check (quote ...) with expressions"
					'(+ 1 2) '(+ 1 2))
				(expect-equal "Check (quote ...) with quote itself"
					''a ''a)
				(expect-equal "Check (quote ...) with string constants"
					"abc" '"abc")
				(expect-equal "Check (quote ...) with numeric constants"
					1234 '1234)
				(expect-equal "Check (quote ...) with boolean true"
					#t '#t)
				(expect-equal "Check (quote ...) with boolean false"
					#f '#f))

			(test-case "Procedure Calls" d e
				(expect-equal "Try to call a function"
					7 (+ 3 4))
				(expect-equal "Can a function slot can be evaluated"
					12 ((if #f + *) 3 4)))

			(test-case "Procedures" d e
				(expect-equal "Lambda expression as function"
					8 ((lambda (x) (+ x x)) 4))
				(expect-equal "Lambda expression bound to a name"
					3 (reverse-subtract 7 10))
				(expect-equal "Lambda expression with closure"
					10 (add4 6))
				(expect-equal "Lambda expression variable arity"
					'(3 4 5 6) ((lambda x x) 3 4 5 6))
				(expect-equal "Lambda expression fixed + variable arity"
					'(5 6) ((lambda (x y . z) z) 3 4 5 6)))

			(test-case "Conditionals" d e
				(expect-equal "Test (if ... ) true condition"
					'yes (if (> 3 2) 'yes 'no))
				(expect-equal "Test (if ... ) false condition"
					'no (if (> 2 3) 'yes 'no))
				(expect-equal "Test (if ... ) general usage"
					1 (if (> 3 2) (- 3 2) (+ 3 2))))
		)
		
		(test-package "Derived Expression Types" pd pe

			(test-case "Conditionals" d e
				(expect-equal "Testing (cond ...)"
					'greater (cond ((> 3 2) 'greater) ((< 3 2) 'less)))
				(expect-equal "Testing (cond ...)"
					'equal (cond 	((> 3 3) 'greater) 
									((< 3 2) 'less)
									(else 'equal)))
				(expect-equal "Testing (cond ...)"
					2 (cond	((assv 'b '((a 1) (b 2))) => cadr)
							(else #f)))

				(expect-equal "Testing (case ...) normal scenario"
					'composite (case	(* 2 3)
										((2 3 5 7) 'prime)
										((1 4 6 8 9) 'composite)))
				(expect-equal "Testing (case ...) with unspecified"
					(if #f #t) (case	(car '(c d))
										((a) 'a)
										((b) 'b)))

				(expect-equal "Testing (and ...) with booleans"
					#t (and (= 2 2) (> 2 1)))
				(expect-equal "Testing (and ...) with booleans"
					#f (and (= 2 2) (< 2 1)))
				(expect-equal "Testing (and ...) with heterogeneous types"
					'(f g) (and 1 2 'c '(f g)))
				(expect-equal "Testing (and ...) short circut"
					#f (and #t #f (/ 1 0)))
				(expect-equal "Testing (and)"
					#t (and))

				(expect-equal "Testing (or ...) with booleans"
					#t (or (= 2 2) (> 2 1)))
				(expect-equal "Testing (or ...) with booleans"
					#t (or (= 2 2) (< 2 1)))
				(expect-equal "Testing (or ...) with booleans"
					#f (or #f #f #f))
				(expect-equal "Testing (or ...) short circut"
					'(b c) (or (memq 'b '(a b c)) (/ 3 0))))

			(test-case "Binding Constructs" d e
				(expect-equal "Basic let syntax"
					6 (let ((x 2) (y 3)) (* x y)))
				(expect-equal "Let syntax in environments"
					35 (let ((x 2) (y 3)) (let ((x 7) (z (+ x y))) (* z x))))

				(expect-equal "Let* syntax"
					70 (let ((x 2) (y 3)) (let* ((x 7) (z (+ x y))) (* z x))))

				(expect-equal "Letrec syntax"
					#t (letrec ((even? 
									(lambda (n)
										(if (zero? n)
											#t
											(odd? (- n 1)))))
								(odd?
									(lambda (n)
										(if (zero? n)
											#f
											(even? (- n 1))))))
							(even? 88))))

			(test-case "Sequencing" d e
				(expect-equal "Test of (begin ...)"
					6 (begin (set! y 5) (+ y 1))))

			(test-case "Iteration" d e
				(expect-equal "Testing (do ...)"
					'#(0 1 2 3 4) (do	(	(vec (make-vector 5))
											(i 0 (+ i 1)))
										((= i 5) vec)
									(vector-set! vec i i)))

				(expect-equal "Testing (do ..)"
					25 (let ((x '(1 3 5 7 9)))
							(do (	(x x (cdr x))
									(sum 0 (+ sum (car x))))
								((null? x) sum))))

				(expect-equal "Testing named let"
					'((6 1 3) (-5 -2))	
						(let loop (	(numbers '(3 -2 1 6 -5))
									(nonneg '())
									(neg '()))
							(cond	((null? numbers)
										(list nonneg neg))
									((>= (car numbers) 0)
										(loop	(cdr numbers)
												(cons (car numbers) nonneg)
												neg))
									((< (car numbers) 0)
										(loop	(cdr numbers)
												nonneg
												(cons (car numbers) neg)))))))
			(test-case "Delayed Evaluation" d e
				(let ((x 1))
				(let ((f (delay (+ x 3))))
					(expect-equal "Delay and Force"
						4 (force f))
					(set! x 22)
					(expect-equal "Delay and Force"
						4 (force f)))))

			(test-case "Quasiquotation" d e
				(expect-equal "Testing (quasiquote ...)"
					'(list 3 4) `(list ,(+ 1 2) 4))
				(expect-equal "Testing (quasiquote ...)"
					'(list a 'a) (let ((name 'a)) `(list ,name ',name)))
				(expect-equal "Testing (quasiquote ...)"
					'(a 3 4 5 6 b) `(a ,(+ 1 2) ,@(map abs '(4 -5 6)) b))
				(expect-equal "Testing (quasiquote ...)"
					(warn "Chicken expects the sqrt to be inexact.")
					'#(10 5 2. 4. 3. 8) 
						`#(10 5 ,(sqrt 4) ,@(map sqrt '(16 9)) 8))
				(expect-equal "Testing (quasiquote ...)"
					'(a `(b ,(+ 1 2) ,(foo 4 d) e) f)
						`(a `(b ,(+ 1 2) ,(foo ,(+ 1 3) d) e) f))
				(expect-equal "Testing (quasiquote ...)"
					'(a `(b ,x ,'y d) e)
						(let ((name1 'x) (name2 'y))
							`(a `(b ,,name1 ,',name2 d) e))))
		)
	)
)

;; test chapter 6 out of the R5RS handbook
(define test-r5rs-ch6-std-procedures
(lambda ()

	;; from page 18 in R5RS
	(define gen-counter
		(lambda ()
			(let ((n 0))
				(lambda () (set! n (+ n 1)) n))))

	;; from page 18 in R5RS
	(define gen-loser
		(lambda ()
			(let ((n 0))
				(lambda () (set! n (+ n 1)) 27))))

	(test-package "Equivalence predicates" pd pe
		(test-case "Testing eqv? conformance" p e

			(gloss "In these expectations, eqv? must return true")

			(expect-true "obj1 and obj2 are both #t"
				(eqv? #t #t))
			(expect-true "obj1 and obj2 are both #f"
				(eqv? #f #f))
			(expect-true "ob1 and ob2 are both symbols and uninterned"
				(let ((sym 'a))
					(and (string=? (symbol->string sym) (symbol->string sym))
						(eqv? sym sym))))
			(expect-true "obj1 and obj2 are numbers numerically equal"
				(eqv? 1 1))
			(expect-true 
				"obj1 and obj2 are the same character according to char=?"
				(and (char=? #\a #\a) (eqv? #\a #\a)))
			(expect-true "obj1 and obj2 are the empty list"
				(eqv? '() '()))
			(expect-true "obj1 and obj2 are pairs in the same locations"
				(let ((d '(1 2)))
					(eqv? d d)))
			(expect-true "obj1 and obj2 are vectors in the same locations"
				(let ((d '#(1 2)))
					(eqv? d d)))
			(expect-true "obj1 and obj2 are strings in the same locations"
				(let ((d "foobar"))
					(eqv? d d)))
			(expect-true "obj1 and obj2 are procedures in the same locations"
				(let ((d (lambda (x) x)))
					(eqv? d d)))

			(gloss "In these expectations, eqv? must return false")

			(expect-false "obj1 and obj2 are of different types"
				(warn "All permutations not being tested")
				(eqv? "obj1" 'obj1))
			(expect-false "obj1 is #t and obj2 is #f"
				(eqv? #t #f))
			(expect-false "obj1 and obj2 are symbols but not equal"
				(let ((sym1 'a) (sym2 'b))
					(and (string=? (symbol->string sym1) (symbol->string sym2))
						(eqv? sym1 sym2))))
			(expect-false "obj1 is an exact number obj2 is inexact"
				(eqv? 1. 1))
			(expect-false "obj1 and obj2 are numbers for which = is false"
				(warn "Funky expression here. Might not fail in the right way.")
				(and (not (= 1 2)) (eqv? 1 2)))
			(expect-false
				"obj1 and obj2 are different characters according to char=?"
				(warn "Funky expression here. Might not fail in the right way.")
				(and (not (char=? #\a #\b)) (eqv? #\a #\b)))
			(expect-false "obj1 is an empty list, obj2 is not"
				(eqv? '() '(1 2 3 4 5)))
			(expect-false "obj1 and obj1 are pairs in distinct locations"
				(let ((d '(1 2)) (e '(1 2)))
					(eqv? d e)))
			(expect-false "obj1 and obj1 are vectors in distinct locations"
				(let ((d '#(1 2)) (e '#(1 2)))
					(eqv? d e)))
			(expect-false "obj1 and obj2 are strings in distinct locations"
				(let ((d "foobar") (e "foobar"))
					(eqv? d e)))

			(todo "Figure out statement about procedures on page 18 with eqv?")

			(gloss "These are unspecified in the R5RS handbook")

			(expect-false "obj1 and obj2 are \"\""
				(warn "CHICKEN decides to return #f")
				(eqv? "" ""))
			(expect-false "obj1 and obj2 are '#()"
				(warn "CHICKEN decides to return #f")
				(eqv? '#() '#()))
			(expect-false "obj1 and obj2 are lambda expressions"
				(warn "CHICKEN decides to return #f")
				(eqv? (lambda (x) x) (lambda (x) x)))
			(expect-false "obj1 and obj2 are alpha-converted lambda expressions"
				(warn "CHICKEN decides to return #f")
				(eqv? (lambda (x) x) (lambda (y) y)))

			(gloss "eqv? and procedures with local state")
			
			(expect-true "obj1 and obj2 are the *same* procedure with state"
				(let ((g (gen-counter)))
					(eqv? g g)))
			(expect-false "obj1 and obj2 are distinct procedures with state"
				(eqv? (gen-counter) (gen-counter)))
			(expect-true "obj1 and obj2 are the *same* procedure with state"
				(let ((g (gen-loser)))
					(eqv? g g)))
			(expect-false "obj1 and obj2 are distinct procedures with state"
				(warn "CHICKEN decides to return #f")
				(eqv? (gen-counter) (gen-counter)))

			(expect-false "obj1 and obj2 are predictive result functions (1)"
				(warn "CHICKEN decides to return #f")
				(letrec (	(f (lambda () (if (eqv? f g) 'both 'f)))
							(g (lambda () (if (eqv? f g) 'both 'g))))
					(eqv? f g)))

			(expect-false "obj1 and obj2 are predictive result functions (2)"
				(letrec (	(f (lambda () (if (eqv? f g) 'f 'both)))
							(g (lambda () (if (eqv? f g) 'g 'both))))
					(eqv? f g)))
			
			(gloss "Depending on the implementation, these could be true/false")

			(expect-false "obj1 and obj2 are quotes of the same list"
				(eqv? '(a) '(a)))
			(expect-false "obj1 and obj2 are the same string"
				(eqv? "foobar" "foobar"))
			(expect-false "obj1 and obj2 are the structurally similar"
				(eqv? '(b) (cdr '(a b))))
		)

		(test-case "Testing eq? conformance" p e
			(warn "eq? is not being tested fully yet in this test-case")
			(expect-true "obj1 and obj2 are symbols"
				(eq? 'a 'a))
			(expect-false "obj1 and obj2 are the same quoted list"
				(warn "CHICKEN decides to return #f")
				(eq? '(a) '(a)))
			(expect-false "obj1 and obj2 are structurally similar lists"
				(eq? (list 'a) (list 'a)))
			(expect-false "obj1 and obj2 are similar strings"
				(warn "CHICKEN decides to return #f")
				(eq? "foobar" "foobar"))
			(expect-true "obj1 and obj2 are '()"
				(eq? '() '()))
			(expect-true "obj1 and obj2 are numerically equal numbers"
				(warn "CHICKEN decides to return #t")
				(eq? 2 2))
			(expect-true "obj1 and obj2 are equal characters"
				(warn "CHICKEN decides to return #t")
				(eq? #\A #\A))
			(expect-true "obj1 and obj2 are the same function location"
				(eq? car car))
			(expect-true "obj1 and obj2 are bound the same allocated location"
				(warn "CHICKEN decides to return #t")
				(let ((n (+ 2 3)))
					(eq? n n )))
			(expect-true "obj1 and obj2 are bound to the same literal location"
				(let ((n '(a)))
					(eq? n n )))
			(expect-true "obj1 and obj2 are bound to the same literal location"
				(let ((x '#()))
					(eq? x x )))
			(expect-true "obj1 and obj2 are bound to the same lambda function"
				(let ((p (lambda (x) x)))
					(eq? p p )))
		)

		(test-case "Testing equal? conformance" p e
			(expect-true "obj1 and obj2 are the same symbol"
				(equal? 'a 'a))
			(expect-true "obj1 and obj2 are structurally similar literal lists"
				(equal? '(a) '(a)))
			(expect-true "obj1 and obj2 are structurally similar literal lists"
				(equal? '(a (b) c) '(a (b) c)))
			(expect-true "obj1 and obj2 are similar literal strings"
				(equal? "foobar" "foobar"))
			(expect-true "obj1 and obj2 are numerically equal numbers"
				(equal? 2 2))
			(expect-true "obj1 and obj2 are structurally equal allocated space"
				(equal? (make-vector 5 'a) (make-vector 5 'a)))
			(expect-false "obj1 and obj2 are structurally equivalent functions"
				(warn "CHICKEN decides to return #f")
				(equal? (lambda (x) x) (lambda (y) y)))
		)
	)

	(test-package "Numbers" pd pe
		(warn "CHICKEN implementes a subset of the Numerical Tower")
		(todo "Perform a more exacting test of the numerical tower")
		(todo "Check numerical predicates")
		(test-case "Numerical operations" p e
			(expect-true "Check exactness predicate (exact object)"
				(exact? 2))
			(expect-false "Check exactness predicate (inexact object)"
				(exact? 2.0))
			(expect-false "Check inexactness predicate (exact object)"
				(inexact? 2))
			(expect-true "Check inexactness predicate (inexact object)"
				(inexact? 2.0))

			(gloss "Exact Integer Operations for the next few relational ops")
			(expect-true "Check ="
				(= 1 1 1 1 1))
			(expect-true "Check <"
				(< 1 2 3 4 5))
			(expect-true "Check >"
				(> 5 4 3 2 1))
			(expect-true "Check <="
				(<= 1 2 3 4 5))

			(gloss "Inexact Integer Operations for the next few relational ops")
			(expect-true "Check ="
				(= 1.0 1.0 1.0 1.0 1.0))
			(expect-true "Check <"
				(< 1.0 2.0 3.0 4.0 5.0))
			(expect-true "Check >"
				(> 5.0 4.0 3.0 2.0 1.0))
			(expect-true "Check <="
				(<= 1.0 2.0 3.0 4.0 5.0))

			(expect-equal "Check + exact form"
				7
				(+ 3 2 2))
			(expect-equal "Check + exact form"
				7
				(+ 3 4))
			(expect-equal "Check + exact form"
				3
				(+ 3))
			(expect-equal "Check + exact form"
				0
				(+))

			(expect-equal "Check + inexact form"
				7
				(+ 3 2 2))
			(expect-equal "Check + inexact form"
				7
				(+ 3 4))
			(expect-equal "Check + inexact form"
				3
				(+ 3))
			(expect-equal "Check + inexact form"
				0
				(+))

			(expect-equal "Check - exact form"
				7
				(- 10 2 1))
			(expect-equal "Check - exact form"
				-1
				(- 3 4))
			(expect-equal "Check - exact form"
				-3
				(- 3))

			(expect-equal "Check - inexact form"
				7.0
				(- 10 2.0 1))
			(expect-equal "Check - inexact form"
				-1.0
				(- 3 4.0))
			(expect-equal "Check - inexact form"
				-3.0
				(- 3.0))

			(expect-equal "Check * exact form"
				20
				(* 10 2 1))
			(expect-equal "Check * exact form"
				12
				(* 3 4))
			(expect-equal "Check * exact form"
				3
				(* 3))
			(expect-equal "Check * exact form"
				1
				(*))

			(expect-equal "Check * inexact form"
				20.0
				(* 10 2.0 1))
			(expect-equal "Check * inexact form"
				12.0
				(* 3 4.0))
			(expect-equal "Check * inexact form"
				3.0
				(* 3.0))

			(expect-near "Check / exact form"
				5 .000001
				(/ 10 2 1))
			(expect-near "Check / exact form"
				5 .000001
				(/ 10 2))
			(expect-near "Check / exact form"
				.333333 .000001
				(/ 3))

			(expect-near "Check / inexact form"
				5.0 .000001
				(/ 10 2.0 1))
			(expect-near "Check / inexact form"
				5.0 .000001
				(/ 10 2.0))
			(expect-near "Check / inexact form"
				.33333333 .000001
				(/ 3.0))
		)

		(test-case "Numerical Functions" p e
			(expect-true "Check zero?"
				(zero? 0))
			(expect-false "Check zero?"
				(zero? 1))
			(expect-false "Check zero?"
				(zero? -1))

			(expect-true "Check zero?"
				(zero? 0.0))
			(expect-false "Check zero?"
				(zero? 1.0))
			(expect-false "Check zero?"
				(zero? -1.0))

			(expect-false "Check positive?"
				(positive? 0))
			(expect-true "Check positive?"
				(positive? 1))
			(expect-false "Check positive?"
				(positive? -1))

			(expect-false "Check positive?"
				(positive? 0.0))
			(expect-true "Check positive?"
				(positive? 1.0))
			(expect-false "Check positive?"
				(positive? -1.0))

			(expect-false "Check negative?"
				(negative? 0))
			(expect-false "Check negative?"
				(negative? 1))
			(expect-true "Check negative?"
				(negative? -1))

			(expect-false "Check negative?"
				(negative? 0.0))
			(expect-false "Check negative?"
				(negative? 1.0))
			(expect-true "Check negative?"
				(negative? -1.0))
			
			(expect-true "Check odd?"
				(odd? 1))
			(expect-false "Check odd?"
				(odd? 2))
			(expect-true "Check even?"
				(even? 2))
			(expect-false "Check even?"
				(even? 1))

			(expect-true "Check odd?"
				(odd? 1.0))
			(expect-false "Check odd?"
				(odd? 2.0))
			(expect-true "Check even?"
				(even? 2.0))
			(expect-false "Check even?"
				(even? 1.0))

			(expect-equal "Check max exact form"
				4
				(max 1 4 3 2))
			(expect-equal "Check max inexact form"
				4.0
				(max 1.0 4 3 2))

			(expect-equal "Check abs exact form"
				1
				(abs 1))
			(expect-equal "Check abs exact form"
				1
				(abs -1))

			(expect-near "Check abs inexact form"
				1.0 .000001
				(abs 1.0))
			(expect-near "Check abs inexact form"
				1.0 .000001
				(abs -1.0))
		)
	)
))

;; run the entire r5rs test suite and collect the result tree
(define r5rs-test-suite
	(lambda ()
		(test-package "R5RS Test Suite" pd pe
			(test-package "R5RS Expressions" ppd ppe
				(test-r5rs-ch4-expressions))
			(test-package "R5RS Standard Procedures" ppd ppe
				(test-r5rs-ch6-std-procedures))
		)
))

;; print out the result tree
(let ((result-tree (r5rs-test-suite)))
	(output-human-simple result-tree))






