// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/nearby_sharing/fake_nearby_connection.h"

FakeNearbyConnection::FakeNearbyConnection() = default;
FakeNearbyConnection::~FakeNearbyConnection() {
  if (!closed_)
    Close();
}

void FakeNearbyConnection::Read(ReadCallback callback) {
  DCHECK(!closed_);
  callback_ = std::move(callback);
  MaybeRunCallback();
}

void FakeNearbyConnection::Write(std::vector<uint8_t> bytes) {
  DCHECK(!closed_);
  write_data_.push(std::move(bytes));
}

void FakeNearbyConnection::Close() {
  DCHECK(!closed_);
  closed_ = true;
  if (disconnect_listener_)
    std::move(disconnect_listener_).Run();

  if (callback_)
    std::move(callback_).Run(base::nullopt);
}

void FakeNearbyConnection::SetDisconnectionListener(
    base::OnceClosure listener) {
  DCHECK(!closed_);
  disconnect_listener_ = std::move(listener);
}

void FakeNearbyConnection::AppendReadableData(std::vector<uint8_t> bytes) {
  DCHECK(!closed_);
  read_data_.push(std::move(bytes));
  MaybeRunCallback();
}

std::vector<uint8_t> FakeNearbyConnection::GetWrittenData() {
  if (write_data_.empty())
    return {};

  std::vector<uint8_t> bytes = std::move(write_data_.front());
  write_data_.pop();
  return bytes;
}

bool FakeNearbyConnection::IsClosed() {
  return closed_;
}

void FakeNearbyConnection::MaybeRunCallback() {
  DCHECK(!closed_);
  if (!callback_ || read_data_.empty())
    return;
  auto item = std::move(read_data_.front());
  read_data_.pop();
  std::move(callback_).Run(std::move(item));
}
