/*
 * ShiftMenu.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  ShiftMenu.cpp                 (Bruchstueck)
  
  Dieses Menu wird aus dem Hauptmenu als eigenstaendiges Top-Window
  emaniert. Er kommuniziert (auch schon im Konstruktor!) direkt mit der
  vorher(!) initialisiert sein muessenden Camera-Instanz, referenziert 
  durch den globalen Zeiger `Br.camera()'. 
  
  Fehlt event_msg()-Routine zur Event-Verarbeitung.
*/
#include "br_defs.hpp"

#ifdef BR_WITH_SHIFT


#include <cassert>
#include <cstdio>
#include <cstring>

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/fl_draw.H>
#include <FL/filename.H>        // fl_filename_relative()
#include "Fl_Table_Row.H"

#include "ShiftMenu.hpp"
#include "Br.hpp"               // Br.imgVec & Br.camera()
//#include  "print_event.hpp"   



class ShiftDetailTable : public Fl_Table_Row
{
protected:
    void draw_cell(TableContext context,        // table cell drawing
               int R=0, int C=0, int X=0, int Y=0, int W=0, int H=0);
//    void callback(TableContext context,       // callback for table events
//             int R, int C);

public:
    ShiftDetailTable(int x, int y, int w, int h, const char *l=0) 
      : Fl_Table_Row(x,y,w,h,l)
      { CTOR(l); end(); }          // end() - hier und in ShiftMenu()?
    
    ~ShiftDetailTable() { DTOR(label()) }
};

// Handle drawing all cells in table
void ShiftDetailTable::draw_cell(TableContext context, 
                                 int R, int C, int X, int Y, int W, int H)
{
    //printf("draw_cell(): contex=%d, R=%d, C=%d\n", context,R,X);
    static char s[128];
    sprintf(s, "%d/%d", R, C);      // text for each cell
    int RR=0;   // camera-index; != imgVec-index if inactive images!

    switch ( context )
    {
    case CONTEXT_STARTPAGE:
        fl_font(FL_HELVETICA, 16);
        return;

    case CONTEXT_COL_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          switch(C) 
          {
            case 0:  fl_draw("Spot",  X, Y, W, H, FL_ALIGN_CENTER); break;
            case 1:  fl_draw("dx, dy",X, Y, W, H, FL_ALIGN_CENTER); break;
            case 2:  fl_draw("Corr",  X, Y, W, H, FL_ALIGN_CENTER); break;
            default: fl_draw(s,       X, Y, W, H, FL_ALIGN_CENTER);
          }
        }
        fl_pop_clip();
        return;

    case CONTEXT_ROW_HEADER:
        printf("used() = %d\n", Br.used(R));
        fl_push_clip(X, Y, W, H);
        {
          sprintf(s, "%d", R);
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
        }
        fl_pop_clip();
        return;
    
    case CONTEXT_CELL:
        fl_push_clip(X, Y, W, H);
        {
          // BG COLOR
          fl_color( row_selected(R) ? selection_color() : FL_WHITE);
          fl_rectf(X, Y, W, H);

          // TEXT
          Br.imgVec[R].active ? fl_color(FL_BLACK) : fl_color(FL_DARK3);
          switch (C)
          {
          case 0:
              sprintf(s, "%d", R);
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); break;
          case 1:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              sprintf(s, "R: %3d, %3d", Br.camera()->channel_shifts[RR].r.x,
                                        Br.camera()->channel_shifts[RR].r.y);
              fl_draw(s, X,Y+H/3);              
              sprintf(s, "G: %3d, %3d", Br.camera()->channel_shifts[RR].g.x,
                                        Br.camera()->channel_shifts[RR].g.y);
              fl_draw(s, X,Y+2*H/3-1);              
              sprintf(s, "B: %3d, %3d", Br.camera()->channel_shifts[RR].b.x,
                                        Br.camera()->channel_shifts[RR].b.y);
              fl_draw(s, X,Y+H-2); break;            
          case 2:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              fl_draw("R:", X,Y+H/3);              
              fl_draw("G:", X,Y+2*H/3-1);              
              fl_draw("B:", X,Y+H-2); break;             
          default:
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
          }

          // BORDER
          fl_color(FL_LIGHT2); 
          fl_rect(X, Y, W, H);
        }
        fl_pop_clip();
        return;

    default:
        return;
    }
}


class ShiftDetailInfo : public Fl_Window
{ 
  ShiftDetailTable* table_;
public:
  ShiftDetailInfo (int W, int H, const char* la=0);
};
  
ShiftDetailInfo::ShiftDetailInfo (int W, int H, const char* la)
  : Fl_Window (W,H,la)
{
  begin();
    table_ = new ShiftDetailTable(0,0,W,H,la);
    table_ -> selection_color(FL_YELLOW);
    table_ -> type(Fl_Table_Row::SELECT_SINGLE);
    table_ -> when(FL_WHEN_RELEASE);   // handle table events on release
    //table_ -> when(FL_WHEN_CHANGED|FL_WHEN_RELEASE);
    table_ -> rows(1);  // number of spots!
    table_ -> cols(3);
    table_ -> col_width(0,40);
    table_ -> col_width(1,80);
    table_ -> col_width(2,60);
    table_ -> row_height_all(60);
    table_ -> col_header(1);       // enable col header
    table_ -> col_resize(4);       // enable col resizing
    table_ -> row_header(0);       // disable row header
    table_ -> row_resize(4);       // enable row resizing
    //table_ -> callback(table_cb, table_);
    //table_ -> callback((Fl_Callback*)cb_table_, this);
    table_ -> end();
  end();
  show();
}


ShiftDetailInfo* pShiftDetailInfo;


/**==========================================================
 *
 * DemoTabel, eigentlich ShiftTable
 * 
 *==========================================================*/
class DemoTable : public Fl_Table_Row
{
protected:
    void draw_cell(TableContext context,        // table cell drawing
               int R=0, int C=0, int X=0, int Y=0, int W=0, int H=0);
    //void callback(TableContext context,       // callback for table events
    //         int R, int C)
    //{ printf("Table Callback: context=%d, R=%d, C=%d\n", context, R, C); }

public:
    DemoTable(int x, int y, int w, int h, const char *la=0) 
      : Fl_Table_Row(x,y,w,h,la)
      { CTOR(la); end(); }         // end() - hier und in ShiftMenu()?
    
    ~DemoTable() { DTOR(label()) }
};

// Handle drawing all cells in table
void DemoTable::draw_cell(TableContext context, 
              int R, int C, int X, int Y, int W, int H)
{
    //printf("draw_cell(): contex=%d, R=%d, C=%d\n", context,R,X);
    static char s[128];
    sprintf(s, "%d/%d", R, C);      // text for each cell
    int RR=0;   // camera-index: != imgVec-index if inactive images

    switch ( context )
    {
    case CONTEXT_STARTPAGE:
        fl_font(FL_HELVETICA, 16);
        return;

    case CONTEXT_COL_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          switch(C) 
          {
            case 0:  fl_draw("Name",  X, Y, W, H, FL_ALIGN_CENTER); break;
            case 1:  fl_draw("Groesse", X, Y, W, H, FL_ALIGN_CENTER); break;
            case 2:  fl_draw("dx",    X, Y, W, H, FL_ALIGN_CENTER); break;
            case 3:  fl_draw("dy",    X, Y, W, H, FL_ALIGN_CENTER); break;
            case 4:  fl_draw("Corr",  X, Y, W, H, FL_ALIGN_CENTER); break;
            case 5:  fl_draw("dx,dy", X, Y, W, H, FL_ALIGN_CENTER); break;
            default: fl_draw(s,       X, Y, W, H, FL_ALIGN_CENTER);
          }
        }
        fl_pop_clip();
        return;

    case CONTEXT_ROW_HEADER:
        printf("used() = %d\n", Br.used(R));
        fl_push_clip(X, Y, W, H);
        {
          sprintf(s, "%d", R);
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
        }
        fl_pop_clip();
        return;
    
    case CONTEXT_CELL:
        fl_push_clip(X, Y, W, H);
        {
          // BG COLOR
          fl_color( row_selected(R) ? selection_color() : FL_WHITE);
          fl_rectf(X, Y, W, H);

          // TEXT
          Br.imgVec[R].active ? fl_color(FL_BLACK) : fl_color(FL_DARK3);
          switch (C)
          {
          case 0:
              fl_filename_relative(s, Br.imgVec[R].name());
              fl_draw(s, X, Y, W, H, FL_ALIGN_LEFT); break;
          case 1:
              snprintf(s, 128, "%d x %d", Br.imgVec[R].width(), Br.imgVec[R].height());
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); break;
          case 2:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              //printf("R=%d, RR=%d\n", R, RR);
              snprintf(s, 128, "%d", Br.camera()->x_shift(RR));
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); break;
          case 3:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              snprintf(s, 128, "%d", Br.camera()->y_shift(RR));
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); break;
          case 4:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              fl_font(FL_HELVETICA, 13);
              fl_draw("R:", X,Y+H/3);              
              fl_draw("G:", X,Y+2*H/3-1);              
              fl_draw("B:", X,Y+H-2);              
              fl_font(FL_HELVETICA, 16); break;
          case 5:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              fl_font(FL_HELVETICA, 13);
              sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].r.x,
                                     Br.camera()->channel_shifts[RR].r.y);
              fl_draw(s, X,Y+H/3);              
              sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].g.x,
                                     Br.camera()->channel_shifts[RR].g.y);
              fl_draw(s, X,Y+2*H/3-1);              
              sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].b.x,
                                     Br.camera()->channel_shifts[RR].b.y);
              fl_draw(s, X,Y+H-2);              
              fl_font(FL_HELVETICA, 16); break;
              
          default:
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
          }

          // BORDER
          fl_color(FL_LIGHT2); 
          fl_rect(X, Y, W, H);
        }
        fl_pop_clip();
        return;

    default:
        return;
    }
}

//-------------------aus "testtablerow.cxx"------------------------------
// Beispiel. Wobei mir unklar, warum `data' und nicht `o' ausgewertet,
// da doch auch `o' schon das rufende Widget, also die Tabelle ist.
//-----------------------------------------------------------------------
void table_cb(Fl_Widget* o, void* data)
{
  Fl_Table *table = (Fl_Table*)data;
  fprintf(stderr, "calling widget = %p,  data = %p\n", o,data);
  fprintf(stderr, "%s callback: row=%d col=%d, context=%d, event=%d  clicks=%d\n",
      (const char*)table->label(),
      (int)table->callback_row(),
      (int)table->callback_col(),
      (int)table->callback_context(),
      (int)Fl::event(),
      (int)Fl::event_clicks());
}


typedef enum { SPOT_ONE, SPOT_2x2, SPOT_4_1, SPOT_3x3 }  SPOT_MODE;

struct { const char* text; SPOT_MODE mode; }
menuMap_spots[] = {
  {"1",   SPOT_ONE},
  {"2x2", SPOT_2x2},
  {"4+1", SPOT_4_1},
  {"3x3", SPOT_3x3}
};

/**================================================================
 *
 * ShiftMenu  -  class 
 *
 * Gedacht als eigenstaendiges Hauptfenster.
 *
 *=================================================================*/
//--------------------------------------------------
// menu_menubar (definition of an static element)...  
//--------------------------------------------------
Fl_Menu_Item ShiftMenu::menu_menubar_[] = {
  {"&File",  0,0,0, FL_SUBMENU},
    {"Load (fut)", 0,0,0, FL_MENU_INACTIVE},
    {"Save (fut)", 0,0,0, FL_MENU_INACTIVE},
    {"&Close", FL_ALT+'c', (Fl_Callback*)ShiftMenu::cb_close_},
    {0},
  {"Spots", 0,0,0, FL_SUBMENU},
    {menuMap_spots[0].text},
    {menuMap_spots[1].text},
    {menuMap_spots[2].text},
    {menuMap_spots[3].text},
    {0},
  {"Compute", 0, (Fl_Callback*)ShiftMenu::cb_compute_},
  {0}
};

/**---------------
 * Constructor... 
 *----------------*/
ShiftMenu::ShiftMenu(int W, int H, const char* la)
  : Fl_Window (W,H, la)
{ 
  CTOR(la)
  assert(Br.camera());  // so denn ueberhaupt FindShift-Werkzeuge in Camera
                      // eingegliedert
  begin();
    { Fl_Menu_Bar* mb = new Fl_Menu_Bar(0,0,W,30); 
      mb->menu(menu_menubar_);
      mb->user_data(this);       // this-Zeiger deponieren
    }
    
    int t1x = 20, t1y = 50,   t1w = W - 40, t1h = H - 150;
    
    DemoTable* table1 = new DemoTable (t1x, t1y, t1w, t1h, "Table 1");
    table_ = (Fl_Table_Row*) table1;
    table1 -> selection_color(FL_YELLOW);
    table1 -> type(Fl_Table_Row::SELECT_SINGLE);
    table1 -> when(FL_WHEN_RELEASE);   // handle table events on release
    //table1 -> when(FL_WHEN_CHANGED|FL_WHEN_RELEASE);
    table1 -> rows(Br.imgVec.size());
    table1 -> cols(6);
    table1 -> col_width(0,80);
    table1 -> col_width(1,100);
    table1 -> col_width(2,40);
    table1 -> col_width(3,40);
    table1 -> row_height_all(40);
    table1 -> col_header(1);       // enable col header
    table1 -> col_resize(4);       // enable col resizing
    table1 -> row_header(1);       // enable row header
    table1 -> row_resize(4);       // enable row resizing
    //table1 -> callback(table_cb, table1);
    table1 -> callback((Fl_Callback*)cb_table_, this);
    table1 -> end();
    
    { Fl_Int_Input* o = new Fl_Int_Input(30, H-70, 40, 25, "nx");
      input_nx_ = o;        
      o -> callback((Fl_Callback*)cb_input_nx_, this);
    }
    { Fl_Int_Input* o = new Fl_Int_Input(30, H-40, 40, 25, "ny");
      input_ny_ = o;        
      o -> callback((Fl_Callback*)cb_input_ny_, this);
    }
    { Fl_Int_Input* o = new Fl_Int_Input(120, H-70, 40, 25, "mx");
      input_mx_ = o;        
      o -> callback((Fl_Callback*)cb_input_mx_, this);
    }
    { Fl_Int_Input* o = new Fl_Int_Input(120, H-40, 40, 25, "my");
      input_my_ = o;        
      o -> callback((Fl_Callback*)cb_input_my_, this);
    }
    set_nm_values();
  end();
  
  callback((Fl_Callback*)cb_window_, this);
  resizable(table1);
}


/**--------------------------------
 * Calling when window is closed...
 *---------------------------------*/
void ShiftMenu::cb_window()   
{
  WINCALL(label())
  hide();           // BTW: this is the default
  if (pShiftDetailInfo) { delete pShiftDetailInfo; pShiftDetailInfo=0; }
}

/**--------------------------------
 * Callback for the Compute item...
 *---------------------------------*/
void ShiftMenu::cb_compute (Fl_Menu_*) 
{
  printf("Compute...\n");
  Br.camera() -> calc_shifts();
  redraw();
  if (!pShiftDetailInfo) 
    pShiftDetailInfo = new ShiftDetailInfo(300,200,"Details");
}

/**-------------------------
 * Callback for the Table...
 *--------------------------*/
void ShiftMenu::cb_table (Fl_Table* table)
{
  //Fl_Table *table = (Fl_Table*)data;
  fprintf(stderr, "%s callback: row=%d col=%d, context=%d, event=%d  clicks=%d\n",
      (const char*)table->label(),
      (int)table->callback_row(),
      (int)table->callback_col(),
      (int)table->callback_context(),
      (int)Fl::event(),
      (int)Fl::event_clicks());
}

/**----------------------------------------------
 * Callback for the n*-m*-inputs... 
 *-----------------------------------------------*/
void ShiftMenu::cb_input_nx (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera() -> shift_nx = res;
  //set_nm_values(res);   // in Menus eintragen
}
void ShiftMenu::cb_input_ny (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera() -> shift_ny = res;
  //set_nm_values(res);   // in Menus eintragen
}
void ShiftMenu::cb_input_mx (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera() -> shift_mx = res;
  //set_nm_values(res);   // in Menus eintragen
}
void ShiftMenu::cb_input_my (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera() -> shift_my = res;
  //set_nm_values(res);   // in Menus eintragen
}

void ShiftMenu::set_nm_values()
{
  char s[32];
  snprintf(s, 32, "%d", Br.camera()->shift_nx); input_nx_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_ny); input_ny_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_mx); input_mx_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_my); input_my_-> value(s);
}

#endif // BR_WITH_SHIFT

// END OF FILE
