# +==========================================================================+
# || CipUX::RPC                                                             ||
# ||                                                                        ||
# || CipUX RPC Basis Class                                                  ||
# ||                                                                        ||
# || Copyright (C) 2007 - 2009 by Christian Kuelker                         ||
# ||                                                                        ||
# || License: GNU GPL - GNU General Public License - version 2              ||
# ||          or (at your opinion) any later version.                       ||
# ||                                                                        ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RPC;

use 5.008001;
use strict;
use warnings;
use Class::Std;
use base qw(CipUX);
use Carp;
use Data::Dumper;
use English qw( -no_match_vars);
use Log::Log4perl qw(:easy);
use Readonly;

{    # BEGIN CLASS

    use version; our $VERSION = qv('3.4.0.8');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};

    # +======================================================================+
    # || OBJECT                                                             ||
    # +======================================================================+
    my %rpc_cfg : ATTR( :get<rpc_cfg> );

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+

    # +======================================================================+
    # || CONSTRUCTOR                                                        ||
    # +======================================================================+
    sub BUILD {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $obj_id, $arg_r ) = @_;

        # add prefix for cfg, if needed
        my $pref
            = exists $arg_r->{pref}
            ? $self->l( $arg_r->{pref} )
            : $EMPTY_STRING;

        my $cache_dir
            = exists $arg_r->{cache_dir}
            ? $self->l( $arg_r->{cache_dir} )
            : $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | prepare

        # +------------------------------------------------------------------+
        # | main
        $rpc_cfg{$obj_id} = $self->cfg(
            { 'sub' => 'rpc', pref => $pref, cache_dir => $cache_dir } );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +======================================================================+
    # || DESTRUCTOR                                                         ||
    # +======================================================================+
    sub DEMOLISH {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident ) = @_;

        # +------------------------------------------------------------------+
        # | main
        delete $rpc_cfg{$ident};

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || open module features                                               ||
    # +======================================================================+

}    # END INSIDE-OUT CLASS

1;

__END__

=pod

=head1 NAME

CipUX::RPC - RPC server base class for CipUX

=head1 VERSION

version 3.4.0.8

=head1 SYNOPSIS

  use CipUX::RPC;

=head1 DESCRIPTION

Provides the functions for CipUX RPC server.

=head1 ABSTRACT

CipUX::RPC is a generic abstract class, which can be used by other classes or
scripts.

=head1 SUBROUTINES/METHODS

The following methods are available in the CipUX::RPC class.

=head2 BUILD

See new.

=head2 DEMOLISH

Destructor call.

=head2 new

Constructor

B<Syntax:>


  my $cipux_rpc = CipUX::RPC->new({});


  my $cfg = '/etc/cipux/cipux-rpc.ini';
  my $cipux_rpc = CipUX::RPC->new({pref=>$cfg});


B<Description:>

I<pref>

Configuration files may be provided for convenience but is not needed. If no
configuration file was provided the default configuration file will be loaded.
This is meant to work in a single server scenario and multi server settings.

=head1 DIAGNOSTICS

None.

=head1 CONFIGURATION AND ENVIRONMENT

CipUX::RPC do not need environemnt variables. It need and provid the CipUX
XML-RPC server configration file.

=head1 DEPENDENCIES

 Carp
 CipUX
 Class::Std
 Data::Dumper
 English
 Log::Log4perl
 Readonly

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX webpage and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2007 - 2009 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut

