;; -*- Mode: Lisp -*-
;; $Id: rmd160.lisp,v 1.2 2002/03/08 18:50:53 apharris Exp $

;; rmd160.lisp - RIPE-MD Hash Algorithm
;;
;; Copyright (C) 2000, Jesse Bouwman
;; This implementation is placed under the public domain.
;;
;; Based on the RIPE-MD implementation by Nikos Mavroyanopoulos
;;
;; Original copyright follows.

;; ripemd.c - Implementation of the RIPE-MD Hash Algorithm
;;
;; Copyright (C) 2000, Nikos Mavroyanopoulos
;; This implementation is placed under the public domain.
;;
;; Based on the SHA-1 implementation by A.M. Kuchling
;;
;; Here are the quotes of the original SHA-1 implementation: 

;; Copyright (C) 1995, A.M. Kuchling
;; Adapted to pike and some cleanup by Niels Mller.
;;
;; Based on SHA code originally posted to sci.crypt by Peter Gutmann
;; in message <30ajo5$oe8@ccu2.auckland.ac.nz>.
;; Modified to test for endianness on creation of SHA objects by AMK.
;; Also, the original specification of SHA was found to have a weakness
;; by NSA/NIST.  This code implements the fixed version of SHA.

(defpackage :crypt)

(in-package :crypt)

;(declaim (optimize (speed 3)
;                   (debug 1)
;                   (safety 1)))

(defconstant +rmd160-init+
  '(#x67452301 #xefcdab89 #x98badcfe #x10325476 #xc3d2e1f0))

(defstruct rmd160-ctx
  (regs (make-array 5 :element-type '(unsigned-byte 32)
                    :initial-contents +rmd160-init+)
        :type (array (unsigned-byte 32) *))
  (input (make-array 16 :element-type '(unsigned-byte 32))
         :type (array (unsigned-byte 32) *)))


(defun clear-ctx-input (ctx)
  (dotimes (i 16)
    (setf (aref (rmd160-ctx-input ctx) i) 0)))

(declaim (inline rotate-32 ones-comp-32 chop-32))

(defun rotate-32 (x n)
  (declare (type (unsigned-byte 32) x)
           (type (mod 32) n))
  (logior (mod (ash x n) (ash 1 32))
          (ash x (* -1 (- 32 n)))))

(defun ones-comp-32 (x)
  (declare (type (unsigned-byte 32) x))
  (- (ash 1 32) 1 x))

(defun chop-32 (x)
  (mod x (ash 1 32)))

;; the five basic functions F(), G() and H() 

(declaim (inline F G H I J))

(defun F (x y z)
  (declare (type (unsigned-byte 32) x y z))
  (logxor x y z))

(defun G (x y z)
  (declare (type (unsigned-byte 32) x y z))
  (logior (logand x y)
          (logand (ones-comp-32 x) z)))

(defun H (x y z)
  (declare (type (unsigned-byte 32) x y z))
  (logxor (logior x (ones-comp-32 y))
          z))

(defun I (x y z)
  (declare (type (unsigned-byte 32) x y z))
  (logior (logand x z)
          (logand y (ones-comp-32 z))))

(defun J (x y z)
  (declare (type (unsigned-byte 32) x y z))
  (logxor x
          (logior y (ones-comp-32 z))))

;; the ten basic operations FF() through III() 

(defmacro FF (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (F ,b ,c ,d) ,x)))
    ; (format t "FF: ~x <= ~x + ~x~%" ,a (F ,b ,c ,d) ,x)
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))

(defmacro GG (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (G ,b ,c ,d) ,x #x5a827999)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro HH (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (H ,b ,c ,d) ,x #x6ed9eba1)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro II (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (I ,b ,c ,d) ,x #x8f1bbcdc)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro JJ (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (J ,b ,c ,d) ,x #xa953fd4e)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro FFF (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (F ,b ,c ,d) ,x)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))

(defmacro GGG (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (G ,b ,c ,d) ,x #x7a6d76e9)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro HHH (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (H ,b ,c ,d) ,x #x6d703ef3)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro III (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (I ,b ,c ,d) ,x #x5c4dd124)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
(defmacro JJJ (a b c d e x s)
  `(progn
    (setq ,a (chop-32 (+ ,a (J ,b ,c ,d) ,x #x50a28be6)))
    (setq ,a (chop-32 (+ (rotate-32 ,a ,s) ,e)))
    (setq ,c (rotate-32 ,c 10))))
  
;;  the compression function.
;;  transforms MDbuf using message bytes X[0] through X[15]

(defun compress (ctx)
  (let ((x (rmd160-ctx-input ctx))
        (mdbuf (rmd160-ctx-regs ctx)))
    (declare (type (array (unsigned-byte 32) *) mdbuf x))
    ;;  (dump-hash-regs mdbuf)
    ;;  (dump-compression-buf x)
    (let ((aa  (aref mdbuf 0))
          (bb  (aref mdbuf 1))
          (cc  (aref mdbuf 2))
          (dd  (aref mdbuf 3))
          (ee  (aref mdbuf 4))
          (aaa (aref mdbuf 0))
          (bbb (aref mdbuf 1))
          (ccc (aref mdbuf 2))
          (ddd (aref mdbuf 3))
          (eee (aref mdbuf 4)))
      (declare (type (unsigned-byte 32) aa bb cc dd ee aaa bbb ccc ddd eee))

      ;; round 1 
      (FF aa bb cc dd ee (aref X  0) 11)
      (FF ee aa bb cc dd (aref X  1) 14)
      (FF dd ee aa bb cc (aref X  2) 15)
      (FF cc dd ee aa bb (aref X  3) 12)
      (FF bb cc dd ee aa (aref X  4)  5)
      (FF aa bb cc dd ee (aref X  5)  8)
      (FF ee aa bb cc dd (aref X  6)  7)
      (FF dd ee aa bb cc (aref X  7)  9)
      (FF cc dd ee aa bb (aref X  8) 11)
      (FF bb cc dd ee aa (aref X  9) 13)
      (FF aa bb cc dd ee (aref X 10) 14)
      (FF ee aa bb cc dd (aref X 11) 15)
      (FF dd ee aa bb cc (aref X 12)  6)
      (FF cc dd ee aa bb (aref X 13)  7)
      (FF bb cc dd ee aa (aref X 14)  9)
      (FF aa bb cc dd ee (aref X 15)  8)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aa bb cc dd ee)
                             
      ;; round 2 
      (GG ee aa bb cc dd (aref X  7)  7)
      (GG dd ee aa bb cc (aref X  4)  6)
      (GG cc dd ee aa bb (aref X 13)  8)
      (GG bb cc dd ee aa (aref X  1) 13)
      (GG aa bb cc dd ee (aref X 10) 11)
      (GG ee aa bb cc dd (aref X  6)  9)
      (GG dd ee aa bb cc (aref X 15)  7)
      (GG cc dd ee aa bb (aref X  3) 15)
      (GG bb cc dd ee aa (aref X 12)  7)
      (GG aa bb cc dd ee (aref X  0) 12)
      (GG ee aa bb cc dd (aref X  9) 15)
      (GG dd ee aa bb cc (aref X  5)  9)
      (GG cc dd ee aa bb (aref X  2) 11)
      (GG bb cc dd ee aa (aref X 14)  7)
      (GG aa bb cc dd ee (aref X 11) 13)
      (GG ee aa bb cc dd (aref X  8) 12)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aa bb cc dd ee)

      ;; round 3 
      (HH dd ee aa bb cc (aref X  3) 11)
      (HH cc dd ee aa bb (aref X 10) 13)
      (HH bb cc dd ee aa (aref X 14)  6)
      (HH aa bb cc dd ee (aref X  4)  7)
      (HH ee aa bb cc dd (aref X  9) 14)
      (HH dd ee aa bb cc (aref X 15)  9)
      (HH cc dd ee aa bb (aref X  8) 13)
      (HH bb cc dd ee aa (aref X  1) 15)
      (HH aa bb cc dd ee (aref X  2) 14)
      (HH ee aa bb cc dd (aref X  7)  8)
      (HH dd ee aa bb cc (aref X  0) 13)
      (HH cc dd ee aa bb (aref X  6)  6)
      (HH bb cc dd ee aa (aref X 13)  5)
      (HH aa bb cc dd ee (aref X 11) 12)
      (HH ee aa bb cc dd (aref X  5)  7)
      (HH dd ee aa bb cc (aref X 12)  5)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aa bb cc dd ee)

      ;; round 4 
      (II cc dd ee aa bb (aref X  1) 11)
      (II bb cc dd ee aa (aref X  9) 12)
      (II aa bb cc dd ee (aref X 11) 14)
      (II ee aa bb cc dd (aref X 10) 15)
      (II dd ee aa bb cc (aref X  0) 14)
      (II cc dd ee aa bb (aref X  8) 15)
      (II bb cc dd ee aa (aref X 12)  9)
      (II aa bb cc dd ee (aref X  4)  8)
      (II ee aa bb cc dd (aref X 13)  9)
      (II dd ee aa bb cc (aref X  3) 14)
      (II cc dd ee aa bb (aref X  7)  5)
      (II bb cc dd ee aa (aref X 15)  6)
      (II aa bb cc dd ee (aref X 14)  8)
      (II ee aa bb cc dd (aref X  5)  6)
      (II dd ee aa bb cc (aref X  6)  5)
      (II cc dd ee aa bb (aref X  2) 12)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aa bb cc dd ee)

      ;; round 5 
      (JJ bb cc dd ee aa (aref X  4)  9)
      (JJ aa bb cc dd ee (aref X  0) 15)
      (JJ ee aa bb cc dd (aref X  5)  5)
      (JJ dd ee aa bb cc (aref X  9) 11)
      (JJ cc dd ee aa bb (aref X  7)  6)
      (JJ bb cc dd ee aa (aref X 12)  8)
      (JJ aa bb cc dd ee (aref X  2) 13)
      (JJ ee aa bb cc dd (aref X 10) 12)
      (JJ dd ee aa bb cc (aref X 14)  5)
      (JJ cc dd ee aa bb (aref X  1) 12)
      (JJ bb cc dd ee aa (aref X  3) 13)
      (JJ aa bb cc dd ee (aref X  8) 14)
      (JJ ee aa bb cc dd (aref X 11) 11)
      (JJ dd ee aa bb cc (aref X  6)  8)
      (JJ cc dd ee aa bb (aref X 15)  5)
      (JJ bb cc dd ee aa (aref X 13)  6)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aa bb cc dd ee)

      ;; parallel round 1 
      (JJJ aaa bbb ccc ddd eee (aref X  5)  8)
      (JJJ eee aaa bbb ccc ddd (aref X 14)  9)
      (JJJ ddd eee aaa bbb ccc (aref X  7)  9)
      (JJJ ccc ddd eee aaa bbb (aref X  0) 11)
      (JJJ bbb ccc ddd eee aaa (aref X  9) 13)
      (JJJ aaa bbb ccc ddd eee (aref X  2) 15)
      (JJJ eee aaa bbb ccc ddd (aref X 11) 15)
      (JJJ ddd eee aaa bbb ccc (aref X  4)  5)
      (JJJ ccc ddd eee aaa bbb (aref X 13)  7)
      (JJJ bbb ccc ddd eee aaa (aref X  6)  7)
      (JJJ aaa bbb ccc ddd eee (aref X 15)  8)
      (JJJ eee aaa bbb ccc ddd (aref X  8) 11)
      (JJJ ddd eee aaa bbb ccc (aref X  1) 14)
      (JJJ ccc ddd eee aaa bbb (aref X 10) 14)
      (JJJ bbb ccc ddd eee aaa (aref X  3) 12)
      (JJJ aaa bbb ccc ddd eee (aref X 12)  6)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aaa bbb ccc ddd eee)

      ;; parallel round 2 
      (III eee aaa bbb ccc ddd (aref X  6)  9)
      (III ddd eee aaa bbb ccc (aref X 11) 13)
      (III ccc ddd eee aaa bbb (aref X  3) 15)
      (III bbb ccc ddd eee aaa (aref X  7)  7)
      (III aaa bbb ccc ddd eee (aref X  0) 12)
      (III eee aaa bbb ccc ddd (aref X 13)  8)
      (III ddd eee aaa bbb ccc (aref X  5)  9)
      (III ccc ddd eee aaa bbb (aref X 10) 11)
      (III bbb ccc ddd eee aaa (aref X 14)  7)
      (III aaa bbb ccc ddd eee (aref X 15)  7)
      (III eee aaa bbb ccc ddd (aref X  8) 12)
      (III ddd eee aaa bbb ccc (aref X 12)  7)
      (III ccc ddd eee aaa bbb (aref X  4)  6)
      (III bbb ccc ddd eee aaa (aref X  9) 15)
      (III aaa bbb ccc ddd eee (aref X  1) 13)
      (III eee aaa bbb ccc ddd (aref X  2) 11)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aaa bbb ccc ddd eee)
    
      ;; parallel round 3 
      (HHH ddd eee aaa bbb ccc (aref X 15)  9)
      (HHH ccc ddd eee aaa bbb (aref X  5)  7)
      (HHH bbb ccc ddd eee aaa (aref X  1) 15)
      (HHH aaa bbb ccc ddd eee (aref X  3) 11)
      (HHH eee aaa bbb ccc ddd (aref X  7)  8)
      (HHH ddd eee aaa bbb ccc (aref X 14)  6)
      (HHH ccc ddd eee aaa bbb (aref X  6)  6)
      (HHH bbb ccc ddd eee aaa (aref X  9) 14)
      (HHH aaa bbb ccc ddd eee (aref X 11) 12)
      (HHH eee aaa bbb ccc ddd (aref X  8) 13)
      (HHH ddd eee aaa bbb ccc (aref X 12)  5)
      (HHH ccc ddd eee aaa bbb (aref X  2) 14)
      (HHH bbb ccc ddd eee aaa (aref X 10) 13)
      (HHH aaa bbb ccc ddd eee (aref X  0) 13)
      (HHH eee aaa bbb ccc ddd (aref X  4)  7)
      (HHH ddd eee aaa bbb ccc (aref X 13)  5)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aaa bbb ccc ddd eee)

      ;; parallel round 4    
      (GGG ccc ddd eee aaa bbb (aref X  8) 15)
      (GGG bbb ccc ddd eee aaa (aref X  6)  5)
      (GGG aaa bbb ccc ddd eee (aref X  4)  8)
      (GGG eee aaa bbb ccc ddd (aref X  1) 11)
      (GGG ddd eee aaa bbb ccc (aref X  3) 14)
      (GGG ccc ddd eee aaa bbb (aref X 11) 14)
      (GGG bbb ccc ddd eee aaa (aref X 15)  6)
      (GGG aaa bbb ccc ddd eee (aref X  0) 14)
      (GGG eee aaa bbb ccc ddd (aref X  5)  6)
      (GGG ddd eee aaa bbb ccc (aref X 12)  9)
      (GGG ccc ddd eee aaa bbb (aref X  2) 12)
      (GGG bbb ccc ddd eee aaa (aref X 13)  9)
      (GGG aaa bbb ccc ddd eee (aref X  9) 12)
      (GGG eee aaa bbb ccc ddd (aref X  7)  5)
      (GGG ddd eee aaa bbb ccc (aref X 10) 15)
      (GGG ccc ddd eee aaa bbb (aref X 14)  8)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aaa bbb ccc ddd eee)

      ;; parallel round 5 
      (FFF bbb ccc ddd eee aaa (aref X 12)  8)
      (FFF aaa bbb ccc ddd eee (aref X 15)  5)
      (FFF eee aaa bbb ccc ddd (aref X 10) 12)
      (FFF ddd eee aaa bbb ccc (aref X  4)  9)
      (FFF ccc ddd eee aaa bbb (aref X  1) 12)
      (FFF bbb ccc ddd eee aaa (aref X  5)  5)
      (FFF aaa bbb ccc ddd eee (aref X  8) 14)
      (FFF eee aaa bbb ccc ddd (aref X  7)  6)
      (FFF ddd eee aaa bbb ccc (aref X  6)  8)
      (FFF ccc ddd eee aaa bbb (aref X  2) 13)
      (FFF bbb ccc ddd eee aaa (aref X 13)  6)
      (FFF aaa bbb ccc ddd eee (aref X 14)  5)
      (FFF eee aaa bbb ccc ddd (aref X  0) 15)
      (FFF ddd eee aaa bbb ccc (aref X  3) 13)
      (FFF ccc ddd eee aaa bbb (aref X  9) 11)
      (FFF bbb ccc ddd eee aaa (aref X 11) 11)
      ;; (format t ";; ~x ~x ~x ~x ~x~%" aaa bbb ccc ddd eee)

      ;; combine results 
      (setf ddd
            (mod (+ ddd cc (aref MDbuf 1)) (ash 1 32)))
      (setf (aref MDbuf 1)
            (mod (+ (aref MDbuf 2) dd eee) (ash 1 32)))
      (setf (aref MDbuf 2)
            (mod (+ (aref MDbuf 3) ee aaa) (ash 1 32)))
      (setf (aref MDbuf 3)
            (mod (+ (aref MDbuf 4) aa bbb) (ash 1 32)))
      (setf (aref MDbuf 4)
            (mod (+ (aref MDbuf 0) bb ccc) (ash 1 32)))
      (setf (aref MDbuf 0)
            ddd)
      ;;(dump-hash-regs MDbuf)
      )))

(defun dump-compression-buf (buf)
  (dotimes (i 16)
    (format t "~& ~2d - ~32,'0b~%" i (aref buf i))))
          
(defun dump-hash-regs (hash)
  (format t "~&Hash: ~x ~x ~x ~x ~x~%"
          (aref hash 0)
          (aref hash 1)
          (aref hash 2)
          (aref hash 3)
          (aref hash 4)))
          

;; puts bytes from strptr into X and pad out; appends length 
;; and finally, compresses the last block(s)
;; note: length in bits == 8 * (lswlen + 2^32 mswlen).
;; note: there are (lswlen mod 64) bytes left in strptr.

(defun MDfinish (MDbuf strptr)
  (clear-ctx-input mdbuf)
  (multiple-value-bind (blocks msg-bytes)
      (floor (length strptr) 64)
    (dotimes (i msg-bytes)
      (multiple-value-bind (word byte)
          (floor i 4)
        (setf (ldb (byte 8 (* 8 byte)) (aref (rmd160-ctx-input mdbuf) word))
              (char-code (aref strptr (+ i (* 64 blocks)))))))
    
    ;; append the bit m_n == 1 
    (setf (ldb (byte 8 (* 8 (mod msg-bytes 4))) (aref (rmd160-ctx-input mdbuf) (floor msg-bytes 4)))
          (ash 1 7))
      
    (when (< 55 msg-bytes)
      (compress MDbuf)
      (clear-ctx-input mdbuf))
    
    ;; append length in bits
    (let ((len (length strptr)))
      (setf (aref (rmd160-ctx-input mdbuf) 14)
            (ash (ldb (byte 32 0) len) 3))
      (setf (aref (rmd160-ctx-input mdbuf) 15)
            (ldb (byte 32 32) len)))
    (compress mdbuf)))
  
(defun bytes-to-dword (string index)
  (+ (ash (char-code (aref string (+ index 3))) 24)
     (ash (char-code (aref string (+ index 2))) 16)
     (ash (char-code (aref string (+ index 1))) 8)
     (char-code (aref string index))))
     
;;
;; returns RMD(message)
;; message should be a string
;;
(defun rmd (message)
  (let ((mdbuf (make-rmd160-ctx))
        (index 0))
    
    ;; process message in 16-word chunks
    (when (< 63 (length message))
      (do ((nbytes (length message) (- nbytes 64)))
          ((<= nbytes 64))
        (dotimes (i 16)
          (setf (aref (rmd160-ctx-input mdbuf) i)
                (bytes-to-dword message index))
          (setq index (+ index 4)))
        (compress mdbuf)))

    ;; length mod 64 bytes left
    ;; finish:
    (MDfinish MDbuf message)

    (let ((hashcode 0))
      (dotimes (i 5)
        (setf (ldb (byte 32 (* 32 i)) hashcode)
              (aref (rmd160-ctx-regs mdbuf) i)))
      hashcode)))

(defun hashcode-to-string (hashcode)
  (let ((str (make-string-output-stream)))
    (dotimes (i 20)
      (format str "~2,'0x" (ldb (byte 8 (* 8 i)) hashcode)))
    (get-output-stream-string str)))

(defun new-hash (string)
  (hashcode-to-string (rmd string)))
