;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wco -*-
;;; $Id: event.lisp,v 1.12 2003/05/27 16:43:16 ayres Exp $
;;;
;;; Copyright (c) 2003 onShore Development, Inc.
;;;
;;; A simple, single-process, synchronous Publish/Subscribe mechanism
;;;

(in-package :odcl)

(defvar *odcl-event-listeners* nil)

;;----------------------------------------------------------------
;; Messages
;;
(eval-when (:load-toplevel :compile-toplevel)

;; Set of message types support by the event bus
(defvar *message-types* (make-hash-table)
  "Hash by name of message types.")

(defmacro def-message-type (name slotspec &rest options)
  (let ((mname (cond
                 ((keywordp name)
                  name)
                 (t (list 'quote name)))))
  `(progn
    (setf (gethash ,mname *message-types*)
     (list ,mname ',slotspec ',options)))))
  
(defun message-types (&aux message-types)
  (maphash #'(lambda (k v)
               (declare (ignore k))
               (push v message-types))
           *message-types*)
  message-types)
           

;; Basic messages to support synchronization
(def-message-type transaction-start
    ((:transaction :type transaction))
  (:documentation "Notify observer that transaction has started."))

(def-message-type transaction-abort
    ((:transaction :type transaction))
  (:documentation "Notify observer that transaction has aborted."))

(def-message-type transaction-commit
    ((:transaction :type transaction))
  (:documentation "Notify observer that transaction has commited."))
)

(defmethod validate-event (event-class event-args)
  (if-bind (eventspec (gethash event-class *message-types*))
      ;; check attribute presence and types
      (dolist (spec (second eventspec))
        (destructuring-bind (name &rest aspec)
            spec
          (unless (getf aspec :optional nil)
            (let* ((sym (gensym))
                   (val (getf event-args name sym)))
              (if (equal val sym)
                  (error "Attribute ~A is required in events of type ~A."
                         name event-class)
                  (unless (typep val (getf aspec :type t))
                    (error "Attribute ~A value ~A is not of type ~A."
                           name val (getf aspec :type t))))))))
      (error "Unknown event type ~A." event-class)))


;; event listener abstract class
(defclass event-listener ()
  ((name :initarg :name)))

(defmethod accept-event ((self event-listener) event)
  (declare (ignore event))
  nil)

(defmethod serve-event ((self event-listener) event)
  (declare (ignore event))
  (error "Must provide a serve-event method for this class."))

(defun event-listener-register (listener)
  ""
  (with-slots (name events)
    listener
    (event-listener-deregister name)
    (push listener *odcl-event-listeners*)))

(defmethod event-listener-deregister ((name symbol))
  ""
  (setf *odcl-event-listeners*
        (remove-if #'(lambda (listener)
                       (equal (slot-value listener 'name) name))
                   *odcl-event-listeners*)))

(defmethod event-listener-deregister ((listener event-listener))
  ""
  (setf *odcl-event-listeners*
        (remove listener *odcl-event-listeners*)))


;; API for publishing events onto the event bus
(defun publish-event (event-class &rest event-args)
  ;; perform event type checking here
  (validate-event event-class event-args)
  (let ((event (append (list
                        event-class
                        :timestamp (get-time))
                       event-args)))
    (dolist (listener *odcl-event-listeners*)
      (when (accept-event listener event)
        (serve-event listener event)))))

;; simple listener, accepts events named in its events slot
(defclass simple-listener (event-listener)
  ((events
    :initform nil
    :initarg :events)))

(defmethod accept-event ((self simple-listener) event)
  (member (car event) (slot-value self 'events)))
