;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; postgresql-sql.cl --- SQL-Interface implementation for PostgreSQL
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: postgresql-sql.lisp,v 1.29 2002/02/21 22:38:33 craig Exp $

(in-package :MAISQL-POSTGRESQL)

;;;; %File Description:
;;;; 
;;;; 
;;;; 

(eval-when (:compile-toplevel :load-toplevel)
  #+cmu
  (defmacro with-lock (lock &body body)
    `(mp:with-lock-held ,lock
      ,@body))
  #-cmu
  (defmacro with-lock (lock &body body)
    `(progn
      ,@body))  
  )                                     ;



(defmethod database-initialize-database-type
  ((database-type (eql :postgresql)))
  t)

(defclass postgresql-database (database)
  ((conn-ptr :accessor database-conn-ptr :initarg :conn-ptr
	     :type psql-conn-ptr)
   (lock
    :accessor database-lock
    #+(or cmu lispworks)
    :initform
    #+(or cmu lispworks) (mp:make-lock #-cmu :name "connection lock"))))

(defmethod database-name-from-spec
  (connection-spec (database-type (eql :postgresql)))
  (check-connection-spec connection-spec database-type
                         (host db user password &optional port options tty))
  (destructuring-bind (host db user password &optional port options tty)
      connection-spec
    (declare (ignore password options tty))
    (concatenate 'string host (if port ":") (if port port) "/" db "/" user)))

(defmethod database-busy ((database postgresql-database))
  (pqisbusy (database-conn-ptr database)))

(defmethod database-connect (connection-spec (database-type (eql :postgresql)))
  (check-connection-spec connection-spec database-type
			 (host db user password &optional port options tty))
  (let ((connection (%pg-database-connection connection-spec)))
    ;; Success, make instance
    (make-instance 'postgresql-database
                   :name (database-name-from-spec connection-spec
                                                  database-type)
                   :spec connection-spec
                   :db-type :postgresql
                   :conn-ptr connection)))

(defmethod database-reconnect ((database postgresql-database))
    (let ((foo (database-lock database)))
      (with-lock (foo)
	(with-slots (maisql-sys::spec conn-ptr)
	    database
	  (setf conn-ptr (%pg-database-connection maisql-sys::spec))
	  database))))

(defmethod database-disconnect ((database postgresql-database))
    (let ((foo (database-lock database)))
      (with-lock (foo)
	(pqfinish (database-conn-ptr database))
	(setf (database-conn-ptr database) nil)
	t)))

(defun %trim-crlf (string)
  (string-right-trim '(#\Return #\Newline) string))

(defmethod database-query (query-expression (database postgresql-database))
  (flet ((error-text ()
           (%trim-crlf (pqerrormessage (database-conn-ptr database)))))
    (with-lock ((database-lock database))
      (let ((result (%pg-query database query-expression)))
        (unwind-protect
             (case (pqresultstatus result)
               (:empty-query
                nil)
               (:tuples-ok
                (loop for tuple-index from 0 below (pqntuples result)
                      collect
                      (loop for i from 0 below (pqnfields result)
                            collect
                            (if (pqgetisnull result tuple-index i)
                                nil
                                (pqgetvalue result tuple-index i)))))
               (:fatal-error
                (let ((msg-text (subseq (error-text) 8)))
                  (error 'maisql-sql-error
                         :database database
                         :expression query-expression
                         :errno (pqresultstatus result)
                         :error msg-text)))
               (t
                (error 'maisql-sql-error
                       :database database
                       :expression query-expression
                       :errno (pqresultstatus result)
                       :error (concatenate 'string
                                           (error-text)
                                           (pqcmdstatus result)))))
          (pqclear result))))))

(defmethod database-create-sequence
  (sequence-name (database postgresql-database))
  (database-execute-command
   (concatenate 'string "CREATE SEQUENCE " (sql-escape sequence-name)) database))

(defmethod database-drop-sequence
  (sequence-name (database postgresql-database))
  (database-execute-command
   (concatenate 'string "DROP SEQUENCE " (sql-escape sequence-name)) database))

(defmethod database-sequence-next (sequence-name (database postgresql-database))
  (parse-integer
   (caar
    (database-query
     (concatenate 'string "SELECT NEXTVAL ('" (sql-escape sequence-name) "')")
     database))))

(defmethod database-execute-command
    (sql-expression (database postgresql-database))
    (with-lock ((database-lock database))
      (let ((conn-ptr (database-conn-ptr database))
	    (result (%pg-exec database sql-expression)))
      (unwind-protect
           (case (pqresultstatus result)
             (:command-ok
              t)
             (:empty-query t)
             ;; (warn ";; Warning, database-execute-command:\
             ;; Empty statement sent to server."))
             (:tuples-ok t)
             ;; (warn ";; Warning, database-execute-command:\
             ;; Recieved tuples ok in response to pqexec."))
             (t
              (error 'maisql-sql-error
                     :database database
                     :expression sql-expression
                     :errno (pqresultstatus result)
                     :error (concatenate 'string
                                         (%trim-crlf (pqerrormessage conn-ptr))
                                         (pqcmdstatus result)))))
        (pqclear result)))))

(defstruct postgresql-result-set
  #+cmu (res-ptr (sap-alien (int-sap 0) psql-result)
                 :type (alien psql-result))
  #+lispworks (res-ptr NIL)
  (num-tuples 0)
  (num-fields 0)
  (tuple-index 0))

(defmethod database-query-result-set
  (query-expression (database postgresql-database) &optional full-set)
  (let ((result (%pg-query-result-set database query-expression)))
    (case (pqresultstatus result)
      ((:empty-query :tuples-ok)
       (if full-set
           (values (make-postgresql-result-set
                    :res-ptr result
                    :num-fields (pqnfields result)
                    :num-tuples (pqntuples result))
                   (pqnfields result)
                   (pqntuples result))
           (values (make-postgresql-result-set
                    :res-ptr result
                    :num-fields (pqnfields result)
                    :num-tuples (pqntuples result))
                   (pqnfields result))))
      (t
       (unwind-protect
            (error 'maisql-sql-error
                   :database database
                   :expression query-expression
                   :errno (pqresultstatus result)
                   :error (pqcmdstatus result))
         (pqclear result))))))

(defmethod database-dump-result-set (result-set (database postgresql-database))
    (let ((res-ptr (postgresql-result-set-res-ptr result-set)))
      (declare (type psql-result-ptr res-ptr))
      (pqclear res-ptr)
      t))

(defmethod database-store-next-row
  (result-set (database postgresql-database) list)
  (let ((result (postgresql-result-set-res-ptr result-set)))
    (declare (type psql-result-ptr result))
    (if (>= (postgresql-result-set-tuple-index result-set)
	    (postgresql-result-set-num-tuples result-set))
	nil
	(loop with tuple-index = (postgresql-result-set-tuple-index result-set)
	      for i from 0 below (postgresql-result-set-num-fields result-set)
	      for rest on list
	      do
	      (setf (car rest)
		    (if (pqgetisnull result tuple-index i)
			nil
			(pqgetvalue result tuple-index i)))
	      finally
	      (incf (postgresql-result-set-tuple-index result-set))
	      (return list)))))


(defmethod database-list-tables ((database postgresql-database)
				 &key (system-tables nil))
  (let ((res (select [tablename] :from [pg_tables] :flatp t)))
    (if (not system-tables)
	(remove-if #'(lambda (table)
		       (equal (subseq table 0 3)
			      "pg_")) res)
        res)))

(defmethod database-list-attributes (table (database postgresql-database))
  (let* ((relname (etypecase table
		    (sql-sys::sql-ident
		     (string-downcase
		      (symbol-name (slot-value table 'sql-sys::name))))
		    (string table)))
	 (result (select [pg_attribute attname]
			 :from '([pg_class] [pg_attribute])
			 :where [and [= [pg_class oid] [pg_attribute attrelid]]
				     [= [pg_class relname] relname]] :flatp t)))
    (if result
	(reverse
         (remove-if #'(lambda (it) (member it '("cmin"
                                                "cmax"
                                                "xmax"
                                                "xmin"
					       "oid"
                                                "ctid") :test #'equal)) result)))))

(defmethod database-attribute-type (attribute table
                                              (database postgresql-database))
  (let ((result
	 (select [pg_type typname] 
		 :from '([pg_type] [pg_class] [pg_attribute])
		 :where [and [= [pg_class oid] [pg_attribute attrelid]]
			     [= [pg_class relname] table]
			     [= [pg_attribute attname] attribute]
			     [= [pg_attribute atttypid] [pg_type oid]]]
                 :flatp t)))
    (if result
	(intern (string-upcase (car result)) :keyword) nil)))


;;(defmethod database-add-attribute (table attribute database)
;;  (let ((attname (slot-value attribute 'name)


(defmethod database-output-sql ((expr sql-sys::sql-typecast-exp) (database postgresql-database))
  (with-slots (sql-sys::modifier sql-sys::components)
    expr
    (if sql-sys::modifier
        (progn
          (sql-sys::output-sql sql-sys::components database)
          (write-char #\: sql-sys::*sql-stream*)
          (write-char #\: sql-sys::*sql-stream*)
          (write-string (symbol-name sql-sys::modifier) sql-sys::*sql-stream*)))))

(defmethod database-output-sql-as-type ((type (eql 'integer)) val (database postgresql-database))
  ;; typecast it so it uses the indexes
  (when val
    (make-instance 'sql-sys::sql-typecast-exp
                   :modifier 'int8
                   :components val)))

