/*
	$Id: cl_endian.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="I/O Data"	

#ifndef header_endian
#define header_endian

class CL_Endian
//: Endianess management class.
// The CL_Endian class is used to convert between different endianess. There
// are also a set of defines to the lazy developer:
// <code>
// #define SWAP_IF_BIG(i)
// #define SWAP_IF_LITTLE(i)
// #define SWAP_IF_BIG_ALOT(i, times)
// #define SWAP_IF_LITTLE_ALOT(i, times)
// #define IS_SYSTEM_64BIT()
// </code>
// Each of these defines call CL_Endian::is_system_big() and CL_Endian::swap.
// For the exact definition of the defines, have a look in
// ClanLib/Layer1/IOData/API/endian.h
{
public:
	static void swap(void *data, int type_size);
	//: Swaps the bytes between little and big endian.
	//!param: data - Data to be swapped.
	//!param: type_size - Size of datatype to be swapped.
	
	static void swap(void *data, int type_size, int total_times);
	//: Swaps larger amounts of data between little and big endian.
	//!param: data -Data to be swapped.
	//!param: type_size - Size of datatype to be swapped.
	//!param: total_times - Number of 'type_size' size data chunks to be swapped.

	static bool is_system_big();
	//: Returns true if big endian system.
	//!retval: True if big endian system, false otherwise.

	static bool is_system_64bit();
	//: Returns true if 64 bit system.
	//!retval: True if 64 bit, false otherwise.
};

// Defines to the lazy developer:
#define SWAP_IF_BIG(i) if (CL_Endian::is_system_big()) CL_Endian::swap(&i, sizeof(i));
#define SWAP_IF_LITTLE(i) if (!CL_Endian::is_system_big()) CL_Endian::swap(&i, sizeof(i));

#define SWAP_IF_BIG_ALOT(i, times) if (CL_Endian::is_system_big()) CL_Endian::swap(&i, sizeof(i), times);
#define SWAP_IF_LITTLE_ALOT(i, times) if (!CL_Endian::is_system_big()) CL_Endian::swap(&i, sizeof(i), times);

#define IS_SYSTEM_64BIT() CL_Endian::is_system_64bit();

#endif
