/*
	$Id: restype_surface.h,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#ifndef header_restype_surface
#define header_restype_surface

#include "API/Core/Display/surface.h"
#include "API/Core/Resources/resource_manager.h"
#include "API/Core/SurfaceProviders/generic_surfaceprovider.h"
#include "API/Core/Display/res_surface.h"

class CL_Res_Surface : public CL_ResourceType
//: The surface resource type.
// This class is used to load surface resources from a resource source
// (datafile or script file).
// <br>
// Use the CL_ResourceManager to get an instance to the resource source, and
// then call one of the static functions here to load the resource:
// <code>
// CL_ResourceManager res_manager("my_datafile.dat", true);
// CL_Surface *my_surface = CL_Res_Surface::load("my_surface", res_manager);
// </code>
//!res_type: surface
//!res_option: tga - Interpret this file as a targa image.
//!res_option: pcx - Interpret this file as a pcx image.
//!res_option: x=<val> - Specify the starting x coordinate in the image file.
//!res_option: y=<val> - Specify the starting y coordinate in the image file.
//!res_option: width=<val> - Specify the number of horizontal pixels to store
//!res_option: from the image.
//!res_option: height=<val> - Specify the number of vertical pixels to store from the image.
//!res_option: array=<xval>x<yval> - Creates a surface containing <xval> * <yval>
//!res_option: sprites. This option requires the existance of x, y, width, height
//!res_option: option values in the resource definition; otherwise the option is
//!res_option: ignored. The sprites are read from left to right, top to bottom,
//!res_option: and are each saved with width, height equal to the width,
//!res_option: height options specified.
//!res_option: tcol=<val> - Specifies a given color as transparency color for this
//!res_option: surface. This option is only meaningful for palettisized input
//!res_option: image formats (PCX).
//!res_option: tcol=(<val>, <val2>, ...) - Same as above, but specifies several transpareny colors.
{
public:
	CL_Res_Surface();

	static std::list<CL_ResourceSource_Surface*> surface_sources;

private:
friend CL_ResourceManager;
	virtual CL_Resource *create_from_location(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent);

	virtual CL_Resource *create_from_serialization(
		std::string name,
		CL_ResourceManager *parent);
};

class CL_WritableSurfaceResource : public CL_Resource
{
protected:
	CL_Surface *loaded_surface;
	bool loaded;
	std::string surface_location;
	CL_ResourceOptions *options;
	CL_ResourceManager *parent;
	int load_count;
public:
	CL_WritableSurfaceResource(std::string name);
	virtual ~CL_WritableSurfaceResource() { ; }

	virtual void serialize_save(CL_OutputSource *output);
	virtual void load();
	virtual void unload();
	virtual int get_load_count() { return load_count; }

	virtual CL_SurfaceProvider *create_provider()=0;
	virtual CL_Surface *create_surface();
};

class CL_SurfaceFileResource : public CL_WritableSurfaceResource
{
public:
	CL_SurfaceFileResource(
		std::string name,
		std::string location,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent);
	virtual ~CL_SurfaceFileResource();

	virtual CL_SurfaceProvider *create_provider();
};

class CL_SurfaceDatafileResource : public CL_WritableSurfaceResource
{
public:
	CL_SurfaceDatafileResource(
		std::string name,
		CL_ResourceManager *parent);
	virtual ~CL_SurfaceDatafileResource();

	virtual CL_SurfaceProvider *create_provider();
};

class CL_MaskTranscolProvider : public CL_SurfaceProvider_Generic
{
	CL_SurfaceProvider *parent;
	bool delete_provider;
	unsigned int *mask_indices;
	int num_mask_indices;
public:
	CL_MaskTranscolProvider(
			CL_SurfaceProvider *parent, 
			bool delete_provider, 
			const unsigned int *mask_indices,
			int num_mask_indices);
	virtual ~CL_MaskTranscolProvider();

	virtual unsigned int get_pitch() const;
	virtual int get_translate_x() const;
	virtual int get_translate_y() const;
	virtual unsigned int get_width() const;
	virtual unsigned int get_height() const;
	virtual unsigned int get_num_frames() const;
	virtual unsigned int get_red_mask() const;
	virtual unsigned int get_green_mask() const;
	virtual unsigned int get_blue_mask() const;
	virtual unsigned int get_alpha_mask() const;
	virtual unsigned int get_depth() const;
	virtual bool is_indexed() const;

	virtual CL_Palette *get_palette() const;
	virtual unsigned int get_src_colorkey() const;
	virtual bool uses_src_colorkey() const;
	
	virtual void *get_data() const;

	virtual void perform_lock();
	virtual void perform_unlock();
};

#endif
