/*
	$Id: datafile_inputprovider.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Datafile input source provider, part of the i/o data component.

*/

#include "Core/precomp.h"

#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <string.h>

#ifndef WIN32
#include <unistd.h>
#else
#include <io.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "datafile_inputprovider.h"
#include "API/Core/System/error.h"


CL_InputSourceProvider *CL_InputSourceProvider::create_datafile_provider(const char *filename)
{
	return new CL_InputSourceProvider_Datafile(filename);
}


CL_InputSourceProvider_Datafile::CL_InputSourceProvider_Datafile(const char *_datafile)
{
	datafile = _datafile;
}

CL_InputSource *CL_InputSourceProvider_Datafile::open_source(const char *filename)
{
	return new CL_InputSource_Datafile(filename, datafile.c_str());
}

CL_InputSourceProvider *CL_InputSourceProvider_Datafile::clone()
{
	return new CL_InputSourceProvider_Datafile(datafile.c_str());
}

/**************************
	CL_InputSource_Datafile
**************************/

#ifdef WIN32
	#define OPENFLAGS O_RDONLY|O_BINARY
#else
	#define OPENFLAGS O_RDONLY
#endif

char datafile_id[]="ClanSoft datafile version 3.0";

CL_InputSource_Datafile::CL_InputSource_Datafile(
	const char *_filename, const char *_datafile)
{
	filename = _filename;
	datafile = _datafile;

	datafile_open = 0;
	index_open = 0;

	open();
}

CL_InputSource_Datafile::CL_InputSource_Datafile(const CL_InputSource_Datafile *source)
{
	filename = source->filename;
	datafile = source->datafile;

	datafile_open = 0;
	index_open = 0;

	open();
}

CL_InputSource_Datafile::~CL_InputSource_Datafile()
{
	close();
}

void CL_InputSource_Datafile::set_system_mode()
{
	// BUG BUG: Not implemented!!!
}

void CL_InputSource_Datafile::set_big_endian_mode()
{
	// BUG BUG: Not implemented!!!
}

void CL_InputSource_Datafile::set_little_endian_mode()
{
	// BUG BUG: Not implemented!!!
}

int CL_InputSource_Datafile::read_int32()
{
	int svar;

	if (gzread(gzfile, &svar, sizeof(int))!=sizeof(int))
	{
		throw CL_Error("CL_InputSource_Datafile::read_int32() failed");
	}

	seek_pos+=sizeof(int);
	return svar;
}

unsigned int CL_InputSource_Datafile::read_uint32()
{
	unsigned int svar;

	if (gzread(gzfile, &svar, sizeof(unsigned int))!=sizeof(unsigned int))
	{
		throw CL_Error("CL_InputSource_Datafile::read_uint32() failed");
	}

	seek_pos+=sizeof(unsigned int);
	return svar;
}

short CL_InputSource_Datafile::read_short16()
{
	short svar;

	if (gzread(gzfile, &svar, sizeof(short))!=sizeof(short))
	{
		throw CL_Error("CL_InputSource_Datafile::read_short16() failed");
	}

	seek_pos+=sizeof(short);
	return svar;
}

unsigned short CL_InputSource_Datafile::read_ushort16()
{
	unsigned short svar;

	if (gzread(gzfile, &svar, sizeof(unsigned short))!=sizeof(unsigned short))
	{
		throw CL_Error("CL_InputSource_Datafile::read_ushort16() failed");
	}

	seek_pos+=sizeof(unsigned short);
	return svar;
}

char CL_InputSource_Datafile::read_char8()
{
	char svar;

	if (gzread(gzfile, &svar, sizeof(char))!=sizeof(char))
	{
		throw CL_Error("CL_InputSource_Datafile::read_char8() failed");
	}

	seek_pos += sizeof(char);
	return svar;
}

unsigned char CL_InputSource_Datafile::read_uchar8()
{
	unsigned char svar;

	if (gzread(gzfile, &svar, sizeof(unsigned char))!=sizeof(unsigned char))
	{
		throw CL_Error("CL_InputSource_Datafile::read_uchar8() failed");
	}

	seek_pos += sizeof(unsigned char);
	return svar;
}

float CL_InputSource_Datafile::read_float32()
{
	float svar;

	if (gzread(gzfile, &svar, sizeof(float))!=sizeof(float))
	{
		throw CL_Error("CL_InputSource_Datafile::read_float32() failed");
	}
		
	seek_pos += sizeof(float);
	return svar;
}

int CL_InputSource_Datafile::read(void *addr, int size)
{
	int svar = gzread(gzfile, addr, size);
	seek_pos += svar;
	return svar;
}

void CL_InputSource_Datafile::open()
{
	if (datafile_open) return;
	datafile_handle = ::open(datafile.c_str(), OPENFLAGS);

	if (datafile_handle == -1)
	{
		std::string err("Could not open datafile ");
		err += std::string(datafile);
		throw CL_Error(err.c_str());
	}

	int id_len = strlen(datafile_id);

	char *temp = new char[id_len+1];

	::read(datafile_handle, temp, id_len);

	temp[id_len] = 0;

	if (strcmp(temp, datafile_id) != 0)
	{
		::close(datafile_handle);
		datafile_handle = -1;

		throw CL_Error("Invalid datafile format");
//		cout << "Invalid datafile format: " << temp << " (expected: " << datafile_id << ")" << endl;
//		cl_assert(1==0);
	}

	delete temp;

	datafile_open = 1;
	open_index();
//	cout << "open complete" << endl;
}

void CL_InputSource_Datafile::close()
{
	if (datafile_open == 0) return;

	close_index();
	::close(datafile_handle);

	datafile_open = 0;
}

void CL_InputSource_Datafile::open_index()
{
	if (index_open != 0) close_index();

	lseek(datafile_handle, strlen(datafile_id), SEEK_SET);	// Skip file ID
	int index_pos;
	::read(datafile_handle, &index_pos, sizeof(int));
	lseek(datafile_handle, index_pos, SEEK_SET);

//	cout << "Searching for " << filename << endl;

	int num_indexes = 0;
	::read(datafile_handle, &num_indexes, sizeof(int));
	for (int i=0; i<num_indexes; i++)
	{
		short length;
		::read(datafile_handle, &length,sizeof(short));

		char *objname = new char[length];
		int objpos;

		::read(datafile_handle, objname, length);
		::read(datafile_handle, &objpos, sizeof(int));
		::read(datafile_handle, &objsize, sizeof(int));

//		cout << "Found index: " << objname << endl;

		if (strcmp(objname, filename.c_str()) == 0)
		{
			lseek(datafile_handle, objpos, SEEK_SET);

			gzfile = gzdopen(dup(datafile_handle), "rb");
			index_open = 1;
			seek_pos = 0;

			delete[] objname;
			return;
		}

		delete[] objname;
	}

	index_open = 0;

	std::string err("ClanLib: Couldn't find datafile index: ");
	err += std::string(filename);
	throw CL_Error(err);
}

void CL_InputSource_Datafile::close_index()
{
	if (index_open==0) return;

	gzclose(gzfile);
	index_open=0;
}

CL_InputSource *CL_InputSource_Datafile::clone() const
{
	return new CL_InputSource_Datafile(this);
}

int CL_InputSource_Datafile::tell() const
{
	return seek_pos;
}

void CL_InputSource_Datafile::seek(int pos, SeekEnum seek_type)
{
	if (seek_type == seek_cur)
	{
		if (pos > 0)
		{
			char *temp = new char[pos];
			read(temp, pos);
			delete temp;
		}
		else if (pos < 0)
		{
			seek(tell()+pos, seek_set);
		}
		return;
	}
	else if (seek_type == seek_set)
	{
		int cur_pos = tell();
		if (pos >= cur_pos)
		{
			seek(pos - cur_pos, seek_cur);
			return;
		}
	}

	std::cout << "ClanLib: seek() in datafiles only supports forward seeks" << std::endl;
	cl_assert(false);
}

int CL_InputSource_Datafile::size() const
{
	return objsize;
}

std::string CL_InputSource_Datafile::read_string()
{
	int size = read_int32();

	char *str = new char[size];
	read(str, size);
	
	std::string ret = str;
	delete[] str;

	return ret;
}

void CL_InputSource_Datafile::push_position()
{
	CL_Zipped_Position indexpos;

	indexpos.gzfile = gzfile;
	indexpos.datafile_pos = lseek(datafile_handle, 0, SEEK_CUR);
	indexpos.seek_pos = seek_pos;

	index_stack.push(indexpos);

	index_open=0;
}

void CL_InputSource_Datafile::pop_position()
{
	if (index_open != 0) close_index();

	index_open = 1;

	CL_Zipped_Position pushed_index = index_stack.top();
	index_stack.pop();
	
	gzfile = pushed_index.gzfile;
	lseek(datafile_handle, pushed_index.datafile_pos, SEEK_SET);
	seek_pos = pushed_index.seek_pos;
}
