/*
	$Id: resource_manager_file.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"
#include "resource_manager_file.h"
#include "resource_tokenizer.h"
#include <API/Core/System/clanstring.h>
#include <API/Core/System/error.h>
#include <API/Core/System/cl_assert.h>

CL_ResourceManager* CL_ResourceManager::create(
	const char *config_file,
	CL_InputSourceProvider *provider,
	bool read_directly_from_source,
	bool delete_inputsource_provider)
{
	return new CL_ResourceManager_File(
		config_file,
		provider,
		read_directly_from_source,
		delete_inputsource_provider);
}

CL_ResourceManager *CL_ResourceManager::create(
	const char *file_name,
	const bool is_datafile)
{
	return new CL_ResourceManager_File(
		file_name,
		is_datafile);
}

CL_ResourceManager_File::CL_ResourceManager_File(
	const char *config_file,
	CL_InputSourceProvider *provider,
	bool read_directly_from_source,
	bool delete_inputsource_provider)
{
	filename = config_file;
	resource_provider = provider;
	delete_resource_provider = delete_inputsource_provider;
	from_source = read_directly_from_source;
	
	if (provider == NULL)
	{
		if (read_directly_from_source)
			resource_provider = CL_InputSourceProvider::create_file_provider("");
		else
			resource_provider = CL_InputSourceProvider::create_datafile_provider(filename.c_str());
		
		delete_resource_provider = true;
	}
	
	parse();
}

CL_ResourceManager_File::CL_ResourceManager_File(
	const char *file_name,
	const bool is_datafile)
{
	filename = file_name;
	delete_resource_provider = true;
	if (is_datafile)
	{
		resource_provider = CL_InputSourceProvider::create_datafile_provider(filename.c_str());
		from_source = false;
	}
	else
	{
		resource_provider = CL_InputSourceProvider::create_file_provider("");
		from_source = true;
	}
	
	parse();
}

CL_ResourceManager_File::~CL_ResourceManager_File()
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		if ((*it)->get_load_count() != 0)
		{
			CL_String warning;
			warning << "Warning, resource load balance is not zero: ";
			warning << (*it)->get_name() << ", " << (*it)->get_load_count();

			cl_info(0, warning.get_string());
		}
		delete *it;
	}
}

void CL_ResourceManager_File::load_all_resources()
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		(*it)->load();
	}
}

void CL_ResourceManager_File::unload_all_resources()
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		while ((*it)->get_load_count() > 0)
			(*it)->unload();
	}
}

void CL_ResourceManager_File::load_section(const char *section_name)
{
	CL_String prefix = section_name;
	prefix << "/";
	
	int prefix_len = prefix.get_length();

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		CL_String name = (*it)->get_name();
		if (name.mid(0, prefix_len) == prefix)
		{
			(*it)->load();
		}
	}
}

void CL_ResourceManager_File::unload_section(const char *section_name)
{
	CL_String prefix = section_name;
	prefix << "/";
	
	int prefix_len = prefix.get_length();

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		CL_String name = (*it)->get_name();
		if (name.mid(0, prefix_len) == prefix)
		{
			while ((*it)->get_load_count() > 0)
				(*it)->unload();
		}
	}
}

CL_Resource *CL_ResourceManager_File::get_resource(std::string res_id)
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		if ((*it)->get_name() == res_id) return *it;
	}
	
	CL_String err;
	err << "Resource " << res_id << " not found.";
	throw CL_Error(err.get_string());
	
	return NULL;
}

std::list<std::string> *CL_ResourceManager_File::get_all_resources()
{
	std::list<std::string> *retval = new std::list<std::string>;

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		retval->push_back((*it)->get_name());
	}
	
	return retval;
}

std::list<std::string> *CL_ResourceManager_File::get_resources_of_type(std::string type_id)
{
	std::list<std::string> *retval = new std::list<std::string>;

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		if ((*it)->get_type() == type_id)
			retval->push_back((*it)->get_name());
	}

	return retval;
}

CL_InputSourceProvider *CL_ResourceManager_File::get_resource_provider()
{
	return resource_provider;
}

void CL_ResourceManager_File::parse()
{
	CL_InputSource *input;

	if (from_source) input = resource_provider->open_source(filename.c_str());
	else input = resource_provider->open_source(default_scriptfile_id);
	
	CL_ResourceTokenizer lexer(filename, input);

	while (true)
	{
		std::string token = lexer.get_next_token();
		if (token == "") break;
		
		if (token == "location") // obsolete. Will read it but issues a warning.
		{
			token = lexer.get_next_token();
			if (token != "=")
				throw CL_Error(lexer.write_error("Missing '=' following global 'location'-declaration"));
			
			token = lexer.get_next_token();
			if (token != "datafile")
				throw CL_Error(lexer.write_error("Syntax error following global 'location'-declaration"));
			
			token = lexer.get_next_token();
			if (token != "(")
				throw CL_Error(lexer.write_error("Missing '(' following 'datafile' keyword"));

			token = lexer.get_next_token();

			token = lexer.get_next_token();
			if (token != ")")
				throw CL_Error(lexer.write_error("Missing ')' following datafile filename"));

			token = lexer.get_next_token();
			if (token != ";")
				throw CL_Error(lexer.write_error("Missing ';' following datafile filename"));

			cl_info(0, "The location keyword is obsolete in resource files. Please don't use it anymore.");
		}
		else // assume it is a section body (resource or section)
		{
			parse_section_body(token, lexer, "");
		}
	}
}

void CL_ResourceManager_File::parse_section_body(
	std::string token,
	CL_ResourceTokenizer &lexer,
	std::string prefix)
{
	if (token == "section")
	{
		CL_String section_prefix = prefix;
		section_prefix << lexer.get_next_token();
		section_prefix << "/";

		token = lexer.get_next_token();
		if (token != "{")
			throw CL_Error(lexer.write_error("Missing '{'"));
			
		while (true)
		{
			token = lexer.get_next_token();
			if (token == "}") break;
			parse_section_body(token, lexer, section_prefix);
		}
	}
	else // must be a resource then
	{
		CL_String name = prefix;
		name << token;

		token = lexer.get_next_token();
		if (token != "=")
		{
			CL_String err;
			err << "Missing '=' following declaration of resource '" << name << "'";
			throw CL_Error(lexer.write_error(err));
		}

		// read the location (if available):
		std::string location;
		token = lexer.get_next_token();
		if(token == "(")
			location = "";
		else {
			location = token;
			token = lexer.get_next_token();
		}

		// read resource options:
		if (token != "(")
		{
			CL_String err;
			err << "Missing '(' following declaration of resource '" << name << "'";
			throw CL_Error(lexer.write_error(err));
		}

		CL_ResourceOptions *options = new CL_ResourceOptions(this);
		while (true)
		{
			token = lexer.get_next_token();
			if (token == ")") break;
			
			std::string option_name = token;
			
			token = lexer.get_next_token();
			if (token != "=")
			{
				CL_String err;
				err << "Syntax error in options following declaration of resource '" << name << "'";
				throw CL_Error(lexer.write_error(err));
			}
			
			token = lexer.get_next_token();
			if (token == "(") // it is a value list
			{
				std::list<std::string> *value_list = new std::list<std::string>;

				while (true)
				{
					token = lexer.get_next_token();
					if (token == ")") break;

					value_list->push_back(token);
					
					token = lexer.get_next_token();
					if (token == ")") break;
					if (token != ",")
					{
						delete value_list;

						CL_String err;
						err << "Missing ',' or ')' following declaration of multiple option values in option '" <<
							   option_name << "' in resource '" << name << "' (found '" << token << "')";
						throw CL_Error(lexer.write_error(err));
					}
				}
				
				options->add(new CL_ResourceOption(option_name, value_list));
			}
			else
			{
				std::string option_value = token;
				
				options->add(new CL_ResourceOption(option_name, option_value));
			}

			token = lexer.get_next_token();
			if (token == ")") break;
			if (token != ",")
			{
				CL_String err;
				err << "Syntax error in options following declaration of resource '" << name << "'";
				throw CL_Error(lexer.write_error(err));
			}
		}
		
		token = lexer.get_next_token();
		if (token != ";")
		{
			CL_String err;
			err << "Missing ';' following declaration of resource '" << name << "'";
			throw CL_Error(lexer.write_error(err));
		}
		
		if (options->exists("type") == false)
		{
			CL_String err;
			err << "No type declared for resource '" << name << "'";
			throw CL_Error(lexer.write_error(err));
		}
		
		CL_Resource *res = create_resource(name, location, *options);
		if (res != NULL) resources.push_back(res);
		else
		{
			CL_String err;
			err << "Unknown type '" << options->get_value("type") << "' declared for resource '" << name << "'";
			throw CL_Error(lexer.write_error(err));
		}
	}
}

CL_Resource *CL_ResourceManager_File::create_resource(
	std::string name,
	std::string location,
	CL_ResourceOptions &options)
{
	std::string type = options.get_value("type");

	for (
		std::list<CL_ResourceType*>::iterator it = CL_ResourceType::resource_types.begin();
		it != CL_ResourceType::resource_types.end();
		it++)
	{
		if ((*it)->get_type() == type)
		{
			if (from_source)
			{
				return (*it)->create_from_location(
					name,
					location,
					&options,
					this);
			}
			else
			{
				return (*it)->create_from_serialization(
					name,
					this);
			}
		}
	}

	return NULL;
}
