/* tolua: type & tag manipulation.
** Support code for Lua bindings.
** Written by Waldemar Celes
** TeCGraf/PUC-Rio
** Jul 1998
** $Id: tolua_tt.c,v 1.1.1.1 2000/04/09 12:18:02 mbn Exp $
*/

/* This code is free software; you can redistribute it and/or modify it. 
** The software provided hereunder is on an "as is" basis, and 
** the author has no obligation to provide maintenance, support, updates,
** enhancements, or modifications. 
*/

#include "tolua.h"
#include "tolua_tt.h"
#include "tolua_st.h"
#include "tolua_tm.h"
#include "tolua_eh.h"

#include <stdio.h>
#include <string.h>

/* exported basic type tags */
int tolua_tag_nil;
int tolua_tag_number;
int tolua_tag_string;
int tolua_tag_userdata;
int tolua_tag_table;
int tolua_tag_cfunction;
int tolua_tag_function;


/* Global tables created in Lua:
   .tolua_itype: indexed by instance tags, stores the instance types.
   .tolua_itag: indexed by instance types, stores the instance tags.
   .tolua_const: indexed by constant tags, stores the tags.
   .tolua_hierarchy: indexed by instance tags, stores the base tags.
*/
static int tbl_itype;	  /* stores reference to ".tolua_itype" table */
static int tbl_itag;	  /* stores reference to ".tolua_itag" table */
static int tbl_const;	  /* stores reference to ".tolua_const" table */
static int tbl_hierarchy; /* stores reference to ".tolua_hierarchy" table */

static char* gettype (int tag)
{
 char* type;
 lua_pushobject(lua_getref(tbl_itype));
 lua_pushnumber(tag);
 type = lua_getstring(lua_gettable());
 if (type==NULL) type = "[undefined]";
 return type;
}

char* toluaI_tt_getobjtype (lua_Object lo)
{
 if (lo == LUA_NOOBJECT)
  return "[no object]";
 else
  return gettype(lua_tag(lo));
}

int toluaI_tt_gettag (char* type)
{
 lua_pushobject(lua_getref(tbl_itag));
 lua_pushstring(type);
 return (int)lua_getnumber(lua_gettable());
}

static int basetag (int tag)
{
 lua_pushobject(lua_getref(tbl_hierarchy));
 lua_pushnumber(tag);
 return (int)lua_getnumber(lua_gettable());
}

static int istype (lua_Object lo, int tag)
{
 int otag = lua_tag(lo);
 if (tag==otag)		/* check simplest case */
  return 1;
 else if (lua_isnil(lo) && 
          tag!=tolua_tag_number &&
          tag!=tolua_tag_table &&
          tag!=tolua_tag_cfunction &&
          tag!=tolua_tag_function
         )
  return 1;
 else if ((tag==tolua_tag_string && lua_isstring(lo)) || /* check convertions */
          (tag==tolua_tag_number && lua_isnumber(lo))
         )
  return 1;
 else
 {
  /* if requested type is const, the non-const is an alternative type */
  int tag2;
  lua_pushobject(lua_getref(tbl_const));
  lua_pushnumber(tag);
  tag2 = (int)lua_getnumber(lua_gettable());
  if (tag2 && tag2==otag)
   return 1;
  /* check for base classes */
  otag = basetag(otag);  
  while (otag)
  {
   if (tag==otag || (tag2 && tag2==otag))
    return 1;
   otag = basetag(otag);
  }
  return 0;
 }
}

void toluaI_tt_init (void)
{
 lua_Object lo;
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_itype");
 lua_pushobject(lo); tbl_itype = lua_ref(1);
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_itag");
 lua_pushobject(lo); tbl_itag = lua_ref(1);
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_const");
 lua_pushobject(lo); tbl_const = lua_ref(1);
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_hierarchy");
 lua_pushobject(lo); tbl_hierarchy = lua_ref(1);
 toluaI_st_store(&tbl_itype);
 toluaI_st_store(&tbl_itag);
 toluaI_st_store(&tbl_const);
 toluaI_st_store(&tbl_hierarchy);

 /* set and register basic Lua type tag */
 lua_pushnil(); tolua_tag_nil = lua_tag(lua_pop());
 lua_pushnumber(0.0); tolua_tag_number = lua_tag(lua_pop());
 lua_pushstring(""); tolua_tag_string = lua_tag(lua_pop());
 lua_pushuserdata(NULL); tolua_tag_userdata = lua_tag(lua_pop());
 lua_pushobject(lo); tolua_tag_table = lua_tag(lua_pop());
 lua_pushcfunction(toluaI_tt_init); tolua_tag_cfunction = lua_tag(lua_pop());
 lua_dostring("return tag(function () end)"); 
 tolua_tag_function = (int)lua_getnumber(lua_getresult(1));

 toluaI_tt_register(tolua_tag_nil,"nil");
 toluaI_tt_register(tolua_tag_number,"number");
 toluaI_tt_register(tolua_tag_string,"string");
 toluaI_tt_register(tolua_tag_userdata,"userdata");
 toluaI_tt_register(tolua_tag_table,"table");
 toluaI_tt_register(tolua_tag_cfunction,"cfunction");
 toluaI_tt_register(tolua_tag_function,"function");

 /* allows cfunction to be used as function */
 toluaI_tt_sethierarchy(tolua_tag_cfunction,tolua_tag_function);
}


void toluaI_tt_register (int tag, char* type)
{
 lua_pushobject(lua_getref(tbl_itype));
 lua_pushnumber(tag);
 lua_pushstring(type);
 lua_settable();

 lua_pushobject(lua_getref(tbl_itag));
 lua_pushstring(type);
 lua_pushnumber(tag);
 lua_settable();
}


void toluaI_tt_class (lua_Object lo, char* derived, char* base)
{
 int tag, ctag;
 char cderived[64] = "const "; strcat(cderived,derived);
 tag = toluaI_tt_gettag(derived);
 ctag = toluaI_tt_gettag(cderived);
 toluaI_tm_instance(tag,lo);
 toluaI_tm_instance(ctag,lo);
 if (*base != 0)
 {
  int btag, cbtag;
  char cbase[64] = "const "; strcat(cbase,base);
  btag = toluaI_tt_gettag(base);
  cbtag = toluaI_tt_gettag(cbase);
  toluaI_tt_sethierarchy(tag,btag);
  toluaI_tt_sethierarchy(ctag,cbtag);
 }
}

void toluaI_tt_sethierarchy (int tag, int btag)
{
 lua_pushobject(lua_getref(tbl_hierarchy));
 lua_pushnumber(tag);
 lua_pushnumber(btag);
 lua_settable();
}

void tolua_usertype (char* type)
{
 /* check if type already register */
 lua_pushobject(lua_getref(tbl_itag));
 lua_pushstring(type);
 if (lua_isnil(lua_gettable()))
 {
  int tag, ctag;
  char ctype[64] = "const "; strcat(ctype,type);
  tag = lua_newtag();
  ctag = lua_newtag();
  toluaI_tt_register(tag,type);
  toluaI_tt_register(ctag,ctype);
  /* set const table */
  lua_pushobject(lua_getref(tbl_const));
  lua_pushnumber(ctag); lua_pushnumber(tag); lua_settable();
 }
}

int toluaI_tt_isusertype (lua_Object lo)
{
 if (lua_isuserdata(lo) && tolua_tag_userdata!=lua_tag(lo))
 {
  lua_pushobject(lua_getref(tbl_itype));
  lua_pushnumber(lua_tag(lo));
  return !lua_isnil(lua_gettable());
 }
 return 0;
}

void tolua_settag (char* type, int* tag)
{
 lua_pushobject(lua_getref(tbl_itag));
 lua_pushstring(type);
 *tag = (int) lua_getnumber(lua_gettable());
 toluaI_st_store(tag);
}

int tolua_istype (int narg, int tag, int def)
{
 lua_Object lo = lua_getparam(narg);
 if (lo == LUA_NOOBJECT)
 {
  if (def==0)
  {
   toluaI_eh_set(narg,toluaI_tt_getobjtype(lo),gettype(tag));
   return 0;
  }
 }
 else
 {
  if (!istype(lo,tag))
  {
   toluaI_eh_set(narg,toluaI_tt_getobjtype(lo),gettype(tag));
   return 0;
  }
 }
 return 1;
}

int tolua_arrayistype (int narg, int tag, int dim, int def)
{
 int i;
 lua_Object lo = lua_getparam(narg);
 lua_Object tf;
 for (i=0; i<dim; ++i)
 {
  lua_beginblock();
  lua_pushobject(lo); lua_pushnumber(i+1);
  tf = lua_gettable();
  if (!istype(tf,tag) && (!def || !lua_isnil(tf)))
  {
   char t1[64], t2[64];
   sprintf(t1,"array of %s",toluaI_tt_getobjtype(tf));
   sprintf(t2,"array of %s (dimension=%d)",gettype(tag),dim);
   toluaI_eh_set(narg,t1,t2);
   lua_endblock();
   return 0;
  }
  lua_endblock();
 }
 return 1;
}

int tolua_isnoobj (int narg)
{
 lua_Object lo = lua_getparam(narg);
 if (lo!=LUA_NOOBJECT)
 {
  toluaI_eh_set(narg,toluaI_tt_getobjtype(lo),
                toluaI_tt_getobjtype(LUA_NOOBJECT)); 
  return 0;
 }
 return 1;
}

