package Charset::Baudot;

# Convert between Baudot and ASCII

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION);

$VERSION = '0.05';

my @charset = (
	"\000E\nA SIU\rDRJNFCKTZWLHYPQOBG2MXV1",
	"\000e\na siu\rdrjnfcktzwlhypqobg2mxv1",
	"\0003\n- \a87\r\$4',!:(5\")2 6019?&3./;0",
	"\000\242\n+ \\#=\r*{~\245|^<[}>]\b@\253\243\254\377\2613%_\2730",
);

my $charset = join('', map { "\000" . substr($_, 1, 26) .
			     "\000" . substr($_, 28, 3) . "\000" }
		           @charset);
push @charset, '';

sub import {
    my $class = shift;
    my ($callpack, $callfile, $calline) = caller;
    my @EXPORT;
    if (@_) {
    	@EXPORT = @_;
    } else {
    	@EXPORT = qw(ascii2baudot baudot2ascii);
    }
    my $sym;
    foreach $sym (@EXPORT) {
	if ($sym eq 'ascii2baudot' || $sym eq 'baudot2ascii') {
	    *{"$callpack\::$sym"} = \&{"$class\::$sym"};
	} else {
	    die "015 Cannot export $sym at $callfile line $calline\n";
	}
    }
}

sub baudot2ascii {
    my ($callpack, $callfile, $calline) = caller;
    @_ == 1 or
	die "013 Usage: baudot2ascii(STRING) at $callfile line $calline\n";
    my $string = shift;
    my $set = 0;
    my $result = '';
    while ($string ne '') {
    	my $chr = ord($string) & 037;
	$string = substr($string, 1);
	if ($chr == 033 || $chr == 037) {
	    $set = ord(substr($charset[$set], $chr, 1)) & 03;
	} else {
	    $result .= substr($charset[$set], $chr, 1);
	}
    }
    $result;
}

sub ascii2baudot {
    my ($callpack, $callfile, $calline) = caller;
    @_ == 1 or
	die "013 Usage: ascii2baudot(STRING) at $callfile line $calline\n";
    my $string = shift;
    my $set = 4;
    my $result = '';
    while ($string ne '') {
    	my $chr = substr($string, 0, 1);
	$string = substr($string, 1);
	my $pos = index(substr($charset[$set], 0, 30), $chr);
	if ($pos < 0 || $pos == 033 || $pos == 037) {
	    $pos = index($charset, $chr);
	    die "109 Invalid character in string: '$chr' at $callfile line $calline\n"
	    	if $pos < 0 || $chr eq "\000";
	    my $s = $pos >> 5;
	    $pos = $pos & 037;
	    if ($set > 3) {
		$result .= ['[_', '__', '_[', '[[']->[$s];
	    } else {
		$result .= ['', '_', '[', '[[',
			    '[_', '', '[', '[[',
			    '_', '__', '', '[[',
			    '_', '__', '_[', '',
			   ]->[($set << 2) | $s];
	    }
	    $set = $s;
	}
	$result .= sprintf("%c", 0x40 + $pos);
    }
    $result;
}

1;

__END__

=head1 NAME

Charset::Baudot - allows to use Baudot string constants in ASCII programs (and v.v.)

=head1 SYNOPSIS

    use Charset::Baudot;

    my $a = baudot2ascii"(Baudot text)";

=head1 DESCRIPTION

I<Charset::Baudot> defines functions to convert between a subset of ASCII and a
subset of nonstandard Baudot - the original Baudot allows only letters,
numbers, and some punctuation. We assume that a "Shift to letters" code
while already in letters mode means "Shift to lowercase" and "Shift to
figures" while already in figures mode means "Shift to symbols". This allows
to use up to 120 characters. However, for simplicity some characters are
available in multiple sets, so the total is less than that.

By default, both functions I<baudot2ascii> and I<ascii2baudot> are
imported in your namespace. If you don't want that, you know how to
avoid it. They do the obvious thing to their first argument and
return the transformed string.

There is another function, I<standardbaudot>, not exported by default,
which sets the module's behaviour to shandard Baudot.

Well, that's all.

=head1 BAUDOT CHARACTER TABLE

The following are the characters recognised. As described, the "shift"
characters have nonstandard meaning.

     set   Letters     Lowercase    Figures    Symbols
  code
    00       N/A          N/A         N/A        N/A
    01        E            e           3        Cents
    02       L/F          L/F         L/F        L/F    (line feed)
    03        A            a           -          +
    04      Space        Space       Space      Space
    05        S            s          BELL        \
    06        I            i           8          #
    07        U            u           7          =
    08       C/R          C/R         C/R        C/R    (carriage return)
    09        D            d           $          *
    10        R            r           4          {
    11        J            j           '          ~
    12        N            n           ,         XOR
    13        F            f           !          |
    14        C            c           :          ^
    15        K            k           (          <
    16        T            t           5          [
    17        Z            z           "          }
    18        W            w           )          >
    19        L            l           2          ]
    20        H            h          N/A     backspace
    21        Y            y           6          @
    22        P            p           0         N/A
    23        Q            q           1        POUND
    24        O            o           9         NOT
    25        B            b           ?        delete
    26        G            g           &         N/A
    27     Figures      Figures     Symbols    Symbols
    28        M            m           .          %
    29        X            x           /          _
    30        V            v           ;         N/A
    31    Lowercase    Lowercase    Letters    Letters

=head1 NOTES

This module reimplements some of the functionality of two other modules
(see L<Exporter>, L<Carp>, I<Reinventing the Wheel>). This is intentional,
as it will leave larger scope for obfuscation in a future release.

=head1 COPYRIGHT

This module is part of CLC-INTERCAL.

Copyright (c) 1999 by Claudio Calvelli E<lt>C<lunatic@assurdo.com>E<gt>,
all (f)rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

=head1 SEE ALSO

A qualified psychiatrist.

