// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// initwindow.cpp
//
// History:
//  23-03-2001 - created
//  29-05-2001 - moved CreateFilesList() to fileio.h
////////////////////////////////////////////////////////////////////////

#include <string.h>

#include "initwindow.h"
#include "fileio.h"
#include "misc.h"
#include "game.h"
#include "globals.h"

#define ICON_QUIT_X	220
#define ICON_QUIT_Y	32
#define ICON_QUIT_W	7
#define ICON_QUIT_H	7

#define IW_WIDGET_LIST  	0
#define IW_WIDGET_START_NEW	1
#define IW_WIDGET_START_SAVED	2
#define IW_WIDGET_LOAD_SAVED	3
#define IW_WIDGET_LOAD_NEW	4
#define IW_WIDGET_PBEM		5
#define IW_WIDGET_AI		6
#define IW_WIDGET_VIDEO_OPTIONS 7


////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::InitWindow
// DESCRIPTION: Create the main menu window. The player(s) can select
//              the level they want to play and set other options.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
//
// HISTORY
//   04-04-2001 - quit button activated by pressing ESCAPE
//   19-08-2001 - added checkbox to play against AI
////////////////////////////////////////////////////////////////////////

#define LOGO_X	0
#define LOGO_Y	64
#define LOGO_W	362
#define LOGO_H	105

InitWindow::InitWindow( View *view ) : Window( WIN_CENTER, view ) {
  // calculate window dimensions
  SetSize( LOGO_W + 20, LOGO_H + sfont->Height() * 15 + 30 );

  logo = new Image( Icons, LOGO_X, LOGO_Y, LOGO_W, LOGO_H );
  ColLight = Color( 0x00FC7738 );
  ColDark = Color( 0x00FE3112 );

  create_files_list( get_levels_dir().c_str(), ".lev", levels );
  create_files_list( get_save_dir().c_str(), ".sav", saves );

  // create widgets
  levwidget = new TextListWidget( IW_WIDGET_LIST, 10, logo->Height() + 20,
                    LOGO_W/2 - 5, sfont->Height() * 15,
                    &levels, -1, WIDGET_ARROWSCROLL, NULL, this );

  startwidget = new ButtonWidget( IW_WIDGET_START_NEW,
                levwidget->LeftEdge() + levwidget->Width() + 10,
                levwidget->TopEdge() + levwidget->Height() - sfont->Height() - 8,
                levwidget->Width(), sfont->Height() + 8, 's',
                WIDGET_DEFAULT, "Start", this );
  startwidget->SetHook( this );

  listctrlwidget = new ButtonWidget( IW_WIDGET_LOAD_SAVED,
                   startwidget->LeftEdge(), levwidget->TopEdge(),
                   startwidget->Width(), startwidget->Height(), 'l', 0,
                   "Load saved game", this );
  listctrlwidget->SetHook( this );

  videowidget = new ButtonWidget( IW_WIDGET_VIDEO_OPTIONS, listctrlwidget->LeftEdge(),
                   listctrlwidget->TopEdge() + listctrlwidget->Height() + 5,
                   listctrlwidget->Width(), listctrlwidget->Height(), 'v', 0,
                   "Video Options", this );
  videowidget->SetHook( this );

  pbemwidget = new CheckboxWidget( IW_WIDGET_PBEM, videowidget->LeftEdge() + 5,
                   videowidget->TopEdge() + videowidget->Height() + 10,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, false, 'e',
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
                   "Play By e-Mail", this );
  pbemwidget->SetHook( this );

  aiwidget = new CheckboxWidget( IW_WIDGET_AI, pbemwidget->LeftEdge(),
                   pbemwidget->TopEdge() + pbemwidget->Height() + 3,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, false, 'c',
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
                   "Play vs. Computer", this );
  aiwidget->SetHook( this );

  // the quit button
  ButtonWidget *btn = new ButtonWidget( GUI_QUIT, w - ICON_QUIT_W - 8,
                      LOGO_H / 2 + 14, ICON_QUIT_W + 4, ICON_QUIT_H + 4,
                      SDLK_ESCAPE, WIDGET_STYLE_GFX, NULL, this );
  btn->SetImage( Icons, Rect( ICON_QUIT_X, ICON_QUIT_Y, ICON_QUIT_W, ICON_QUIT_H ),
                        Rect( ICON_QUIT_X, ICON_QUIT_Y, ICON_QUIT_W, ICON_QUIT_H ) );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::~InitWindow
// DESCRIPTION: Destroy window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

InitWindow::~InitWindow( void ) {
  while ( !levels.IsEmpty() ) {
    TLWNode *n = static_cast<TLWNode *>( levels.RemHead() );
    delete [] n->name;
    delete n;
  }
  delete logo;
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::Draw
// DESCRIPTION: Draw the main window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void InitWindow::Draw( void ) {
  Window::Draw();
  logo->Draw( this, (w - logo->Width()) / 2, 10 );

  DrawBox( Rect( videowidget->LeftEdge(),
                 pbemwidget->TopEdge() - 5,
                 videowidget->Width(),
                 startwidget->TopEdge() - pbemwidget->TopEdge() - 5),
           BOX_CARVED );
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::Activate
// DESCRIPTION: Handle activation of widgets in the window.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status InitWindow::Activate( ButtonWidget *button, Window *win ) {

  switch ( button->ID() ) {
  case IW_WIDGET_START_NEW:
  case IW_WIDGET_START_SAVED: {
    TLWNode *n = static_cast<TLWNode *>( levwidget->Selected() );
    if ( !n ) new NoteWindow( "Error!", "No level selected!", 0, view );
    else {
      unsigned short game_flags = 0;
      string lname;
      if ( button->ID() == IW_WIDGET_START_NEW ) {
        lname.assign( get_levels_dir() );
        if ( pbemwidget->Clicked() &&
             !pbemwidget->Disabled() ) game_flags |= GI_PBEM;
        if ( aiwidget->Clicked() &&
             !aiwidget->Disabled() ) game_flags |= GI_AI;
      } else lname.assign( get_save_dir() );
      lname.append( n->name );

      Gam = new Game( game_flags, view );
      if ( Gam->Load( lname.c_str() ) ) {
        NoteWindow *nw = new NoteWindow( "Error!", "Failed to load mission!", 0, view );
        nw->button->SetID( GUI_RESTART );
      } else {
        view->CloseWindow( this );
        Gam->NextTurn();         // let the games begin...
      }
    }
    break; }
  case IW_WIDGET_LOAD_NEW:
    startwidget->SetID( IW_WIDGET_START_NEW );
    listctrlwidget->SetID( IW_WIDGET_LOAD_SAVED );
    listctrlwidget->SetTitle( "Load saved game" );
    listctrlwidget->Draw();
    levwidget->SwitchList( &levels, -1 );
    pbemwidget->Enable();
    pbemwidget->Draw();
    aiwidget->Enable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_LOAD_SAVED:
    startwidget->SetID( IW_WIDGET_START_SAVED );
    listctrlwidget->SetID( IW_WIDGET_LOAD_NEW );
    listctrlwidget->SetTitle( "Launch new game" );
    listctrlwidget->Draw();
    levwidget->SwitchList( &saves, -1 );
    pbemwidget->Disable();
    pbemwidget->Draw();
    aiwidget->Disable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_PBEM:
    if ( pbemwidget->Clicked() ) aiwidget->Disable();
    else aiwidget->Enable();
    aiwidget->Draw();
    Show();
    break;
  case IW_WIDGET_AI:
    if ( aiwidget->Clicked() ) pbemwidget->Disable();
    else pbemwidget->Enable();
    pbemwidget->Draw();
    Show();
    break;
  case IW_WIDGET_VIDEO_OPTIONS:
    new VideoOptionsWindow( view );
    break;
  }

  return GUI_OK;
}

#define VOW_WIDGET_MODES	0
#define VOW_WIDGET_OK		1
#define VOW_WIDGET_BACK		GUI_CLOSE
#define VOW_WIDGET_FULLSCREEN	3


////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::VideoOptionsWindow
// DESCRIPTION: Show the video options menu.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
//
// HISTORY
//   24-10-2001 - don't show modes smaller than 640x480
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::VideoOptionsWindow( View *view ) : Window( WIN_CENTER, view ) {
  static SDL_Rect std_sizes[] = { { 1280, 1024 }, { 1024, 768 },
                                  { 800, 600}, { 640, 480 } };
  SDL_Rect *std_modes[5];
  short current = -1;

  // calculate window dimensions
  SetSize( lfont->Width() * 14 + 30, sfont->Height() * 12 + lfont->Height() + 45 );

  // create modes list
  SDL_Rect **sdl_modes = SDL_ListModes( NULL, SDL_FULLSCREEN );

  // if any mode is ok offer some standard ones
  if ( sdl_modes == (SDL_Rect **)-1 ) {
    sdl_modes = std_modes;
    std_modes[0] = &std_sizes[0];
    std_modes[1] = &std_sizes[1];
    std_modes[2] = &std_sizes[2];
    std_modes[3] = &std_sizes[3];
    std_modes[4] = NULL;
  }
    
  if ( sdl_modes != NULL ) {
    char buf[12];

    for ( int i = 0; sdl_modes[i]; i++ ) {
      if ( (sdl_modes[i]->w >= 640) && (sdl_modes[i]->h >= 480) ) {
        TLWNode *n = new TLWNode;
        sprintf( buf, "%d x %d", sdl_modes[i]->w, sdl_modes[i]->h );

        n->name = new char[ strlen(buf) ];
        n->user_data = sdl_modes[i];
        strcpy( n->name, buf );
        modes.AddTail( n );

        if ( (sdl_modes[i]->w == view->Width()) &&
             (sdl_modes[i]->h == view->Height()) ) current = i;
      }
    }
  }

  // create widgets
  modewidget = new TextListWidget( VOW_WIDGET_MODES, 10, lfont->Height() + 20,
                    w - 20, sfont->Height() * 10, &modes, current,
                    WIDGET_ARROWSCROLL|WIDGET_ALIGN_CENTER, NULL, this );

  ButtonWidget *ok = new ButtonWidget( VOW_WIDGET_OK, 1,
                     Height() - sfont->Height() - 9, (Width() - 2) / 2,
                     sfont->Height() + 8, 's', 0, "Set", this );
  ok->SetHook( this );

  new ButtonWidget( VOW_WIDGET_BACK, 1 + (Width() - 2) / 2, Height() - sfont->Height() - 9,
                    (Width() - 2) / 2, sfont->Height() + 8, 'c',
                    WIDGET_DEFAULT, "Close", this );

  fswidget = new CheckboxWidget( VOW_WIDGET_FULLSCREEN, modewidget->LeftEdge() + 10,
                   modewidget->TopEdge() + modewidget->Height() + 10,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, view->IsFullScreen(), 'f',
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
                   "Fullscreen", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::~VideoOptionsWindow
// DESCRIPTION: Destroy window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::~VideoOptionsWindow( void ) {
  while ( !modes.IsEmpty() ) {
    TLWNode *n = static_cast<TLWNode *>( modes.RemHead() );
    delete [] n->name;
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::Draw
// DESCRIPTION: Draw the video options window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void VideoOptionsWindow::Draw( void ) {
  Window::Draw();

  const char *title = "Video Options";

  lfont->Write( title, this, (w - lfont->TextWidth(title)) / 2 + 3,
                             11, ColDark );
  lfont->Write( title, this, (w - lfont->TextWidth(title)) / 2,
                             8, ColLight );
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::Activate
// DESCRIPTION: When the user pushes the 'Use' button, switch to the
//              selected video mode.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status VideoOptionsWindow::Activate( ButtonWidget *button, Window *win ) {
  TLWNode *mode = static_cast<TLWNode *>( modewidget->Selected() );

  if ( mode ) {
    bool fs = fswidget->Clicked();
    unsigned long mode_flags = SDL_HWSURFACE|(fs ? SDL_FULLSCREEN : 0);

    // if selected mode is the same as current mode only check for fullscreen
    // dimensions of the selected mode are available in the user_data field
    SDL_Rect *dim = (SDL_Rect *)mode->user_data;
    if ( (dim->w == view->Width()) && (dim->h == view->Height()) ) {
      if ( fs != view->IsFullScreen() ) view->ToggleFullScreen();
    } else {
      view->SetVideoMode( dim->w, dim->h, DISPLAY_BPP, mode_flags );
    }
  }

  return GUI_OK;
}

