/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#ifndef _P11_MANAGER_HH_
#define _P11_MANAGER_HH_


#include <openssl/rsa.h>
#include <openssl/evp.h>

#include "pkcs11/pkcs11_util.h"
#include "pkcs11/pkcs11_find.h"
#include "Certificate.hh"
#include "P11Error.hh"
#include "P11Slot.hh"

#include <vector>
#include <string>

namespace Cryptonit
{

/** Class responsible for managing PKCS#11 operations.
 * You should refer to the PKCS#11 Standard, section 6.6 .
 */
class P11Manager
{

#define P11LIBCRYPTONIT_MAX_PATH_LENGTH 1024

public:
	/** The constructor starts Cryptoki (the PKCS#11 API), 
	 * lists the slots and initiates read-only public sessions.
	 * @param em : pointer to a subclass of P11Error
	 * @param pm : pointer to a subclass of PasswordManager
	 * @param lib_path : string indicating the path to the PKCS#11 library
	 * module
	 * 
	 * The path must be NULL-terminated, it will be truncated at
	 * P11LIBCRYPTONIT_MAX_PATH_LENGTH, which is currently 1024.
	 */
	P11Manager(P11Error * em, 
		PasswordManager * pm,
		const char * lib_path);
	
	/** The destructor closes all sessions and stops Cryptoki.
	 */
	~P11Manager();
	
	/** Restarts the Cryptoki interface. 
	 * In case you need to take into account newly plugged devices,
	 * you will need to call this function, which does as if you
	 * destroyed and re-created a P11Manager object.
	 */
	int restart();
	
	/** @return true if the PKCS#11 library has been successfully loaded */
	bool live() {return loaded;}
	
	bool getInfos( int &cryptokiVersionMajor,
			int &cryptokiVersionMinor,
			std::string &manufacturerID,
			std::string &libraryDescription,
			int &libraryVersionMajor,
			int &libraryVersionMinor);
	
	/** Used to retrieve the list of certificates currently on the PKCS#11 devices.
	 * @return a vector of Certificate
	 * Once the user is shown this list of certificates, he should be able to
	 * choose one to perform cryptographic functions (see getKey to retrieve
	 * the corresponding private key handle).
	 */
	std::vector<Certificate> listCertificates();
	
	int signWithPrivKey(EVP_PKEY * key, int flen, const unsigned char *from,
		     unsigned char *to, int padding);
	int decryptWithPrivKey(EVP_PKEY * key, int flen, const unsigned char *from,
			   unsigned char *to, int padding);
	
	/** Get a handle on the key corresponding to the given certificate.
	 * Notice that the key object does NOT contain the private key itself, which 
	 * stays on the token, but instead it contains functions pointers to 
	 * the private cryptographic functions on the token.
	 */
	GetKeyReturn getKey(Certificate & c, Key **key);
	
	/** Should normally not be used but for testing purposes */
	int updateSlotList();
	
	/** For testing and information displaying purposes */
	std::vector<P11Slot*> getSlotList() {return slots;}
	
private:
	CK_RV start();
	CK_RV stop();
	CK_RV listSlots(bool withToken = false); //!< Scan or rescan the available slots
	void clearSlots();

	P11Error * err;
	PasswordManager * passMan;
	
	//! functions available through the Cryptoki API
	CK_FUNCTION_LIST_PTR funcs;
	std::vector<P11Slot *> slots;
	std::string * libPath;
	bool loaded; //!< true if the PKCS#11 library has been successfully loaded

}; // class P11Manager

} // namespace Cryptonit

#endif // _P11_MANAGER_HH_
