/* path.cxx
     $Id: path.cxx,v 1.3 1998/07/27 05:39:07 elf Exp $

   written by Marc Singer
   27 October 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   in a file called COPYING along with this program; if not, write to
   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
   02139, USA.

   -----------
   DESCRIPTION
   -----------

*/

  /* ----- Includes */

#include "std.h"
#include <sys/stat.h>
#include <pwd.h>		// Password file parsing routines
#include "path.h"


  /* ----- Class Globals/Statics */

#define CB_PATH_ALLOC		32
#define CH_PATHSEP		'/'
#define SZ_PATHSEP		"/"

  /* ----- Methods */

/* LPath::alloc

   on return, it guarantees that there are at least cchNeed bytes
   available for path data.  We make sure there is one more byte
   available for a terminating NULL.

*/

void LPath::alloc (int cchNeed)
{
  ++cchNeed;			// Space for a NULL
  if (cchNeed <= m_cchMax)
    return;

  int cchNeedNew = (((cchNeed + CB_PATH_ALLOC - 1)/CB_PATH_ALLOC)
		    *CB_PATH_ALLOC);
  assert_ (cchNeedNew >= cchNeed);
  char* sz = (char*) malloc (cchNeedNew);
  assert_ (sz);
  if (m_sz) {
    memcpy (sz, m_sz, m_cchMax);
    free (m_sz);
    m_sz = NULL;
  }  /* if */
  else
    *sz = 0;
  m_sz = sz;
  m_cchMax = cchNeedNew;
}  /* LPath::alloc */


/* LPath::cat

   appends a complete path element to the end of the current path.
   A complete element is a directory or filename.  To append a piece
   of a pathname, use cat_literal.

*/

void LPath::cat (const char* sz)
{
  int cch = length ();
  assert_ (sz);
  int cchNeed = cch + strlen (sz) + 1;		// +1 for separator
  alloc (cchNeed);

        // Put a path separator at the end
  if (cch && m_sz[cch - 1] != CH_PATHSEP)
    strcat (m_sz + cch, SZ_PATHSEP);

  strcat (m_sz, sz);
  clean ();
}  /* LPath::cat */


void LPath::cat_literal (const char* sz)
{
  alloc (length () + strlen (sz) + 1);
  strcat (m_sz, sz);
}  /* LPath::cat_literal */


/* LPath::clean

   convert path to a connicalized form.  We collapse references to '.'
   and to '..'.  We do permit a leading './' as well as any number of
   leading '../'.

   We also interpret ~user expansions.  In cleaning the path, we
   convert the ~ notation to the full name of the home directory.

*/

void LPath::clean (void)
{
  if (!m_sz)
    return;

				// -- Substitution for ~username
  if (*m_sz == '~') {
    int cchUser = strcspn (m_sz + 1, "/");
    setpwent ();
    struct passwd* pEnt;
    while ((pEnt = getpwent ())) {
      if (cchUser) {
	if (strncmp (pEnt->pw_name, m_sz + 1, cchUser) == 0
	    && !pEnt->pw_name[cchUser])
	  break;
      }
      else if (pEnt->pw_uid == getuid ())
	break;
    }
    assert_ (pEnt);		// FIXME: we really should have a fallback
    int cchDir = strlen (pEnt->pw_dir);
    if (pEnt->pw_dir[cchDir - 1] == '/' && m_sz[cchUser + 1] == '/')
      --cchDir;
    alloc (length () + cchDir - (cchUser + 1));
    memmove (m_sz + cchDir, m_sz + cchUser + 1, length () - (cchUser + 1) + 1);
    memcpy (m_sz, pEnt->pw_dir, cchDir);
  }

#if defined (MSDOS)
        // Make all path separators forward slashes
  int cch = length ();
  char* pch = m_sz;
  for (; cch--; ++pch)
    if (*pch == '\\')
      *pch = CH_PATHSEP;
#endif

  char* pchPrefix = m_sz;
  	// Skip over prefix of "../"'s or "./"
  for (; *pchPrefix && !strncmp (pchPrefix, "../", 3);
       pchPrefix += 3)
    ;
  if (pchPrefix == m_sz && !strncmp (pchPrefix, "./", 2))
    pchPrefix += 2;

        // -- Collapse '.' and '..' elements
  char* pch = m_sz + length ();
  char* pchSep = pch;
  int state = 0;
  int cchCopy = 0;
  for (int cDelete = 0; --pch > pchPrefix;) {
    ++cchCopy;
        // -- Process path elements on encountering path separators
    if (*pch == CH_PATHSEP) {
      ++cchCopy;
      switch (state) {
      case 2:			// Reference to parent directory '..'
	cDelete += 2;		// Delete this `..' and preceeding element
		// -- fall through
      case -1:			// Ordinary path element
      case 0:			// Empty path element, reference to `//'?
	if (!cDelete)
	  break;
	else
	  --cDelete;
                // -- fall through
      case 1:		// Reference to current directory '.'
	cchCopy -= pchSep - pch;
	memmove (pch, pchSep, cchCopy);
	break;
      }  /* switch */
      pchSep = pch;
      state = 0;
      continue;
    }  /* if */

        // -- Non-special path characters
    if (*pch != '.') {
      state = -1;
      continue;
    }  /* if */

        // -- Analyze special directory references
    switch (state) {
      default:
        break;

      case 0:	// Beginning of element
        state = 1;
        break;

      case 1:	// "." element already found
        state = 2;
        break;

      case 2:	// ".." element already found
        state = -1;
        break;
    }  /* switch */
  }  /* for */
}  /* LPath::clean */


/* LPath::dup

   duplicates the contents of the given string in this->m_sz.

*/

void LPath::dup (const char* sz)
{
  erase ();
  if (!sz || !*sz) {
    alloc (0);
    *m_sz = 0;
  }
  else {
    int cch = strlen (sz);
    alloc (cch);
    strncpy (m_sz, sz, cch + 1);
  }  /* if */
}  /* LPath::dup */

void LPath::dup (const char* sz, int cch)
{
  erase ();
  if (!sz || !*sz || !cch) {
    alloc (0);
    *m_sz = 0;
  }
  else {
    alloc (cch);
    strncpy (m_sz, sz, cch);
    m_sz[cch] = 0;
  }  /* if */
}  /* LPath::dup */


/* LPath::extension

   returns a pointer to the filename extension, NULL if there isn't
   one.  The return value points to the '.' of the extension if one is
   found.

*/

const char* LPath::extension (void)
{
  char* pch = m_sz;

	// Find the last path element
  int cch;
  while (pch[cch = strcspn (pch, SZ_PATHSEP)])
    pch += cch + 1;

	// Find the extension
  for (; *pch; ++pch)
    if (*pch == '.')
      return pch;

  return NULL;
}  /* LPath::extension */


/* LPath::init

   zeroes the path.

*/

void LPath::init (void)
{
  m_sz = NULL;
  m_cchMax = 0;
}  /* LPath::init */


/* LPath::is_relative

   returns true if the path is NOT absolute.  Absolute paths have a
   driver letter prefix (MSDOS) or a leading '/' that specifies a path
   starting at the filesystem root.

*/

bool LPath::is_relative (void) const
{
  return m_sz && !(   *m_sz == CH_PATHSEP
#if defined (MSDOS) && defined (CH_DRIVESEP)
		   || (m_cchMax && m_sz[1] == CH_DRIVESEP && isalpha(*m_sz))
#endif
		   );
}  /* LPath::is_relative */


/* LPath::relative_to

   make this->m_sz path relative to the directory szDir.  It returns
   true if the path has changed and is valid.

*/

bool LPath::relative_to (const char* szDir)
{
  char* szPath = m_sz;

	// -- Degenerative case of CWD as szDir
  if (!strcmp (szDir, "."))
    return true;

	// -- Strip `./' from CWD relative paths
  if (!strncmp (szPath, "./", 2))
      szPath += 2;
  if (!strncmp (szDir, "./", 2))
      szDir += 2;

	// -- Check for absolute/relative mismatch
  if ((*szPath == CH_PATHSEP || *szDir == CH_PATHSEP) && *szDir != *szPath)
    return false;

//  int cchPath = strlen (szPath);
  int cchDir = strlen (szDir);

  	// -- Degenerative case: no directory
  if (cchDir == 0)
    return true;

  	// -- Find length of common prefix
  int cchPrefix;
  for (cchPrefix = 0;
       szPath[cchPrefix] && szDir[cchPrefix]
       && szPath[cchPrefix] == szDir[cchPrefix];
       ++cchPrefix)
    ;
  if (!szDir[cchPrefix] && szPath[cchPrefix] == CH_PATHSEP)
    ++cchPrefix;

  	// -- Degenerative case: directory equivalent to path
//  if (cchPrefix == cchPath)
//    return false;

	// -- Determine length of common root path (includes final sep. or \0)
  int cchRoot;
  for (cchRoot = cchPrefix; cchRoot; --cchRoot) {
    if (  (szDir [cchRoot - 1] == CH_PATHSEP || !szDir[cchRoot - 1])
	&& szPath[cchRoot - 1] == CH_PATHSEP)
      break;
  }  /* for */

	// -- Count ascending directories from szDir to common root
  int iDirAscend = 0;
  if (cchRoot < cchDir) {
    if (!cchRoot)
      ++iDirAscend;
    for (int ich = cchRoot ? cchRoot - 1 : 0; szDir[ich]; ++ich)
      if (szDir[ich] == CH_PATHSEP && szDir[ich + 1])
	++iDirAscend;
  }  /* if */

  LPath pathNew;
  if (iDirAscend) {
    while (iDirAscend--)
      pathNew.cat ("..");
  }  /* if */
  else
    pathNew.cat (".");

  pathNew.cat (szPath + cchRoot);
  *this = pathNew;
  return true;
}  /* LPath::relative_to */


/* LPath::release

   release memory.

*/

void LPath::release (void)
{
  if (m_sz)
    free (m_sz);
  m_sz = NULL;
  m_cchMax = 0;
}  /* LPath::release */


/* LPath::tail

   returns a const char* the last path element.

*/

const char* LPath::tail (void)
{
  char* pch = m_sz;

  int cch;
  while (pch[cch = strcspn (pch, SZ_PATHSEP)])
    pch += cch + 1;

  return pch;
}  /* LPath::tail */


/* LPath::trim

   trims the last path element, file or directory, from the path.

*/

void LPath::trim (void)
{
  char* pch = m_sz;
  char* pchSep = NULL;


  int cch;
  while (pch[cch = strcspn (pch, SZ_PATHSEP)]) {
    pchSep = pch + cch;
    pch += cch + 1;
  }  /* while */

#if defined (MSDOS) && defined (CH_DRIVESEP)
  if (pchSep && pchSep != m_sz && *(pchSep - 1) == CH_DRIVESEP)
    ++pchSep;
#endif

  *pchSep = 0;
}  /* LPath::trim */
