/************************************************n***************************
                          spielfeld.cpp  -  description
                             -------------------
    begin                : Sat Oct 2 1999
    copyright            : (C) 1999 by immi
    email                : cuyo@pcpool.mathematik.uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpainter.h>
#include <qarray.h>
#include <qpoint.h>
#include <cstdio>

#include "nachbariterator.h"
#include "cuyo.h"
#include "spielfeld.h"

#define papa ((Cuyo *) parent())


/***** Konstanten, die was einstellen *****/


/* Wie weit unten mssen Graue sein, damit ein neues Fall kommt? */
#define neues_fall_platz 5


/* Wenn grade neue Graue auf einen maximal hohen Turm gefallen sind,
   soll man auf jeden Fall noch ein bisschen Zeit haben, ihn wieder
   zu verkleinern. Deshalb tauchen Graue ein Stck weiter unten als
   der Hetzrand auf:
 */
/* Wo relativ zum Hetzrand... */
/* ... tauchen neue Steine auf? */
#define hetzrand_dy_auftauch 8
/* ... ist die maximal erlaubte Turmhhe? */
#define hetzrand_dy_erlaubt 0

/* Wie oft wird versucht, die "+"-Blops am Spielanfang unverbunden
   zu machen? */
#define startlevel_rand_durchgaenge 10


/* Wie lange luft die Rochade-Animation? */
#define rochade_zeit 40
#define rochade_blink_zeit 7


/***** Rckgabe-Konstanten u. . *****/

	
/* Werte fr mModus */
#define modus_keinspiel 0 // In diesem Modus ist kein Spielfeld zu sehen (Beim
// Warten auf Leertaste am Anfang vom Spiel luft das Spiel eigentlich schon,
// aber spielSchritt() wird noch nicht aufgerufen.)
#define modus_testflopp 30  // nur ein bergangsmodus: Es muss noch getestet werden, ob was platzt
#define modus_flopp 31
#define modus_rutschnach 40
#define modus_neue_graue 41
// da war grad was am runterfallen, whrend das Spiel
// beendet wurde; da lassen wir es halt platzen:
#define modus_fallplatz 90
// das Spiel ist zu Ende (alle Schlussanimationen sind fertig abgelaufen,
// wir warten nur noch darauf, dass stopLevel() aufgerufen wird):
#define modus_warte_auf_stop 91

/* Modus-Diagramm:

         ,-- warte_auf_stop <----,
         |        ^              |
         |        |          fallplatz <-,
         |        |                      |
         |        +-----------------> testflopp --> flopp
         |        |                     |    ^        |
         |        |                     |    |        |
         |        |                     |    |        |
         v        |                     |    |        |
   keinspiel --> neue_graue <-----------    |        |
                       |                     |        |
                       `-------------> rutschnach <---+



   - testflopp wird schon am Anfang von Spielschritt ausgewertet. (d. h. der
     bergang von da woandershin dauert keinen Zeitschritt)
   - Nur in keinspiel ist kein Spielfeld sichtbar
	 - Mit Escape kann von jedem Modus zu keinspiel gewechselt werden
*/

				
/* Werte fr mRueberReihenModus */
#define rrmodus_nix 0
#define rrmodus_gib_rueber 10
#define rrmodus_gib_runter 11
#define rrmodus_bekomm_hoch 20
#define rrmodus_bekomm_rueber 21




/* Rckgabewerte fr rutschNach() */
#define rutschnach_nix 0
#define rutschnach_wenig 1 // Wenn nur im unteren Bereich noch was nachrutscht,
// so dass das neue Fall kommen kann
#define rutschnach_viel 3
// Brauche: rutschnach_viel | rutschnach_wenig = rutschnach_viel





Spielfeld::Spielfeld(bool re, QWidget *parent, const char *name )
  : QWidget(parent,name), mBild(grx * gric, gry * gric),
    mFall(this)
{
  resize(grx * gric, gry * gric);
  setPalette( QPalette( white ) );

  mBlaeschen.setAutoDelete(true);

  /* fr das Reihenrberschieben... */
  rechterSpieler = re;

  mModus = modus_keinspiel;
}

Spielfeld::~Spielfeld(){
}


/** Sollte einmal direkt nach dem Start von cuyo aufgerufen werden;
    initialisiert den aktuellen Level und benutzt das Startbild als
    Titelbild. */
void Spielfeld::erzeugTitelbild() {
  startLevel();
  mTitelbild.resize(grx * gric, gry * gric);
  {
    QPainter p(&mTitelbild);
    malUpdate(p);
  }
  stopLevel(false);
}



/**  */
void Spielfeld::paintEvent(QPaintEvent * e){

  QRect r = e->rect();
  bool schrift_mitte = true;

  {
    QPainter p(this);

    if (mModus == modus_keinspiel) {
      /* Kein Spiel */
      
      if (mTitelbild.isNull()) {
	p.setBrush(QColor(150, 150, 150));
	p.setPen(NoPen);
	p.drawRect(r);
      } else {
	p.drawPixmap(0, 0, mTitelbild);
	/* Damit die Schrift das Titelbild nicht strt, nach unten
	   verschieben */
	schrift_mitte = false;
      }

    } else if (papa->spielPause()) {
      /* Spiel steht auf Pause */

      p.setBrush(QColor(150, 150, 150));
      p.setPen(NoPen);
      p.drawRect(r);
      if (papa->mPauseBild) {
	papa->mPauseBild->malBild(p,
				  (gric*grx - papa->mPauseBild->getBreite())/2,
				  (gric*gry - papa->mPauseBild->getHoehe())/2);
	schrift_mitte = false;
      }

    } else {
      /* Whrend des Spiels... */
      /* Achtung! malUpdateIntern() (und damit auch malUpdate() muss
         nicht nur nicht aufgerufen werden,
         sondern darf sogar nicht aufgerufen werden, da whrend einer
         hifea-Animation manuell noch Dinge in mBild gemalt werden. */
         
      p.drawPixmap(0, 0, mBild);
    }

    /* Schrift drbermalen */
    malSchrift(p, schrift_mitte);

  } // QPainter p(this)
}





/** Initialisiert alles fr's Spiel. Schaltet die Spielfeldanzeige an. */
void Spielfeld::startLevel() {
  /* Startmodus: Graue drfen kommen. Da noch keine da sind, kommt ein neues
     Fall */
  mModus = modus_neue_graue;
  mHifea = hifea_nix;
  mKettenreaktion = false;
	
  mRueberReihenModus = rrmodus_nix;
  mHochVerschiebung = 0; // Bild unverschoben
  mGrauAnz = 0; // Anzahl der Grauen, die auf ihr Kommen warten

  /* Spielfeld leeren. */
  mDaten.init();
		
  /* Kein Fallendes */
  mFall.zerstoere();

  /* Eventuelle Restblschen entfernen */
  while (mBlaeschen.removeFirst());
		
  /* Anfangsgraskonfiguration schreiben */

  __String zeile;
  int y = gry - ld->anfangszeilen.count();

  /* Etwas heuristisch dafr sorgen, dass keine gleichen Blops
     nebeneinander sind. Hier sind die Variablen dafr. */
  /* pr = PlusRand */
#define prmax (grx * gry * startlevel_rand_durchgaenge)
  /* Welche Blops sollen noch nachbarentfremdet werden? */
  int prx[prmax], pry[prmax], pranz = 0;


  /* Zeilen durchgehen... */
  for (zeile = ld->anfangszeilen.first(); zeile != 0;
       zeile = ld->anfangszeilen.next(), y++) {
    ASSERT(zeile.length() == grx);
    for (int x = 0; x < grx; x++) {
      char c = zeile.data()[x];

      if (c == '+') {
	mDaten.getFeld(x, y) = Blop(blopart_farbe, rand() % ld->mAnzFarben);
	/* "+"-Blop: Ein paar mal ins Nachbarentfremdungsarray
	   speichern... */
	for (int i = 0; i < startlevel_rand_durchgaenge; i++) {
	  ASSERT(pranz < prmax);
	  prx[pranz] = x;
	  pry[pranz] = y;
	  pranz++;
	}

      } else if (c == '-') {
	mDaten.getFeld(x, y) = Blop(blopart_grau);

      } else if (c != '.') {
	/* Provisorisch! */
	int n = c - 'A';
	ASSERT(n >= 0);
	mDaten.getFeld(x, y) = Blop(blopart_gras, n);
      }
    }
  }

#undef prmax

  /* Hier kommt das eigentliche Nachbarentfremden */
  while (pranz > 0) {
    /* Zuflliges Element aus dem pr-Array rausnehmen */
    int prnr = rand() % pranz;
    int x = prx[prnr];
    int y = pry[prnr];
    pranz--;
    prx[prnr] = prx[pranz];
    pry[prnr] = pry[pranz];

    /* Welche Farbe ist wie oft bei den Nachbarn vertreten? */
    int besetzt[max_farben_zahl];
    for (int i = 0; i < ld->mAnzFarben; i++)
      besetzt[i] = 0;
    /* !! Sehr provisorisch: Nur Nachbarschaftsverhltnisse von
       !! Farbe 0 bercksichtigen!!! */
    for (NachbarIterator ni(&ld->mFarbSorten[0], x, y); ni; ++ni) {
      if (mDaten.koordOK(ni.mX, ni.mY)) {
	int f = mDaten.getFeld(ni.mX, ni.mY).getFarbe();
	if (f != keine_farbe) besetzt[f]++;
      }
    }
			
    /* Wie wenig gleiche Nachbarn knnen wir erreichen? Und mit welchen
	 Farben? */
    int best = 0x7fff;
    int bestanz = 0;
    int bestli[max_farben_zahl];
    for (int j = 0; j < ld->mAnzFarben; j++) {
      if (besetzt[j] < best) {
	best = besetzt[j];
	bestanz = 0;
      }
      if (besetzt[j] == best)
	bestli[bestanz++] = j;
    }
				
    /* OK, jetzt eine dieser Mglichkeiten auswhlen. */
    mDaten.getFeld(x, y) = Blop(blopart_farbe, bestli[rand() % bestanz]);
  }


  /* Levelzeit... */
  mZeit = 0;
  setHetzrandYPix(0);

  malUpdateIntern();
}





/** Ein Schritt vom Spiel; wenn dieser Spieler nicht mitspielt,
    wird nix getan (d. h. egal wie viele Spieler mitspielen,
    spielSchritt() kann man fr alle aufruefen) */
void Spielfeld::spielSchritt() {

  ASSERT(mModus != modus_keinspiel);
  
  /* Whrend hifea-Animationen muss hifeaSchritt() statt
     Spielschritt() aufgerufen werden. */
  ASSERT(mHifea == hifea_nix);

  /* Hetzrand runterbewegen und testen, ob dabei was berdeckt wird. */
  bewegeHetzrand();
  		
  /* Ggf. sich selbst Graue schicken */
  zufallsGraue();
	
  /* Debug-Zeilen: Einfluss der Hoch-Verschiebung austesten... */
  //mHochVerschiebung = ("ABCDCB"[mZeit % 6] - 'A') * 10;
  //setUpdateFastAlles();

  /* Um Fall kmmern. */
  mFall.spielSchritt();
		
  /* Feste Blops animieren.
     da_ist_was_am_platzen wird in modus_flopp bentigt
     (um nach dem Platzen in den nchsten Modus zu schalten) */
  bool da_ist_was_am_platzen = mDaten.animiere();				
	
  /* Um Blschen kmmern */
  bewegeBlaeschen();

  /* Um Rberreihen kmmern */
  rueberReihenSchritt();
	
		
  switch (mModus) {
	
  case modus_testflopp:
    /* Ein guter Zeitpunkt fr's Spielende */
    if (!papa->spielLaeuft()) {
				/* Evtl. Fall platzen lassen */
      if (mFall.existiert()) {
	mFall.lassPlatzen();
	mModus = modus_fallplatz;
      } else
	mModus = modus_warte_auf_stop;
    }
		
    /* Hat jemand einen Platztest bestellt? (Allerdings mit dem Platzen noch
       warten, falls das Fall grade am zerfallen ist.) */
    if (mDaten.getTestPlatz() && !mFall.istEinzel()) {
    
      /* Bevor zum richtigen Platztest kommen, ist hier ein guter Zeitpunkt
         fr das Rochade-hifea. */
      if (ld->mHiddenFeature == hifea_rochade)
        vielleichtRochade();
      
      /* Platzt was? (calcFlopp sendet ggf. auch die Grauen an den Mitspieler) */
      if (calcFlopp(true)) {
	mModus = modus_flopp;
	break;
      }
    }
			
    /* Keine Sofort-Explosion, also keine Kettenreaktion */
    mKettenreaktion = false;
			
    /* OK, nichts will platzen. Wollen wir dann vielleicht neue Graue? */
    if (!mFall.existiert())
      mModus = modus_neue_graue;			
			
    break;
  					
  case modus_flopp: // etwas ist am explodieren
  	
    /* Einfach nur warten, bis die Blops fertig explodiert sind... */	 	
    if (!da_ist_was_am_platzen) {
      /* Hier ist auch ein guter Rochade-Test-Zeitpunkt. (Grade ist
         eine Explosion zu Ende, und wenn nicht gleich rochiert wird, ist
         vielleicht die Lcke zwischendrin schon wieder blockiert.) */
      if (ld->mHiddenFeature == hifea_rochade)
        vielleichtRochade();
      
      /* Wenn wir wieder in Modus testflopp kommen (und sofort eine
	 Explosion auftaucht), ist es eine Kettenreaktion. */
      mModus = modus_rutschnach;
      mKettenreaktion = true;
    }
    break;
  		
  case modus_rutschnach: {
    /* Steine runterrutschen lassen. */
    bool passiert_was = rutschNach(false) != rutschnach_nix;
  		
    /* Ist jetzt (in diesem Schritt) wirklich was passiert? */
    if (!passiert_was) {
      /* Nein. Das Nachrutschen ist also fertig. */
  			
      /* Gute Gelegenheit, um eine Reihe zu bekommen. */
      bekommVielleichtReihe();

      /* Jetzt: Nchste Explosion? */
      mModus = modus_testflopp;
    }		// if (es ist nix mehr nachgerutscht)
    break;
  }
  case modus_neue_graue: {
    /* Graue Steine runterrutschen lassen. */
    int rn = rutschNach(true);
  		
    /* Die Grauen sind weit genug unten, um ein neues Fall loszuschicken */
    if (rn != rutschnach_viel) {

      /* Aber nur, wenn das Spiel noch luft... */
      if (papa->spielLaeuft()) {
  		
	if (mFall.erzeug()) {
	  /* OK, neues Fall unterwegs. Aber sind berhaupt noch Graue unterwegs? */
	  if (rn == rutschnach_wenig)
	    mModus = modus_rutschnach;
	  else
	    mModus = modus_testflopp;
	} else {
	  /* Kein Platz mehr fr neuen Stein => tot */
	  emit sigTot();
	  ASSERT(!papa->spielLaeuft());
	  /* Allerdings knnen wenigstens die Grauen noch fertig runterfallen.
	     Also sind wir noch nicht bereit zum stoppen. */
	}
      } else {
	/* Spiel luft eigentlich gar nicht mehr. Wenn die Grauen ganz auf dem
	   Boden angekommen sind, dann sind wir auch sterbebereit */
	if (rn == rutschnach_nix)
	  mModus = modus_warte_auf_stop;
      }
    }
  		
    break;
  }
  	  			  	
  case modus_fallplatz: // Steine, die am runterfallen waren, zerplatzen 
    // in der Luft, weil das Spiel zu Ende ist
  	
    /* fertig explodiert? */
    if (!mFall.getAmPlatzen()) {
      /* Wir sind jetzt mit der Animation fertig und warten auf spielStop() */
      mFall.zerstoere();
      mModus = modus_warte_auf_stop;
    }
  		
    break;
  		
  case modus_warte_auf_stop:
    /* Wir warten nur darauf, dass wir ein stopLevel() bekommen */
    break;
  default:
    ASSERT(false);
  } // switch mModus
	
  /* Jetzt alle Grafik-Updates, die so angefallen sind, auf einmal machen */
  {
    QPainter p(this);
    malUpdate(p);
  }
}   // spielSchritt()



/** Whrend eine hifea-Animation luft, sollte nicht
    spielSchritt() aufgerufen werden, sondern hifeaSchritt().
    Tut gar nix, wenn's keine hifea-Animation gibt. (Dann
    ist die Animation vermutlich beim anderen Spieler.) */
void Spielfeld::hifeaSchritt() {
  if (mHifea == hifea_nix)
    return;
  
  switch (mHifea) {
  case hifea_rochade: {

    /* Das brauchen wir noch fter: */  
    int tx = mRoGross ? 1 : 8;

    /* Am Anfang: */
    if (mRoZeit == 0) {
      /* Punkte bekommen */
      emit sigBekommPunkte(rechterSpieler, mRoGross ? 150 : 100);

      /* Die Figuren vom Brett nehmen */
      mRoKoenig = mDaten.getFeld(5, mRoY);
      mDaten.getFeld(5, mRoY) = Blop();
      mRoTurm = mDaten.getFeld(tx, mRoY);
      mDaten.getFeld(tx, mRoY) = Blop();      
    }
    
    /* Text an-/aus-blinken? Der Text wird manuell gesetzt, und
       nicht mit setText(), weil keine update-Events erzeugt werden
       sollen. */
    if (mRoZeit % rochade_blink_zeit == 0) {
      if (mRoGross)
        mText = "Big Rochade\n150 Bonus Points";
      else
        mText = "Small Rochade\n100 Bonus Points";
    }
    if (mRoZeit % rochade_blink_zeit == rochade_blink_zeit / 2) {
      mText = "";
    }
    
    
    /* Den Animationsbereich updaten, um das vorige
       Animationsbild zu lschen */
    QPoint p0 = getFeldKoord(mRoGross ? 1 : 5, mRoY);
    setUpdateRect(QRect(p0.x(), p0.y(), gric * (mRoGross ? 5 : 4), gric));
    //for (int x = (mRoGross ? 1 : 5); x <= (mRoGross ? 5 : 8); x++)
    //  mDaten.setUpdateFeld(x, mRoY);
    
    /* Jetzt alles updaten */
    malUpdateIntern();
    
    /* Zustzlich das neue Animationsbild auf mBild malen. */
    { 
      QPainter p(&mBild);
      QPoint p1, p2;
      int t2 = rochade_zeit - mRoZeit;

      /* Turmposition berechnen; und malen. */
      p1 = getFeldKoord(tx, mRoY);
      p2 = getFeldKoord(mRoGross ? 4 : 6, mRoY);
      mRoTurm.malen(p,
        (p1.x() * t2 + p2.x() * mRoZeit) / rochade_zeit,
        p1.y());

      /* Knigsposition berechnen; und malen. */
      p1 = getFeldKoord(5, mRoY);
      p2 = getFeldKoord(mRoGross ? 3 : 7, mRoY);
      mRoKoenig.malen(p,
        (p1.x() * t2 + p2.x() * mRoZeit) / rochade_zeit,
        p1.y());
    }
  
    /* Jetzt kommt's auf den Schirm. */  
    {
      QPainter p(this);
      /* kein malUpdate() oder so aufrufen, da
         malUpdateIntern() ja schon aufgerufen wurde. */
      p.drawPixmap(0, 0, mBild);
      malSchrift(p, true);
    }
    
    mRoZeit++;
    
    /* Zu gegebener Zeit die Animation beenden. */
    if (mRoZeit >= rochade_zeit) {
      mHifea = hifea_nix;
      mDaten.getFeld(mRoGross ? 3 : 7, mRoY) = mRoKoenig;
      mDaten.getFeld(mRoGross ? 4 : 6, mRoY) = mRoTurm;
      
      mText = "";  
    }
    break;
  }
  default:
    ASSERT(false);
  }
}




/** Liefert die Koordinaten eines Felds in Pixeln zurck (ungespiegelt) */
QPoint Spielfeld::getFeldKoord(int x, int y) const {
  return QPoint(x * gric, y * gric - mHochVerschiebung -
		(ld->mSechseck && (x & 1)) * gric / 2);
}





/** Spiel abbrechen (sofort, ohne Animation; oder die Animation ist schon
    vorbei).
    Wenn malen = true ist, Bildschirm sofort updaten */
void Spielfeld::stopLevel(bool malen){
  mModus = modus_keinspiel;
  if (malen)
    repaint(false);
}


/** Bewegt das Fall eins nach links */
void Spielfeld::tasteLinks() {
  mFall.tasteLinks();
}

/** Bewegt das Fall eins nach rechts */
void Spielfeld::tasteRechts() {
  mFall.tasteRechts();
}

/** Dreht das Fall */
void Spielfeld::tasteDreh() {
  mFall.tasteDreh();
}

/** ndert die Fallgeschwindigkeit vom Fall */
void Spielfeld::tasteFall() {
  mFall.tasteFall();
}




/** berechnet, ob und welche Blops platzen mssen. Bei platzen = false
    wird nur den Blops die Kettengre mitgeteilt; sonst platzen die Blops
    evtl., und es werden ggf. auch Graue an den Mitspieler geschickt.
    liefert true, wenn (bei platzen = true) was passiert ist */
bool Spielfeld::calcFlopp(bool platzen) {
  int x, y;
  bool wasPassiert = false;
  /* Die Punkte werden nachher nur gesendet, wenn wirklich was
     explodiert ist. */
  int punkte = mKettenreaktion ? punkte_fuer_kettenreaktion : 0;
	
  /* Anzahl der Grauen fr den anderen Spieler: Wenn die minimal mgliche
     Blop-Zahl platzt, gibt's genau einen Grauen...
     + ggf. welche fr Kettenreaktion */
  int grz = -ld->mPlatzAnzahl + 1;
  if (mKettenreaktion) grz += graue_bei_kettenreaktion;

  /* Array fr Zusammenhangskomponentensuche */
  int flopp[grx][gry];
	
  /* Erst mal muss nix platzen */
  for (x = 0; x < grx; x++)
    for (y = 0; y < gry; y++)
      flopp[x][y] = 0;
			
  /* Spielfeld absuchen */
  for (x = 0; x < grx; x++)
    for (y = 0; y < gry; y++)
      /* Normaler farbiger Blop, der noch nicht am platzen ist? */
      if (mDaten.getFeld(x, y).getArtVerkleidet() == blopart_farbe &&
	  !mDaten.getFeld(x, y).getAmPlatzen()) {
				/* Dann Kettengre berechnen... */
	int anz = calcFloppRec(flopp, x, y,
			       mDaten.getFeld(x, y).getFarbeVerkleidet(),
			       -1);
				
	/* ... und allen Blops in der Kette mitteilen. */
	int neue_pt = calcFloppRec(flopp, x, y,
				   mDaten.getFeld(x, y).getFarbeVerkleidet(),
				   1, /* Gre mitteilen... */
				   platzen && anz >= ld->mPlatzAnzahl, /* Platzen? */
				   mKettenreaktion || !ld->mGrasBeiKettenreaktion, /* Auch Gras? */
				   anz);
	ASSERT(platzen && anz >= ld->mPlatzAnzahl  ||  neue_pt == 0);
				
	/* Wenn's Punkte gab, ist wohl auch was geplatzt */
	if (neue_pt > 0) {
	  punkte += neue_pt;
	  wasPassiert = true;
	  grz += anz;
	}	
      }
	
  ASSERT(!wasPassiert || platzen);
  /* Ist noch was explodiert? */
  if (wasPassiert) {
    /* Dann sende graue */
    emit sigSendeGraue(grz);
    /* und bekomme Punkte */			
    emit sigBekommPunkte(rechterSpieler, punkte);
  }
	
  return wasPassiert;
} // calcFlopp()

/** sucht die Zusammenhangskomponente von x, y mit Farbe n.
    Setzt flopp[][] an den entspr. Stellen auf w.
    - Aufruf mit w = -1 um rauszufinden, wie viele gleichfarbige
      beisammen sind; liefert Anzahl der gleichfarbigen zurck;
    - Danach Aufruf mit w = 1 und richtigem anz-Wert, um den Blops ihre neue
      Kettengre mitzuteilen.
      Wenn platzen = true bergeben wird,
      - platzen die Blops (auch angrenzende
        Graue, und bei auch_gras = true auch angrenzendes Gras).
      - Liefert Anzahl der Punkte zurck, die's dafr gibt.
      - Wenn ntig wird sigGrasGeplatzt() emittet.
    @return Je nach w und platzen: Gre der Zshgskomp. oder Anz. d. Punkte oder 0 */
int Spielfeld::calcFloppRec(int flopp[grx][gry], int x, int y,
			    int n, int w, bool platzen /*= false*/,
			    bool auch_gras /*= false*/, int anz /*= 0*/) {
  int a = mDaten.getFeldArt(x, y);
  /* Die Rberreihe zhlt beim explodieren nicht mit. */
  if (y == gry) a = blopart_ausserhalb;

  if (a == blopart_ausserhalb) // Auerhalb vom Spielfeld?
    return 0;
	
  /* Nicht auerhalb? Dann knnen wir jetzt direkt auf das Blop zugreifen,
     um nochmal genauer nachzufragen...
     (Der a == blopart_ausserhalb-Test reicht noch nicht, weil die Blop-Art
     _ber_ dem Spielfeld blopart_nix ist.) */
  if (mDaten.koordOK(x, y))
    a = mDaten.getFeld(x, y).getArtVerkleidet();
		
  if (flopp[x][y] == w) // Hatten wir dieses Feld schon?
    return 0;
			
  switch (a) {
  case blopart_keins:
    return 0;		
  case blopart_farbe:
    if (mDaten.getFeld(x, y).getFarbeVerkleidet() != n) // Stein hat falsche Farbe
      return 0;
    else {
      int ret = 0;
      flopp[x][y] = w;
      if (w == 1) {
	mDaten.getFeld(x, y).setKettenGroesse(anz);
	if (platzen) {
	  mDaten.getFeld(x, y).lassPlatzen();
	  ret = punkte_fuer_normales;
	}
      } else {
	/* Normalerweise 1... nur z. B. beim Go-Level nicht immer. */
	ret = mDaten.getFeld(x, y).getKettenBeitrag();
      }
      for (NachbarIterator i(&ld->mFarbSorten[n], x, y); i; ++i)
	ret += calcFloppRec(flopp, i.mX, i.mY, n, w, platzen, auch_gras, anz);
      return ret;
    }
  case blopart_grau:
    if (w == -1 || !platzen) {
      return 0;
    } else {
      flopp[x][y] = w;
      mDaten.getFeld(x, y).lassPlatzen();
      return punkte_fuer_graues;
    }
  case blopart_gras:
    if (w == -1 || !platzen || !auch_gras) {
      return 0;
    } else {
      flopp[x][y] = w;
      mDaten.getFeld(x, y).lassPlatzen();
      emit sigGrasGeplatzt();
      return punkte_fuer_gras;
    }	
  }  // switch (a)
  ASSERT(0);
  return 0;  // Um keine Warnung zu bekommen
}




/** Ruft malUpdateIntern() auf und malt das interne Bild dann
    auf p */
void Spielfeld::malUpdate(QPainter & p) {
  malUpdateIntern();
  /* Jetzt auf den Bildschirm klatschen (oder was auch immer p ist): */
  p.drawPixmap(0, 0, mBild);
}



/** Hier findet die Grafik statt: Alles was in den mUpdateXXX-
    Variablen angegeben ist, wird auf den internen Puffer neu
    gezeichnet. */
void Spielfeld::malUpdateIntern() {

  /* Provisorisch: Update-Daten aus mDaten holen */
  for (int x = 0; x < grx; x++)
    for (int y = 0; y <= gry; y++)
      if (mDaten.mUpdaten[x][y]) {
	mDaten.mUpdaten[x][y] = false;
	QPoint pos = getFeldKoord(x, y);
	setUpdateRect(pos.x(), pos.y());
      }

  /* Erst mal alles offscreen malen */
  { 
    QPainter p(&mBild);
  
    /* Hintergrund updaten */

    // Gut, dass mUpdateRegion schon gespiegelt ist...
    p.setClipRegion(mUpdateRegion);
    
    p.setBrush(ld->mHintergrundFarbe);
    p.setPen(NoPen);
    p.drawRect(0, 0, width(), height());
    if (ld->mMitHintergrundbildchen) {
      ld->bhintergrund.malBild(p, 0,
			       ld->mSpiegeln ? 0 :
			       height() - ld->bhintergrund.getHoehe());
    }

    /* Clipping funktioniert fr die Bildchen eh nicht
       (wg. bitBlt statt drawPixmap). Also ausschalten. */
    p.setClipping(false);
    
    /* Blschen updaten */
    for (BlaeschenDaten * bd = mBlaeschen.first();bd != 0;
	 bd = mBlaeschen.next()) {
      if (bd->mUpdate) {
	bd->mUpdate = false;
	bd->malen(p);
	/* Beim Malen des Blschens knnten Dinge bermalt
	   worden sein. Die neu malen: */
	setUpdateRect(bd->x, bd->y, gric, gric, ebene_blops);
      }
    }
    
    /* Blops updaten */
    int y0 = mHetzrandYPix / gric; // oberste sichtbare Reihe
    for (int x = 0; x < grx; x++)
      for (int y = y0; y < mDaten.getGrY(); y++) // Evtl. auch Rberreihe
	if (mUpdateDaten[x][y]) {
	  mUpdateDaten[x][y] = false;
	  
	  QPoint pos = getFeldKoord(x, y);
	  mDaten.getFeld(x, y).malen(p, pos.x(), pos.y());
          
          /* Im Schach-Level mssen jetzt evtl. noch Springer-Verbindungs-
             Linien (neu) gemalt werden: */
          for (int i = 0; i < 8; i++) {
            int x1 = x + i["01120000"] - '1';
            int y1 = y + i["00000112"] - '1';
            int x2 = x + i["12012222"] - '1';
            int y2 = y + i["22221201"] - '1';
            int fa1 = mDaten.getFeldArt(x1, y1);
            int fa2 = mDaten.getFeldArt(x2, y2);
            if (fa1 == blopart_ausserhalb || fa1 == blopart_keins)
              goto keine_springerverbindung;
            if (fa2 == blopart_ausserhalb || fa2 == blopart_keins)
              goto keine_springerverbindung;
            {
              Blop & b = mDaten.getFeld(x1, y1);
              if (!b.verbindetMit(mDaten.getFeld(x2, y2)))
                goto keine_springerverbindung;
              b.getSorte()->malSpringerStrich(p, pos.x(), pos.y(),
                    i["15260437"] - '0');
            }
         keine_springerverbindung:;
          } // Ende Springerverbindungen

          
  	  /* Beim Malen des Blops knnte das Fall oder der Hetzrand
	     bermalt worden sein. Ggf. neu malen: */
	  setUpdateRect(pos.x(), pos.y(), gric, gric, ebene_fall);
	}

    /* Fallendes updaten */
    if (mUpdateFall) {
      mUpdateFall = false;
      mFall.malen(p);

      /* Beim Malen des Falls knnte der Hetzrand
	 bermalt worden sein. Ggf. neu malen: */
      setUpdateFall(ebene_hetzrand);
    }
    
    /* Hetzrand updaten */
    if (mUpdateHetzrand) {
      mUpdateHetzrand = false;

      /* Scheint etwas schneller zu gehen, wenn man hier nicht clippt. */
      //p.setClipRegion(mUpdateRegion);

      /* Der nichtbild-Hetzrand */
      p.setBrush(ld->hetzrandFarbe);
      p.setPen(NoPen);
      p.drawRect(spiegelRect(QRect(0, 0, width(), mHetzrandYPix)));
      
      /* Der Bild-Hetzrand */
      if (ld->mMitHetzbildchen) {
	ld->bhetz.malBild(p, 0,
			  ld->mSpiegeln ?
			  height() - mHetzrandYPix - ld->mHetzrandUeberlapp :
			  mHetzrandYPix + ld->mHetzrandUeberlapp - ld->bhetz.getHoehe()
			  );
      }

      //p.setClipping(false);
    }	
    
    
    /* Update-Region lschen. Darf erst hier am Ende passieren, weil
       sie zwischendrin nochmal vergrtert werden knnte. */
    mUpdateRegion = QRegion();

  } // QPainter p(&mBild)
  /* Jetzt enhlt mBild den richtigen Bildschirminhalt... */  
} // malUpdateIntern






/** Graue von anderem Spieler bekommen; wird ignoriert, falls dieser
    Spieler grad nicht spielt */
void Spielfeld::empfangeGraue(int g){
  if (mModus == modus_keinspiel)
    return;
  mGrauAnz += g;
}




/** liefert die Hhe vom hchsten Trmchen... unter der Annahme, dass
		er schon komplett zusammengesackt ist. */
int Spielfeld::getHoehe(){
  int h = 0;
  for (int x = 0; x < grx; x++) {
    int nh = 0;
    for (int y = 0; y < gry; y++)
      nh += mDaten.getFeldArt(x, y) != blopart_keins;
    if (nh > h) h = nh;
  }
	
  return h;		
}










/** sollte nur aufgerufen werden, wenn papa->spielLaeuft()
    false liefert; liefert true, wenn alle Spiel-Stop-
    Animationen fertig sind; liefert brigens auch true,
    wenn dieser Spieler gar nicht mitspielt */
bool Spielfeld::bereitZumStoppen(){
  if (mModus == modus_keinspiel || mModus == modus_warte_auf_stop)
    return true;
  else
    return false;
}




/** ndert die Hhe vom Hetzrand auf y (in Pixeln). */
void Spielfeld::setHetzrandYPix(int y) {
  if (y == mHetzrandYPix)
    return;
	
  int vy = mHetzrandYPix;
  mHetzrandYPix = y;
	
  /* Wichtig: setUpdateRect darf erst aufgerufen werden,
     _nachdem_ mHetzrandYPix  gendert wurde (weil sonst
     der Hetzrand evtl. noch gar nicht in dem
     bergebenen Rechteck drin liegt und deshalb kein
     frisch-malen-Flag kriegt. */
  if (ld->mMitHetzbildchen)
    setUpdateRect(0, vy - ld->bhetz.getHoehe() + ld->mHetzrandUeberlapp,
		  gric * grx, ld->bhetz.getHoehe() + y - vy);
  else
    setUpdateRect(0, vy, gric * grx, y - vy);
}

/** Zeigt t gro an. */
void Spielfeld::setText(__String t, bool sofort_updaten /*= true*/) {
  mText = t;
  if (sofort_updaten)
    repaint(false);
  else
    update();
}

/** Wenn der Level gespiegelt ist, wird auch r gespiegelt */
QRect Spielfeld::spiegelRect(const QRect & r){
  if (ld->mSpiegeln) {
    // bottom = top + height - 1 ... schade eigentlich.
    return QRect(r.left(), gry * gric - (r.bottom() + 1), r.width(), r.height());
  } else
    return r;
}

/** Setzt das Rechteck r auf upzudaten. Der ebene-Parameter sollte
    (im Moment) nur von malUpdate() benutzt werden. */
void Spielfeld::setUpdateRect(QRect r, int ebene /*= ebene_hintergrund*/) {

  /* Erst mal die Update-Regionen updaten. */
  QRegion nreg(spiegelRect(r));
  if (mUpdateRegion.isNull())
    mUpdateRegion = nreg;
  else
    mUpdateRegion = mUpdateRegion.unite(nreg);
  
  /* Und jetzt noch die speziellen update-Variablen setzen */
	
  if (ebene <= ebene_blaeschen) {
    /* Blschen */
    for (BlaeschenDaten * bd = mBlaeschen.first();
	 bd != 0; bd = mBlaeschen.next()) {
      if (r.intersects(QRect(bd->x, bd->y, gric, gric)))
	bd->mUpdate = true;
    }
  }
	
  /* Fr die Daten und das Fallende das Rechteck vorher verschieben... */
  QRect r2 = r;
  r2.moveBy(0, mHochVerschiebung);
  r2 = r2.intersect(QRect(0, 0, grx * gric, gry * gric));
	
  if (ebene <= ebene_blops) {
    /* Blops */
    for (int x = 0; x < grx; x++)
      for (int y = 0; y <= gry; y++) {
	QPoint pos = getFeldKoord(x, y);
	if (r.intersects(QRect(pos.x(), pos.y(), gric, gric)))
	  mUpdateDaten[x][y] = true;
      }
  }
	
  if (ebene <= ebene_fall) {
    /* Fallendes */
    if (mFall.existiert() && r2.intersects(mFall.getRect()))
      mUpdateFall = true;
  }

  if (ebene <= ebene_hetzrand) {
    /* Hetzrand */
    if (r.top() < mHetzrandYPix + ld->mHetzrandUeberlapp)
      mUpdateHetzrand = true;
  }
}

/** Setzt das Rechteck x, y, w, h auf upzudaten. */
void Spielfeld::setUpdateRect(int x, int y, int w /*= gric*/, int h /*= gric*/,
			      int ebene /*= ebene_hintergrund*/) {
  setUpdateRect(QRect(x, y, w, h), ebene);
}

/** setzt den Bereich, wo sich das
    Blschen befindet, auf upzudaten */
void Spielfeld::setUpdateBlaeschen(BlaeschenDaten * bd) {
  setUpdateRect(bd->x, bd->y);
}

/** setzt den Bereich vom Fallenden auf upzudaten */
void Spielfeld::setUpdateFall(int ebene /*= ebene_hintergrund*/) {
  if (mFall.existiert())
    setUpdateRect(mFall.getRect(), ebene);
}

/** Setzt alles auer den Hetzrand auf upzudaten. */
void Spielfeld::setUpdateFastAlles() {
  setUpdateRect(0, mHetzrandYPix, grx * gric, gry * gric - mHetzrandYPix);
}

/** Lsst den Hetzrand schnell runterkommen (fr die
		Zeitbonus-Animation). Liefert true, wenn fertig. */
bool Spielfeld::bonusSchritt() {

  ASSERT(mModus != modus_keinspiel);
	
  /* Wenn der Hetzrand schon weiter unten ist, als wir ihn runterkommen
     lassen wrden, gar nix tun. */
  int unten = gric * gry - ld->mHetzrandUeberlapp;
  if (mHetzrandYPix > unten)
    return true;

  /* Hetzrand runterkommen lassen */
  int ny = mHetzrandYPix + bonus_geschwindigkeit;
  if (ny > unten) ny = unten;

  setHetzrandYPix(ny);
	
  /* Ausgeben... */
  malUpdateIntern();
  /*{
    QPainter p(this);
    malUpdate(p);
    malSchrift(p);
    } // QPainter p(this)*/
  /* Nein, nicht ausgeben. Das ndern des Textes whrend der
     Bonus-Animation fhrt sowieso zu einem Paint-Event */
		
  /* Hetzrand unten angekommen? */
  return ny == unten;
}

/** Malt die Schrift auf den Bildschirm; genauer: auf p.
    In der richtigen Farbe. */
void Spielfeld::malSchrift(QPainter & p, bool mitte /*= true*/) {
	
  p.setPen(mModus == modus_keinspiel ? black : ld->schriftFarbe);
  
  p.setFont(QFont("times", 18, QFont::Bold, TRUE));
  p.drawText(rect(),
	     WordBreak | AlignHCenter |
	     (mitte ? AlignVCenter : AlignBottom),
	     mText);
}

/** Liefert true, wenn grade ein Fallendes unterwegs ist.
    Wird vom KIPlayer bentigt */
bool Spielfeld::getFallModus() {
  //return mModus == modus_fall;
  return mFall.existiert();
}

/** Liefert einen Pointer auf das Blopgitter zurck. */
/*const*/ BlopGitter * Spielfeld::getDatenPtr() /*const*/ {
  return &mDaten;
}

/** Liefert einen Pointer auf die fallenden Blops zurck.
    Wird von KIPlayer einmal am Anfang aufgerufen. */
const Blop * Spielfeld::getFallPtr() {
  return mFall.getBlopPtr();
}

/** Liefert die Pos. zurck, an der neue Dinge oben
    auftauchen. */
int Spielfeld::getHetzrandYAuftauch() const{
  /* Siehe def. der Konstante... */
  return (mHetzrandYPix + hetzrand_dy_auftauch) / gric;
}

/** Liefert die Pos. zurck, bis wohin noch Dinge liegen
    drfen, ohne dass man tot ist. */
int Spielfeld::getHetzrandYErlaubt() const{
  /* Siehe def. der Konstante... */
  return (mHetzrandYPix + hetzrand_dy_erlaubt) / gric;
}

/** Bewegt den Hetzrand eins nach unten. Testet auch, ob dabei
    was berdeckt wird. */
void Spielfeld::bewegeHetzrand(){
  /* Hat der Hetzrand Steine berdeckt? (Bruchte eigentlich
     nur getestet werden, wenn sich
     getHetzrandYErlaubt() seit dem letzten Check erhht hat...) */
  int hye = getHetzrandYErlaubt();
  if (hye > 0) {
    for (int x = 0; x < grx; x++)
      if (mDaten.getFeldArt(x, hye - 1) != blopart_keins) {
	/* Hetzrand berdeckt Stein => tot */
	emit sigTot();
	ASSERT(!papa->spielLaeuft());
	/* Evtl. muss auch bei uns noch eine Animation fertig
	   ablaufen; deshalb noch nicht gleich den mModus auf
	   modus_warte_auf_stop setzen, sondern
	   einfach die Dinge laufen lassen */
	break;
      }
  }
  /* Levelzeit hochzhlen, Hetzrand kommen lassen...; das darf
     erst geschehen,  nachdem getestet wurde, ob was berdeckt
     wurde, weil vielleicht vorher grade
     was in der Hhe aufgetaucht ist, wo es jetzt berdeckt wrde */
  mZeit++;
  setHetzrandYPix(mZeit / ld->hetzrandZeit);
}
/** Sendet sich selbst (ggf.) zufllige Graue */
void Spielfeld::zufallsGraue(){
  /* Ggf. sich selbst Graue schicken */
  if (ld->mZufallsGraue != zufallsgraue_keine)
    if (rand() % ld->mZufallsGraue == 0)
      empfangeGraue(1);
}

/** Bewegt ggf. die Blschen */
void Spielfeld::bewegeBlaeschen(){
  if (ld->mMitBlaeschen) {
    if (rand() % blaeschen_haeufigkeit == 0) {
      /* Der Konstruktor von Blaeschendaten wrfelt die Position aus */
      mBlaeschen.insert(0, new BlaeschenDaten());
    }
		
    /* Blschen, die unterwegs sind, bewegen */
    QListIterator<BlaeschenDaten> i(mBlaeschen), j(mBlaeschen);
    for (; i.current();) {
      /* An alter Pos. lschen */
      setUpdateBlaeschen(i.current());
      i.current()->animiere();
      /* Wenn das Blschen gelscht werden muss, mssen wir
	 aufpassen, dass wir
	 mit der for-Schleife nicht durcheinander kommen:
	 Erst i hochzhlen, dann
	 lschen. (Sonst wird beim Lschen i _vielleicht_
	 schon hochgesetzt) */
      j = i;
      ++i;
      if (j.current()->y + gric <= getHetzrandYErlaubt())
	mBlaeschen.remove(j);
      else {
	/* Blschen existiert noch. Also auch an neuer Stelle updaten */
	setUpdateBlaeschen(j.current());
      }
    } // for i
		
  }
}

/** Lsst in der Luft hngende Blops ein Stck runterfallen.
    Liefert zurck, ob sich nichts bewegt hat, nur unten oder auch oben.
    Bei auchGraue = true, kommen auch Graue, die ganz ber
    dem Spielfeld hngen. */
int Spielfeld::rutschNach(bool auchGraue){
  int ret = rutschnach_nix;
		
  /* Anzahl der Spalten, in denen Platz fr ein neues Graues ist */
  int grauplatz = 0;

  /* Zeile, in der neue Graue auftauchen */
  int hya = getHetzrandYAuftauch();
			
  /* Steine nachrutschen lassen */
  for (int x = 0; x < grx; x++) {
    int y = gry - 1;
			
    /* Zunchst mal die Steine, die schon im Spielfeld sind, runterrutschen
       lassen; die knnen brigens auch noch leicht ber dem Hetzrand sein
       (also sicherheitshalber bis ganz oben gehen) */
    while (y >= 0 && mDaten.getFeldArt(x, y) != blopart_keins)
      y--; // suche unterstes leeres

    if (y >= 0) {	// wenn es leere Felder gab, dann nachrutschen
      for (; y > 0; y--) {
	if (mDaten.getFeldArt(x, y - 1) != blopart_keins) {
	  mDaten.verschiebBlop(x, y - 1, x, y);
	  if (y > hya + neues_fall_platz)
	    ret |= rutschnach_wenig;
	  else
	    ret |= rutschnach_viel;
	}
      }
				
    } // Ende von da ist noch Platz in der Spalte
			
    /* Ist in dieser Spalte Platz fr ein Graues? */
    if (mDaten.getFeld(x, hya).getArt() == blopart_keins)
      grauplatz++;
  } // for x

  /* Drfen neue Graue kommen? */
  if (auchGraue) {
    for (int x = 0; x < grx; x++) {
      /* Ist in Grauauftauchhhe der Spalte Platz? (Achtung:
	 Spalte voll ist nicht der einzige Grund fr keinen
	 Grauplatz. Der Hetzrand knnte auch
	 grade ein Feld runtergekommen sein.) */
      if (mDaten.getFeldArt(x, hya) == blopart_keins) {
	ASSERT(grauplatz > 0);
	if (rand() % grauplatz < mGrauAnz) {
	  mDaten.getFeld(x, hya) = Blop(blopart_grau);
	  ret |= rutschnach_viel;
	  mGrauAnz--;
	}
	grauplatz--;
      }
    }
  }
	
  return ret;
}
/** Liefert die Pos. vom Hetzrand in Pixeln zurck. Wird
    vom Fall gebraucht. */
int Spielfeld::getHetzrandYPix() const {
  return mHetzrandYPix;
}




/** Kmmert sich um hin- und hergeben von Reihen. */
void Spielfeld::rueberReihenSchritt() {
  switch (mRueberReihenModus) {
  case rrmodus_nix:
    break;
			
  case rrmodus_gib_rueber:
    /* Nichts tun. Nur warten, bis uns der andere die Reihe ganz
       weggenommen hat. */
    if (mRestRueberReihe == 0)
      mRueberReihenModus = rrmodus_gib_runter;

    /*
    if (rechterSpieler) {
      if (mDaten.getFeldArt(0, gry) == blopart_keins)
	mRueberReihenModus = rrmodus_gib_runter;
    } else {
      if (mDaten.getFeldArt(grx - 1, gry) == blopart_keins)
	mRueberReihenModus = rrmodus_gib_runter;
	}*/
  		
    break;
  case rrmodus_gib_runter:
    /* alles zum updaten markieren */
    setUpdateFastAlles();
  		
    /* Bild verschieben */
    mHochVerschiebung -= reihe_rueber_senkrecht_pixel;
  	
    /* Weit genug verschoben? */
    if (mHochVerschiebung <= 0) {
      mHochVerschiebung = 0;
  	  	
      /* Rueberreihe vernichten */
      mDaten.setRueberReihe(false);
  	  	
      /* und fertig */
      mRueberReihenModus = rrmodus_nix;
    }
    break;
  case rrmodus_bekomm_hoch:
	
    /* alles zum updaten markieren */
    setUpdateFastAlles();
  			 	
    /* Bild verschieben */
    mHochVerschiebung += reihe_rueber_senkrecht_pixel;
  	
    /* Weit genug verschoben? */
    if (mHochVerschiebung >= gric) {
      mHochVerschiebung = gric;
  	  	  	  	
      mRueberReihenModus = rrmodus_bekomm_rueber;
      mRestRueberReihe = grx;
    }
    break;
			
  case rrmodus_bekomm_rueber:
    /* Neuen Stein vom anderen Spieler bestellen */
    Blop neuer;
    emit sigWillStein(neuer);
  		
    /* Von welcher Seite kommt die Reihe? */
    if (rechterSpieler) {
      /* von links: Reihe nach rechts schieben */
      for (int x = grx - 1; x > 0; x--)
	mDaten.verschiebBlop(x - 1, gry, x, gry);
      mDaten.getFeld(0, gry) = neuer;
  			
    } else {
      /* von rechts: Reihe nach links schieben */
      for (int x = 0; x < grx - 1; x++)
	mDaten.verschiebBlop(x + 1, gry, x, gry);
      mDaten.getFeld(grx - 1, gry) = neuer;
    }

    mRestRueberReihe--;

    /* Fertig? */
    if (mRestRueberReihe == 0) {
      mRueberReihenModus = rrmodus_nix;

      /* Im Datenarray alles nach oben schieben. */
      for (int x = 0; x < grx; x++)
	for (int y = 0; y < gry; y++)
	  mDaten.verschiebBlop(x, y + 1, x, y);
      mDaten.setRueberReihe(false);
      /* Optisch soll sich nix ndern; jetzt ist das Spielfeld wieder in
	 Normalposition */
      mHochVerschiebung = 0;
      	
      /* Es wird erwartet, dass der normale Modus jetzt automatisch in
	 testflopp bergeht... */  			
    }
    break;
  }    // switch (mRueberReihenModus)
}

/** liefert (in ret) zurck, ob wir dem anderen
    Spieler eine Reihe geben (er hat Hhe h) */
void Spielfeld::gebReihe(int h, bool & ret){
  ret = false;

  /* Wenn wir nicht mitspielen, geben wir natrlich auch keine Reihe her */	
  if (mModus == modus_keinspiel)
    return;
	
  /* Wenn das Spiel zu Ende gehen soll, keine Reihe mehr rbergeben */
  if (!papa->spielLaeuft())
    return;
		
  /* Wenn wir schon irgendwie mit einer Reihe beschftigt sind, geben wir
     auch keine her. */
  if (mRueberReihenModus != rrmodus_nix)
    return;
		
  /* Sind wir hinreichend hher als der andere? Oder testen wir im
     Debug-Modus das Reihen hin und her schieben? */
  if (getHoehe() >= h + 2 || papa->mRueberReihenTest) {
		
    /* Ist unsere unterste Reihe (komplett) verfgbar? */
    ret = true;
    for (int x = 0; x < grx; x++)
      if (mDaten.getFeldArt(x, gry - 1) == blopart_keins ||
	  mDaten.getFeld(x, gry - 1).getAmPlatzen())
	ret = false;
  }
	
  /* Wenn wir dem anderen Spieler eine Reihe geben wollen, dann
     machen wir uns auch mal dafr bereit */
  if (ret) {
    mRueberReihenModus = rrmodus_gib_rueber;
    mRestRueberReihe = grx;
    
    /* Im Datenarray alles nach unten schieben. */
    for (int x = 0; x < grx; x++)
      for (int y = gry; y > 0; y--)
	mDaten.verschiebBlop(x, y - 1, x, y);
    mDaten.setRueberReihe(true);
    /* Optisch soll sich noch nix ndern... */
    mHochVerschiebung = gric;
    
    /* Hier fehlt noch ein Signal an die Steine der neuen untersten Reihe,
       um ihnen mitzuteilen, dass ihre Zshgskomponente kleiner geworden ist. */

    /* brigens: Das Reihe-Rbergeben fngt (fr uns)
       noch nicht gleich an; erst, wenn der andere
       sein Zeug hochgeschoben hat. */
  }
}


/** gibt einen Stein an den anderen Spieler
    rber; in s wird die Steinfarbe zurckgeliefert */
void Spielfeld::gebStein(Blop & s){
  ASSERT(mRueberReihenModus == rrmodus_gib_rueber);
  ASSERT(mRestRueberReihe > 0);
	
  if (rechterSpieler) {
    /* nach links... */
    s = mDaten.getFeld(0, gry);
    for (int x = 0; x < grx - 1; x++)
      mDaten.verschiebBlop(x + 1, gry, x, gry/*, blopart_wirklich_keins*/);
  } else {
    /* nach rechts... */
    s = mDaten.getFeld(grx - 1, gry);
    for (int x = grx - 1; x > 0; x--)
      mDaten.verschiebBlop(x - 1, gry, x, gry/*, blopart_wirklich_keins*/);
  }

  mRestRueberReihe--;
}


/** Prft, ob ein Reihenbekommen sinnvoll wre und initiiert es ggf.
    (Unterhlt sich auch mit dem anderen Spieler */
void Spielfeld::bekommVielleichtReihe() {
  /* Knnen wir eine Reihe vertragen? Dann mal
     beim anderen Spieler anfragen, ob wir eine bekommen */
  if (mRueberReihenModus == rrmodus_nix) {
    bool erg;
    emit sigWillReihe(getHoehe(), erg);
    if (erg) {
      mDaten.setRueberReihe(true);
      mRueberReihenModus = rrmodus_bekomm_hoch;
    }
  }
}



/** Liefert die (rberreihenbedingte) Hochverschiebung
    des gesamten Spielfelds. Wird vom Fall bentigt (um
    seine Koordinaten in Feldern zu berechnen). */
int Spielfeld::getHochVerschiebung(){
  return mHochVerschiebung;
}


/** Liefert true, wenn grade eine hifea-Animation luft */
bool Spielfeld::getHifea() {
  return mHifea != hifea_nix;
}


/** Fr hifea_rochade... */
void Spielfeld::vielleichtRochade() {
  /* Versionen (beim Gras): 0 = Springer, 1 = Lufer,
     2 = Turm, 3 = Bauer, 4 = Knig, 5 = Dame */
  for (int y = 0; y < gry; y++) {
    /* Knig in richtiger Spalte? */
    const Blop & bk = mDaten.getFeld(5, y);
    if (bk.getArt() == blopart_gras && bk.getVersion() == 4) {
      /* Ja, Knig da. */
            
      /* Groe Rochade? */
      const Blop & bt1 = mDaten.getFeld(1, y);
      if (bt1.getArt() != blopart_gras || bt1.getVersion() != 2)
        goto keine_grosse_rochade;
      for (int x = 2; x < 5; x++)
        if (mDaten.getFeldArt(x, y) != blopart_keins)
          goto keine_grosse_rochade;
      /* OK, groe Rochade */
      mHifea = hifea_rochade;
      mRoY = y;
      mRoGross = true;
      mRoZeit = 0;
      return;
      
      keine_grosse_rochade:;

      /* Kleine Rochade? */
      const Blop & bt2 = mDaten.getFeld(8, y);
      if (bt2.getArt() != blopart_gras || bt2.getVersion() != 2)
        goto keine_kleine_rochade;
      for (int x = 6; x < 8; x++)
        if (mDaten.getFeldArt(x, y) != blopart_keins)
          goto keine_kleine_rochade;
      /* OK, kleine Rochade */
      mHifea = hifea_rochade;
      mRoY = y;
      mRoGross = false;
      mRoZeit = 0;
      return;
      
      keine_kleine_rochade:;

    }
  }
}



