/* Player.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002-2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import "Format.h"
#import "GeneralPreference.h"
#import "Output.h"
#import "Preference.h"

#define PLAYER_M 1
#import "Player.h"

#define BUFFER_SIZE 176400

static NSNotificationCenter *nc = nil;

NSString *PlayerPlayingNotification = @"PlayerPlayingNotification";
NSString *PlayerStoppedNotification = @"PlayerStoppedNotification";
NSString *PlayerPausedNotification = @"PlayerPausedNotification";
NSString *PlayerResumedNotification = @"PlayerResumedNotification";
NSString *PlayerSongEndedNotification = @"PlayerSongEndedNotification";

@implementation Player : NSObject

+ (void) initialize
{
  nc = [NSNotificationCenter defaultCenter];
}

- (id) init
{
  if ((self = [super init]))
    {
      output = nil;
      stream = nil;
      delegate = nil;
      muted = NO;
      paused = NO;
      playing = NO;
      awaitingNewStream = NO;
      rate = 0;
      channels = 0;
    }

  return self;
}

- (void) setDelegate: (id) anObject
{
  if (delegate)
    [nc removeObserver: delegate name: nil object: self];

  delegate = anObject;

  if ([delegate respondsToSelector: @selector (playerPlaying:)])
    [nc addObserver: delegate
	selector: @selector (playerPlaying:)
	name: PlayerPlayingNotification
	object: self];
  if ([delegate respondsToSelector: @selector (playerStopped:)])
    [nc addObserver: delegate
	selector: @selector (playerStopped:)
	name: PlayerStoppedNotification
	object: self];
  if ([delegate respondsToSelector: @selector (playerPaused:)])
    [nc addObserver: delegate
	selector: @selector (playerPaused:)
	name: PlayerPausedNotification
	object: self];
  if ([delegate respondsToSelector: @selector (playerResumed:)])
    [nc addObserver: delegate
	selector: @selector (playerResumed:)
	name: PlayerResumedNotification
	object: self];
  if ([delegate respondsToSelector: @selector (playerSongEnded:)])
    [nc addObserver: delegate
	selector: @selector (playerSongEnded:)
	name: PlayerSongEndedNotification
	object: self];
}

- (id) delegate
{
  return delegate;
}

- (void) _playLoopIteration
{
  char buffer[BUFFER_SIZE];
  int size;
  NSData *streamChunk;

  if (muted)
    {
      memset (buffer, 0, BUFFER_SIZE);
      size = BUFFER_SIZE;
    }
  else
    size = [stream readNextChunk: buffer withSize: BUFFER_SIZE];

  if (size > 0)
    {
      totalBytes += size;
      streamChunk = [NSData dataWithBytes: buffer length: size];
      [output playChunk: streamChunk];
    }
  else
    {
      awaitingNewStream = YES;
      [nc postNotificationName: PlayerSongEndedNotification object: self];
    }
}

- (void) _reInitOutputIfNeeded
{
  unsigned int newChannels;
  unsigned long newRate;

  newChannels = [stream readChannels];
  newRate = [stream readRate];
  if (rate != newRate
      || channels != newChannels)
    {
      if ([output prepareDeviceWithChannels: newChannels andRate: newRate])
        {
          rate = newRate;
          channels = newChannels;
        }
      else
        NSLog (@"error preparing output for %d channels at a rate of %d",
               newChannels, newRate);
    }
}

- (void) setStream: (id <Format>) newStream
{
  totalBytes = 0;

  if (stream)
    {
      [stream streamClose];
      [stream release];
    }
  stream = newStream;
  if (stream)
    {
      [stream retain];
      if (output)
        [self _reInitOutputIfNeeded];
      if (awaitingNewStream)
        {
          [self _playLoopIteration];
          awaitingNewStream = NO;
        }
    }
  else
    {
      /* FIXME: the PlaylistController (or the PlayerController?) should
         manage the validity of the songs it sends to the Player. The current
         solution is not clean. */
      if (awaitingNewStream)
        [nc postNotificationName: PlayerSongEndedNotification object: self];
    }
}

- (int) timer
{
  return totalBytes / (rate * channels * 2);
}

- (BOOL) playing
{
  return playing;
}

- (void) _ensureOutput
{
  GeneralPreference *generalPreference;
  Class outputClass;

  generalPreference = [GeneralPreference instance];
  outputClass = [generalPreference preferredOutputClass];
  if (output && [output class] != outputClass)
    {
      [output release];
      output = nil;
      rate = 0;
      channels = 0;
      [self _reInitOutputIfNeeded];
    }

  if (!output)
    {
      output = [outputClass new];
      [output setParentPlayer: self];
      [self _reInitOutputIfNeeded];
    }
}

- (void) play
{
  [self _ensureOutput];

  if ([output openDevice])
    {
      playing = YES;
      [self _playLoopIteration];
      [nc postNotificationName: PlayerPlayingNotification
          object: self];
    }
}

- (void) stop
{
  [output closeDevice];
  [stream streamClose];
  [stream release];
  stream = nil;
  playing = NO;
  paused = NO;
  awaitingNewStream = NO;
  [nc postNotificationName: PlayerStoppedNotification object: self];
}

- (void) setPaused: (BOOL) aBool
{
  if (!paused && aBool)
    {
      paused = YES;
      [nc postNotificationName: PlayerPausedNotification object: self];
    }
  else if (paused && !aBool)
    {
      paused = NO;
      [self _playLoopIteration];
      [nc postNotificationName: PlayerResumedNotification object: self];
    }
}

- (BOOL) paused
{
  return paused;
}

- (void) setMuted: (BOOL) aBool
{
  muted = aBool;
}

- (BOOL) muted
{
  return muted;
}

- (void) seek: (unsigned int) seconds
{
  [stream seek: seconds];
  totalBytes = seconds * rate * channels * 2;
}

- (void) chunkFinishedPlaying
{
  if (playing && !paused)
    [self _playLoopIteration];
}

@end
