/* Copyright (C) 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <gtk--/main.h>
#include <gtk--/pixmap.h>
#include <gdk/gdkkeysyms.h> // the key codes are here

#include "addressbook.h"
#include "dialogs.h"
#include "addressbook_icons.h"

#define ADDRESS_FILE ".efax-gtk_addressbook"
#define ADDRESS_DIVIDER 3

int AddressBook::is_address_list = 0;

AddressBook::AddressBook(const int size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), standard_size(size),
			     parent(window), in_run_loop(false),
			     table(2, 2, false), list_table(5, 2, false),
			     ok_button("OK"), cancel_button("Cancel"),
                             address_list(2) {

  // notify the existence of this object in case I later decide to use this dialog as modeless
  // by omitting the set_modal() call below
  is_address_list++;

  address_list_scroll_window.set_policy(GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
  // use Gtk::ScrolledWindow::add() not Gtk::ScrolledWindow::add_with_viewport()
  // to prevent the vertical scroll scrolling the titles
  address_list_scroll_window.add(address_list);

  address_list_scroll_window.set_usize(standard_size * 13, standard_size * 6);

  // set up the address list
  {
    using namespace Gtk::CList_Helpers;

    address_list.set_column_title(0, "Name");
    address_list.set_column_title(1, "Number");
    address_list.column_title_passive(0);
    address_list.column_title_passive(1);
    address_list.column_titles_show();
    address_list.set_column_visibility(0, true);
    address_list.set_column_visibility(1, true);

    // get the row list
    read_list();

    address_list.set_selection_mode(GTK_SELECTION_SINGLE);
  }

  Gtk::Pixmap* addIcon_p = manage(new Gtk::Pixmap(add_xpm));
  add_button.add(*addIcon_p);

  Gtk::Pixmap* deleteIcon_p = manage(new Gtk::Pixmap(delete_xpm));
  delete_button.add(*deleteIcon_p);

  Gtk::Pixmap* upIcon_p = manage(new Gtk::Pixmap(up_arrow_xpm));
  up_button.add(*upIcon_p);

  Gtk::Pixmap* downIcon_p = manage(new Gtk::Pixmap(down_arrow_xpm));
  down_button.add(*downIcon_p);

  Gtk::Label* dummy_p = manage(new Gtk::Label);

  list_table.attach(add_button, 0, 1, 0, 1,
		    0, 0, standard_size/2, 0);
  list_table.attach(delete_button, 0, 1, 1, 2,
		    0, 0, standard_size/2, 0);
  list_table.attach(up_button, 0, 1, 2, 3,
		    0, 0, standard_size/2, 0);
  list_table.attach(down_button, 0, 1, 3, 4,
		    0, 0, standard_size/2, 0);
  list_table.attach(*dummy_p, 0, 1, 4, 5,
		    0, GTK_EXPAND, 0, 0);
  list_table.attach(address_list_scroll_window, 1, 2, 0, 5, GTK_EXPAND | GTK_FILL,
         GTK_EXPAND | GTK_FILL, 0, 0);

  tooltips.set_tip(add_button, "Add new address", 0);
  tooltips.set_tip(delete_button, "Delete address", 0);
  tooltips.set_tip(up_button, "Move address up", 0);
  tooltips.set_tip(down_button, "Move address down", 0);

  table.attach(list_table, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
         GTK_EXPAND | GTK_FILL, standard_size/3, standard_size/3);
  table.attach(ok_button, 0, 1, 1, 2, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);
  table.attach(cancel_button, 1, 2, 1, 2, GTK_EXPAND,
	 0, standard_size/3, standard_size/3);

  add_button.clicked.connect(SigC::slot(this, &AddressBook::add_address_prompt));
  delete_button.clicked.connect(SigC::slot(this, &AddressBook::delete_address_prompt));
  ok_button.clicked.connect(SigC::slot(this, &AddressBook::ok_slot));
  ok_button.set_usize(standard_size * 3, standard_size);
  cancel_button.clicked.connect(SigC::slot(this, &AddressBook::finish));
  cancel_button.set_usize(standard_size * 3, standard_size);
  up_button.clicked.connect(SigC::slot(this, &AddressBook::move_up));
  down_button.clicked.connect(SigC::slot(this, &AddressBook::move_down));

  table.set_border_width(standard_size/3);

  set_title("efax-gtk: Address book");
  set_position(GTK_WIN_POS_CENTER);
  add(table);


  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);
  set_position(GTK_WIN_POS_CENTER);

  show_all();
}

AddressBook::~AddressBook(void) {
  // notify the destruction of this object
  is_address_list--;
}

string AddressBook::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
  return result;
}

void AddressBook::ok_slot(void) {
  result = get_number();
  if (!result.empty()) {
    accepted(result);
    finish();
  }
  else beep();
}

void AddressBook::finish(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint AddressBook::delete_event_impl(GdkEventAny*) {
  finish();
  return true; // returning true prevents destroy sig being emitted
}

string AddressBook::get_number(void) {

  using namespace Gtk::CList_Helpers;

  string number;
  SelectionList selection = address_list.selection();
  if (!selection.empty()) {
    number = selection.front()[1].get_text();
  }
  return number;
}

void AddressBook::add_address_prompt(void) {

  AddressDialog* dialog_p = new AddressDialog(standard_size, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in AddressBook::add_address_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &AddressBook::add_address));
  // there is no memory leak -- AddressDailog will delete its own memory
  // when it is closed
}

void AddressBook::add_address(const vector<string>& address) {

  using namespace Gtk::CList_Helpers;

  // add the new address by adding it to the RowList for the address_list object
  address_list.rows().push_back(address);
  save_list();
  address_list.columns_autosize();
}

void AddressBook::delete_address_prompt(void) {

  using namespace Gtk::CList_Helpers;

  SelectionList selection = address_list.selection();
  if (!selection.empty()) {
    SelectionList::iterator iter = selection.begin();
    PromptDialog* dialog_p = new PromptDialog("Delete selected address?", "Delete address", standard_size, *this);
    if (!dialog_p) {
      cerr << "Memory allocation error in AddressBook::delete_address_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &AddressBook::delete_address), iter));
    // there is no memory leak -- the memory will be deleted when PromptDialog closes
  }
  else beep();
}

void AddressBook::delete_address(Gtk::CList_Helpers::SelectionList::iterator iter) {

  // delete the address by removing it from the RowList for the address_list object
  address_list.rows().remove(*iter);
  save_list();
  address_list.columns_autosize();
}

void AddressBook::read_list(void) {

  string filename(prog_config.homedir);
  filename += "/" ADDRESS_FILE;

#ifdef HAVE_IOS_NOCREATE
  ifstream filein(filename.c_str(), ios::in | ios::nocreate);
#else
  // we must have Std C++ so we probably don't need a ios::nocreate
  // flag on a read open to ensure uniqueness
  ifstream filein(filename.c_str(), ios::in);
#endif

  if (filein) {
    
    using namespace Gtk::CList_Helpers;

    // get the row list
    RowList& row_list = address_list.rows();
    row_list.clear();

    string line;
    vector<string> list_entry;
    while (getline(filein, line)) {
      if (!line.empty()) {
	string::size_type pos = line.find(ADDRESS_DIVIDER, 0); // pos now is set to end of name value
	list_entry.push_back(line.substr(0, pos));
	pos++; // pos now is set to the beginning of the number value
	list_entry.push_back(line.substr(pos, line.size() - pos));
	row_list.push_back(list_entry);
	list_entry.clear();
      }
    }
    address_list.columns_autosize();
  }
}

void AddressBook::save_list(void) {

  string filename(prog_config.homedir);
  filename += "/" ADDRESS_FILE;
  ofstream fileout(filename.c_str(), ios::out);

  if (fileout) {
    
    using namespace Gtk::CList_Helpers;

    string line;
    RowList& row_list = address_list.rows();
    RowList::iterator iter;
    for (iter = row_list.begin(); iter != row_list.end(); ++iter) {
      line = (*iter)[0].get_text();
      line += ADDRESS_DIVIDER;
      line += (*iter)[1].get_text();
      line += '\n';
      fileout << line;
    }
  }
}

void AddressBook::move_up(void) {

  using namespace Gtk::CList_Helpers;

  SelectionList selection = address_list.selection();
  if (!selection.empty()) {
    gint row_num = selection.front().get_row_num();
    if (row_num > 0) {  // not first item!
      address_list.rows().swap(address_list.row(row_num), address_list.row(row_num - 1));
      save_list();
    }
    else beep();
  }
  else beep();
}

void AddressBook::move_down(void) {

  using namespace Gtk::CList_Helpers;

  SelectionList selection = address_list.selection();
  if (!selection.empty()) {
    gint row_num = selection.front().get_row_num();
    if (row_num < static_cast<gint>(address_list.rows().size() - 1)) {  // not last item!
      address_list.rows().swap(address_list.row(row_num), address_list.row(row_num + 1));
      save_list();
    }
    else beep();
  }
  else beep();
}

AddressDialog::AddressDialog(const int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("OK"), cancel_button("Cancel"),
			     name_label("Name:"), number_label("Number:"),
			     table(3, 2, false), parent(window) {

  name_label.set_usize(standard_size * 2, standard_size);
  number_label.set_usize(standard_size * 2, standard_size);

  name_entry.set_usize(standard_size * 8, standard_size);
  number_entry.set_usize(standard_size * 8, standard_size);

  name_box.pack_start(name_label, false, false, standard_size/2);
  name_box.pack_start(name_entry, true, true, 0);

  number_box.pack_start(number_label, false, false, standard_size/2);
  number_box.pack_start(number_entry, true, true, 0);

  ok_button.set_usize(standard_size * 3, standard_size);
  cancel_button.set_usize(standard_size * 3, standard_size);

  table.attach(name_box, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(number_box, 0, 2, 1, 2, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(cancel_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &AddressDialog::selected), true));
  cancel_button.clicked.connect(SigC::bind(SigC::slot(this, &AddressDialog::selected), false));

  add(table);
  
  set_title("Add Address");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  set_border_width(standard_size/2);

  name_entry.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void AddressDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void AddressDialog::selected(bool accept) {
  // check pre-conditions to closing the dialog
  if (accept && (!name_entry.get_text_length() || !number_entry.get_text_length())) beep();
  
  else {   
    parent.set_sensitive(true); // do this before we emit accepted()
    hide_all();
    if (accept) {
      vector<string> out_val;
      out_val.push_back(name_entry.get_text());
      out_val.push_back(number_entry.get_text());
      accepted(out_val);
    }
    if (in_run_loop) Gtk::Main::quit();
    // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
    else delete this;
  }
}

gint AddressDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint AddressDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  else if (event_p->keyval == GDK_Return && !cancel_button.has_focus()) selected(true);
  else Gtk::Window::key_press_event_impl(event_p);
  return true; // processing ends here
}
