/* Copyright (C) 2001 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>

#include <iostream>
#include <strstream>
#include <fstream>
#include <iomanip>
#include <ctime>
#include <cstring>
#include <cstdlib>

#include "efax_controller.h"

static pid_t child_pid;

static volatile sig_atomic_t state = EfaxController::inactive;
static volatile sig_atomic_t change_state_flag = false;


EfaxController::EfaxController(void) {
  // set up state_messages

  state_messages.push_back("Inactive");
  state_messages.push_back("Sending fax");
  state_messages.push_back("Answering call");
  state_messages.push_back("Answering call");
  state_messages.push_back("Standing by to receive calls");
  state_messages.push_back("");
  state_messages.push_back("");
  state_messages.push_back("");
  state_messages.push_back("");
}

void EfaxController::efax_closedown(void) {
  if (state != inactive) {
    state = closedown;
    kill_child();
    while (state != inactive) usleep(500000);
    if (*prog_config.receive_dirname) {
      // now delete prog_config.receive_dirname if it is empty
      string full_dirname;
      if (!prog_config.homedir.empty()) {
	full_dirname = prog_config.homedir;
	full_dirname += "/faxin/";
      }
      full_dirname += prog_config.receive_dirname;
      rmdir(full_dirname.c_str()); // only deletes the directory if it is empty

      *prog_config.receive_dirname = 0;
    }
  }
  unjoin_child();
}

int EfaxController::get_state(void) {
  return state;
}

void EfaxController::sendfax(string& file, string& number) {

  if (state != inactive) beep();

  else {
    stdout_pipe.open(Pipe_fifo::non_block);
    // efax is sensitive to timing, so set pipe write to non-block also
    stdout_pipe.make_write_non_block();
    
    state = sending;
    last_file_sent = file;
    sentfax_number = number;

    child_pid = fork();

    if (child_pid == -1) {
      write_error("Fork error - exiting\n");
      exit(FORK_ERROR);
    }
    if (!child_pid) {  // child process - as soon as everything is set up we are going to do an exec()

      // now we have forked, we can connect stdout_pipe to stdout
      // and connect MainWindow::error_pipe to stderr
      stdout_pipe.connect_to_stdout();
      connect_to_stderr();

      // now fill the copy of prog_config.parms in this process with the remaining parameters
      // (this will not change the copy maintained by the parent process)
     
      struct tm* time_p;
      time_t time_count;

      time(&time_count);
      time_p = localtime(&time_count);

      char date_string[23];
      const char format[] = "%a %d-%b-%Y %H:%M";
      strftime(date_string, 23, format, time_p);

      string temp("-h");
      temp += date_string;
      temp += "    From ";
      temp += prog_config.my_name + " (";
      temp += prog_config.my_number + ") --> ";
      temp += number + "    p.%d/%d";
      prog_config.parms.push_back(temp);
      temp = "-t";
      if (prog_config.tone_dial) temp += 'T';
      else temp += 'P';
      temp += number;
      prog_config.parms.push_back(temp);

      vector<string> result = make_fax(file);

      if (!result.empty()) {
	vector<string>::iterator iter;
	for (iter = result.begin(); iter != result.end(); ++iter) {
	  prog_config.parms.push_back(*iter);
	}
	char** exec_parms = new char*[prog_config.parms.size() + 1]; // this does not create a leak
                                                                     // it will be deleted by the system
                                                                     // when the child process terminates
	char**  temp_pp = exec_parms;
	for (iter = prog_config.parms.begin(); iter != prog_config.parms.end(); ++iter, ++temp_pp) {
	  *temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                         // it will be deleted by the system
                                                 // when the child process terminates
	  strcpy(*temp_pp, iter->c_str());
	}

	*temp_pp = 0;
	execvp("efax", exec_parms);

	// if we reached this point, then the execvp() call must have failed
	write_error("Can't find the efax program - please check your installation\n"
	            "and the PATH environmental variable\n");
      }
      // this child process must end here - use _exit() not exit()
      usleep(200000);
      _exit(EXEC_ERROR); 
    } // end of child process

    // this is the parent process
    stdout_pipe.make_readonly();   // since the pipe is unidirectional, we can close the write fd
    
    join_child();
    change_state_flag = true;
  }
}

vector<string> EfaxController::make_fax(string filename) {
  // convert the postscript file into tiffg3 fax files, beginning at [filename].001
  // we will use ghostscript

  vector<string> vec; // this is the return value

  string::size_type pos = filename.find_last_of('/');

  if (pos == string::npos || pos + 2 > filename.size()) {
    write_error("Not valid file name\n");
  }
    
  else {

    pid_t pid = fork();

    if (pid == -1) {
      write_error("Fork error\n");
      _exit(FORK_ERROR); // we have already forked, so use _exit() not exit()
    }
    if (!pid) {  // child process - when everything is set up, we are going to do an exec()

      // unfortunately ghostscript does not handle long file names
      // so we need to separate the file name from the full path (we will chdir() to the directory later)
      // pos already is set to the position of the last '/' character
      string dirname(filename.substr(0, pos));
      pos++;
      string basename(filename.substr(pos));

      vector<string> parms;
      string temp;
      parms.push_back("gs");
      parms.push_back("-q");
      parms.push_back("-sDEVICE=tiffg3");
      temp = "-r";
      temp += prog_config.resolution;
      parms.push_back(temp);
      parms.push_back("-dNOPAUSE");
      parms.push_back("-dSAFER");
      temp = "-sOutputFile=";
      temp += basename + ".%03d";
      parms.push_back(temp);
      temp = "-sPAPERSIZE=";
      temp += prog_config.page_size;
      parms.push_back(temp);
      parms.push_back(basename);
      
      char** exec_parms = new char*[parms.size() + 1]; // this does not create a leak
                                                       // it will be deleted by the system
                                                       // when the child process terminates

      vector<string>::iterator iter;
      char**  temp_pp = exec_parms;
      for (iter = parms.begin(); iter != parms.end(); ++iter, ++temp_pp) {
	*temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                       // it will be deleted by the system
                                               // when the child process terminates
	strcpy(*temp_pp, iter->c_str());
      }

      *temp_pp = 0;

      // now start up ghostscript
      // first we need to connect stdin to /dev/null to make ghostscript terminate
      int fd = open("/dev/null", O_RDWR);
      close(0);
      dup(fd);
      // now close stdout
      close(1);
      dup(fd);
      close(fd); // now stdin and stdout read/write to /dev/null, we can close the /dev/null file descriptor

      // unfortunately ghostscript does not handle long file names
      // so we need to chdir()
      chdir(dirname.c_str());
      execvp("gs", exec_parms);

      // if we reached this point, then the execvp() call must have failed
      write_error("Can't find the ghostscript program - please check your installation\n"
		  "and the PATH environmental variable\n");
      // this child process must end here - use _exit() not exit()
      usleep(200000);
      _exit(EXEC_ERROR); 
    } // end of child process

    // this is the parent process
    // wait until ghostscript has produced the fax tiffg3 fax file

    wait(0);

    int partnumber = 1;
    ostrstream strm;
    strm << filename.c_str() << '.' << setfill('0') << setw(3) << partnumber << ends;
    const char* test_name = strm.str();
    int result = access(test_name, R_OK);
    
    while (!result) {  // file OK
      vec.push_back(test_name);
      delete[] test_name;
      
      partnumber++;
      ostrstream strm;
      strm << filename.c_str() << '.' << setfill('0') << setw(3) << partnumber << ends;
      test_name = strm.str();
      result = access(test_name, R_OK);
    }
    delete[] test_name;
    
    if (vec.empty()) write_error("Not valid postscript file\n");
  }

  return vec;
}

void EfaxController::receive(int mode) {

  if (state != inactive) beep();

  else {
    // get a time value to create the directory name into which the fax is to be saved
    // and to insert in prog_config.receive_dirname for parent and child
    struct tm* time_p;
    time_t time_count;
    
    time(&time_count);
    time_p = localtime(&time_count);

    string fax_pathname;
    if (!prog_config.homedir.empty()) {
      fax_pathname = prog_config.homedir;
      fax_pathname += "/faxin/";
    }
    else {
      write_error("You don't have the $HOME environmental variable set.\n"
		  "You may need to search for the directory in which\n"
		  "the received fax is saved, and it probably won't\n"
		  "appear in the fax list!\n");
    }

    const char format[] = "%y%m%d%H%M%S";
    strftime(prog_config.receive_dirname, MAX_TEMP_NAME + 1, format, time_p);
    string temp(fax_pathname);
    temp += prog_config.receive_dirname;

    // check whether directory already exists or can't be created
    int count;
    for (count = 0; count < 4 && mkdir(temp.c_str(), S_IRUSR | S_IWUSR | S_IXUSR); count++) {
      sleep(1); // wait a second to get a different time
      time(&time_count);
      time_p = localtime(&time_count);
      strftime(prog_config.receive_dirname, MAX_TEMP_NAME + 1, format, time_p);
      temp = fax_pathname + prog_config.receive_dirname;
    }
    if (count == 4) {
      write_error("Can't create directory to save fax\n");
      *prog_config.receive_dirname = 0;
      return;
    }

    fax_pathname += prog_config.receive_dirname;

    stdout_pipe.open(Pipe_fifo::non_block);
    // efax is sensitive to timing, so set pipe write to non-block also
    stdout_pipe.make_write_non_block();
    
    state = mode;

    child_pid = fork();
    if (child_pid == -1) {
      write_error("Fork error - exiting\n");
      exit(FORK_ERROR);
    }
    if (!child_pid) {  // child process - as soon as everything is set up we are going to do an exec()

      // now we have forked, we can connect stdout_pipe to stdout
      // and connect MainWindow::error_pipe to stderr
      stdout_pipe.connect_to_stdout();
      connect_to_stderr();

      // now fill the copy of prog_config.parms in this process with the remaining parameters
      // (this will not change the copy maintained by the parent process)

      string temp("-r");
      temp += prog_config.receive_dirname;
      prog_config.parms.push_back(temp);

      if (mode == receive_takeover) prog_config.parms.push_back("-w");
      
      else if (mode == receive_standby) {
	temp = "-jS0=";
	temp += prog_config.rings;
	prog_config.parms.push_back(temp);
	prog_config.parms.push_back("-s");
	prog_config.parms.push_back("-w");
      }
      
      char** exec_parms = new char*[prog_config.parms.size() + 1]; // this does not create a leak
                                                            // it will be deleted by the system
                                                            // when the child process terminates

      vector<string>::iterator iter;
      char**  temp_pp = exec_parms;
      for (iter = prog_config.parms.begin(); iter != prog_config.parms.end(); ++iter, ++temp_pp) {
	*temp_pp = new char[iter->size() + 1]; // this does not create a leak
	                                       // it will be deleted by the system
                                               // when the child process terminates
	strcpy(*temp_pp, iter->c_str());
      }

      *temp_pp = 0;

      chdir(fax_pathname.c_str());

      // now start up efax in receive mode
      execvp("efax", exec_parms);

      // if we reached this point, then the execvp() call must have failed
      write_error("Can't find the efax program - please check your installation\n"
		  "and the PATH environmental variable\n");
      // this child process must end here - use _exit() not exit()
      usleep(200000);
      _exit(EXEC_ERROR); 
    } // end of child process

    // this is the parent process
    stdout_pipe.make_readonly();   // since the pipe is unidirectional, we can close the write fd

    join_child();
    change_state_flag = true;
  }
}

void EfaxController::read_pipe_slot(gint fd, GdkInputCondition) {
  char pipe_buffer[PIPE_BUF + 1];
  ssize_t result;

  if (fd == stdout_pipe.get_read_fd()) {
    while ((result = stdout_pipe.read(pipe_buffer, PIPE_BUF)) > 0) {
      pipe_buffer[result] = 0;
      stdout_message(pipe_buffer);
    }
  }
}

void EfaxController::stop_slot(void) {
 if (state != inactive && state != close_child_connections) {      // else we need to deal with it here
   stdout_message("\n*** Stopping send/receive session ***\n\n");
   kill_child();
 }
  else beep();
}

void EfaxController::timer_event(void) {

  if (change_state_flag) { // we only really need to test change_state_flag to trigger
    // a call to display_state(), but we need to test here at the outset to make sure
    // that timer_event() is atomic for all further tests below, as all the variables
    // being tested can be changed by efax_controller_childexit_handler()

    if (state == close_child_connections || state == restart_standby) {
    
      if (*prog_config.receive_dirname) {
	// now delete receive_dirname if it is empty
	string full_dirname;
	if (!prog_config.homedir.empty()) {
	  full_dirname = prog_config.homedir;
	  full_dirname += "/faxin/";
	}
	full_dirname += prog_config.receive_dirname;
	rmdir(full_dirname.c_str()); // only deletes the directory if it is empty
	
	*prog_config.receive_dirname = 0;
      }

      unjoin_child();
      int state_val = state;
      state = inactive; // this is needed even if we are going to call receive()
      // now restart if in standby mode
      if (state_val == restart_standby) receive(receive_standby);
    }

    else if (state == send_success) {
      save_sent_fax();
      unjoin_child();
      state = inactive;
    }

    change_state_flag = false;
    display_state();
  }
}

void EfaxController::display_state(void) {
  write_state(state_messages[state].c_str());
}

void EfaxController::join_child(void) {
  stdout_connection = Gtk::Main::input.connect(SigC::slot(this, &EfaxController::read_pipe_slot),
					       stdout_pipe.get_read_fd(), GDK_INPUT_READ);
}

void EfaxController::unjoin_child(void) {
  stdout_pipe.close();
  stdout_connection.disconnect();
}

void EfaxController::kill_child(void) {
  if (child_pid) kill(child_pid, SIGTERM);
  usleep(500000);               // now really make sure
  if (child_pid) kill(child_pid, SIGKILL);
}

void EfaxController::save_sent_fax(void) {
  
  string::size_type pos = last_file_sent.find_last_of('/');
  if (pos == string::npos || pos + 2 > last_file_sent.size()) {
    write_error("Not valid file name to save -- can't save sent fax\n");
  }

  else {
    // extract the base file name
    pos++;
    // get the file base name for later use in making the fax description
    const string file_basename(last_file_sent.substr(pos));

    // get a time value to create the directory name into which the fax is to be saved
    struct tm* time_p;
    time_t time_count;
    
    time(&time_count);
    time_p = localtime(&time_count);

    // now create the directory into which the fax files to be saved
    string fileout_name;
    if (!prog_config.homedir.empty()) {
      fileout_name = prog_config.homedir;
      fileout_name += "/faxsent/";
    }
    else {
      write_error("You don't have the $HOME environmental variable set.\n"
		  "You may need to search for the directory in which\n"
		  "the fax is saved, and it probably won't appear in\n"
		  "the fax list!\n");
    }

    const char dirname_format[] = "%y%m%d%H%M%S";
    char top_dirname[MAX_TEMP_NAME + 1];
    strftime(top_dirname, MAX_TEMP_NAME + 1, dirname_format, time_p);
    string temp(fileout_name);
    temp += top_dirname;

    // check whether directory already exists or can't be created
    int count;
    for (count = 0; count < 4 && mkdir(temp.c_str(), S_IRUSR | S_IWUSR | S_IXUSR); count++) {
      sleep(1); // wait a second to get a different time
      time(&time_count);
      time_p = localtime(&time_count);
      strftime(top_dirname, MAX_TEMP_NAME + 1, dirname_format, time_p);
      temp = fileout_name + top_dirname;
    }
    if (count == 4) {
      write_error("Can't create directory to save fax\n");
      return;
    }

    // now make fileout_name the same as the directory name for the saved faxes
    // that we have just created
    fileout_name += top_dirname;
    const string fileout_dirname = fileout_name; // keep for later to enter a description

    // and now complete the unsuffixed file name of the destination files
    fileout_name += '/';
    fileout_name += top_dirname;

    // now create a string containing the date for the fax description
    // glibc has const struct tm* as last param of strftime()
    // but the const does not appear to be guaranteed by POSIX
    // so do localtime() again just in case
    time_p = localtime(&time_count);
    const char date_description_format[] = "(%H%M %Z %d %b %Y)";
    const int max_description_datesize = 126;
    char date_description[max_description_datesize];
    strftime(date_description, max_description_datesize, date_description_format, time_p);

    // make the suffix
    int partnumber = 1;
    ostrstream strm;
    strm << '.' << setfill('0') << setw(3) << partnumber << ends;
    const char* suffix = strm.str();

    // make the suffixed source and destination files
    string suffixed_inname = last_file_sent + suffix;
    string suffixed_outname = fileout_name + suffix;

    delete[] suffix;

    ifstream filein;
    ofstream fileout;
    const int BLOCKSIZE = 1024;
    char block[BLOCKSIZE];

    while (!access(suffixed_inname.c_str(), R_OK)
	   && (filein.open(suffixed_inname.c_str(), ios::in), filein) // use comma operator to check filein
	   && (fileout.open(suffixed_outname.c_str(), ios::out), fileout)) { // ditto for fileout
      while (filein) {
	filein.read(block, BLOCKSIZE);
	fileout.write(block, filein.gcount());
      }
      filein.close();
      filein.clear();
      fileout.close();
      fileout.clear();
      
      partnumber++;
      ostrstream strm;
      strm << '.' << setfill('0') << setw(3) << partnumber << ends;
      suffix = strm.str();
      suffixed_inname = last_file_sent + suffix;
      suffixed_outname = fileout_name + suffix;
      
      delete[] suffix;
    }
    fileout.clear();
    if (partnumber < 2) write_error("There was a problem saving the sent fax\n");
    else {
      const string description_filename(fileout_dirname + "/Description");
      fileout.open(description_filename.c_str(), ios::out);
      if (fileout) fileout << file_basename << " --> " << sentfax_number << ' ' << date_description;
    }
  }
}

bool efax_controller_childexit_handler(pid_t pid, int exit_code) {
  if (pid == child_pid) {
    child_pid = 0;
    unlink(prog_config.lock_file.c_str());  // this won't work if efax is suid root and we are not root
    if (state == EfaxController::closedown) // tell EfaxController::efax_closedown() the child is now dead
      state = EfaxController::inactive;
    else if (state == EfaxController::receive_standby
	     && (!exit_code || exit_code == 3)) // get EfaxController::timer_event() to restart efax()
      state = EfaxController::restart_standby;
    else if (state == EfaxController::sending
	     && !exit_code)            // get EfaxController::timer_event() to save the file in sent fax list
      state = EfaxController::send_success;
    else if (state != EfaxController::inactive) // get EfaxController::timer_event() to close dead file descriptors and dead SigC connections
      state = EfaxController::close_child_connections;

    change_state_flag = true;
    return true;
  }
  return false;
}
