# -*- coding: utf-8 -*-
# Copyright (C) 2007 by Benjamin Kampmann <big.master.ben@web.de>
# This code is published under MIT licence
# All rights reserved.

"""
Gnome Screen Saver ServiceProvider component class

This Component can block the GnomeScreenSaver in different modes. That
is very useful, if you have a remote control or other input device
which is not reconized by the Screensaver on its own.

This Compontent blocks with this usage also the PowerManager of GNOME,
so that the sleep/Screen procedures shouldn't happen.

"""

__maintainer__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

from elisa.base_components.service_provider import ServiceProvider
from elisa.core import common, component
from elisa.core.player import PlayerPlaying, PlayerStopping

import dbus
if getattr(dbus, 'version', (0,0,0)) >= (0,41,0):
    import dbus.glib
from dbus.exceptions import DBusException

class GnomeScreensaverService(ServiceProvider):
    """
    This class connect to the Gnome Screen Saver and blocks it
    """

    config_doc = {'blocking_mode' : 'Block the Screensaver. Available modes'
                                    ' are: \n * 1 : block on playing only\n'
                                    ' * 2 : block from the start to the end,'
                                    ' use this, if you are only using remotes,'
                                    ' on which the screensaver is not'
                                    ' reacting'
                                    ' * else: do not block!\n'}

    default_config = {'blocking_mode' : 2} 

    def initialize(self):
        ServiceProvider.initialize(self)        
        self.pm_iface = None
        self._pm_cookie = None

        try:
            self.ses = dbus.SessionBus()
        except DBusException, error:
            msg = "DBus is not running."
            raise component.InitializeFailure(self.name, msg)

        try:
            # Even if there is also a org.gnome.PowerManager, the
            # ScreenSaver is for our use the better one
            self.pm = self.ses.get_object('org.gnome.ScreenSaver',
                                          '/org/gnome/Screensaver')
            self.pm_iface = dbus.Interface(self.pm, "org.gnome.ScreenSaver")
        except Exception, e:
            msg = "Could not connect to the Gnome ScreenSaver: %s" % e
            raise component.InitializeFailure(self.name, msg)
        
        blocking_mode = self.config.get('blocking_mode', 1)
        if blocking_mode == 1:
            self.info("Using the stop-on-playing-only mode")
            bus = common.application.bus
            bus.register(self._player_started, PlayerPlaying)
            bus.register(self._player_stopped, PlayerStopping)
        elif blocking_mode == 2:
            self.info("Using full-block mode")
            self._pm_cookie = self.pm_iface.Inhibit('elisa', 'running')

    def clean(self):
        if self._pm_cookie:
            self.pm_iface.UnInhibit(self._pm_cookie)
        ServiceProvider.clean(self)

    def _player_started(self, msg, sender):
        self.debug("Asking for Inhibition of PowerManager")
        self._pm_cookie = self.pm_iface.Inhibit('elisa', 'playing media') 

    def _player_stopped(self, msg, sender):
        self.debug("Asking for UnInhibition of PowerManager")
        self.pm_iface.UnInhibit(self._pm_cookie)
        self._pm_cookie = None
