/* Emcast - Endhost multicast utility
 * Copyright (C) 2001  The Regents of the University of Michigan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <signal.h>
#include <string.h>
#include <assert.h>

#include "libemcast.h"
#include "util.h"

static void usage (char* prog, int exitval);
static int  parse_and_set_opt (char* name, char* val);
static void sig_int_cb (int ignore);

#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif

#define BUFLEN 8192	/* Default buffer length */

#define EERROR(STR, NUM) do { \
  fprintf (stderr, "emcast error: %s: %s\n", STR, emcast_strerror(NUM));\
  if (emcast) emcast_leave (emcast); \
  exit (EXIT_FAILURE); } while (0)

#define PERROR(STR) do { \
  perror (STR); \
  if (emcast) emcast_leave (emcast); \
  exit (EXIT_FAILURE); } while(0)

static Emcast* emcast = NULL;

int
main (int argc, char* argv[])
{
  int rv;
  extern char* optarg;
  extern int   optind;
  char c;
  int i;

  char* url;
  int emcastfd;
  char* buf;
  int buflen = BUFLEN;
  int quiet = 0;
  int l = 0;
  int loopback = 0;
  int ttl = -1;
  fd_set read_fdset;
  int size;

  /*   extern char* malloc_options; */
  /*   malloc_options = "A"; */

  while ((c = getopt(argc, argv, "b:lo:O:qt:h")) != -1) 
    {
      switch (c) 
	{
	case 'b':	
	  buflen = atoi(optarg);  
	  break;
	  
	case 'l':
	  loopback = 1;
	  break;

	case 'o':
	case 'O':
	  if (!argv[optind])
	    usage (argv[0], EXIT_FAILURE);

	  optind++;
	  break;

	case 'q':
	  quiet = 1;
	  break;

	case 't':
	  ttl = atoi(optarg);
	  break;

	case 'h':
	case '?':
	  usage (argv[0], EXIT_SUCCESS);
	  break;
	}
    }

  if (argc == optind || buflen <= 0)
    {
      usage (argv[0], EXIT_FAILURE);
    }


  url = argv[optind];
  buf = (char*) malloc (buflen);

  /* On sig int, leave group */
  signal (SIGINT, sig_int_cb);

  /* Create a new Emcast */
  rv = emcast_new (&emcast, url);
  if (rv == EMCAST_ENOPROTO)
    {	
      fprintf (stderr, "Can't find protocol handler for %s\n", url);	
      exit (EXIT_FAILURE);
    }
  else if (rv < 0)
    EERROR ("new", rv);

  /* Parse any pre-join options */
  for (i = 1; argv[i]; ++i)
    {
      if (argv[i][0] == '-' && argv[i][1] == 'O')
	{
	  char* name;
	  char* val;

	  /* Get name */
	  if (argv[i][2])
	    name = &argv[i][2];
	  else
	    name = argv[1 + i++];
	  val = argv[i+1];

	  if (parse_and_set_opt (name, val))
	    fprintf (stderr, "%s: set %s failed: %s\n", 
		     argv[0], name, emcast_strerror(rv));
	  i++;
	}
    }

  /* Join group */
  emcastfd = emcast_join (emcast, url);
  if (emcastfd < 0)
    EERROR ("join", emcastfd);

  /* Turn loopback off */
  l = 0;
  rv = emcast_setopt (emcast, "loopback", &l, sizeof(l));
  if (rv != 0)
    fprintf (stderr, "%s: set loopback failed: %s\n", argv[0], emcast_strerror(rv));

  /* Turn loopback on if the user specified it */
  if (loopback)
    {
      loopback = htonl (loopback);
      rv = emcast_setopt (emcast, "loopback", &loopback, sizeof(loopback));
      if (rv != 0)
	fprintf (stderr, "%s: set loopback failed: %s\n", argv[0], emcast_strerror(rv));
      loopback = ntohl (loopback);
    }

  /* Set TTL */
  if (ttl != -1)
    {
      ttl = htonl (ttl);
      size = sizeof(ttl);
      rv = emcast_setopt (emcast, "ttl", &ttl, sizeof(ttl));
      if (rv != 0)
	fprintf (stderr, "%s: set TTL failed: %s\n", argv[0], emcast_strerror(rv));
      ttl = ntohl (ttl);
    }

  /* Parse any other post-join options */
  for (i = 1; argv[i]; ++i)
    {
      if (argv[i][0] == '-' && argv[i][1] == 'o')
	{
	  char* name;
	  char* val;

	  /* Get name */
	  if (argv[i][2])
	    name = &argv[i][2];
	  else
	    name = argv[1 + i++];
	  val = argv[i+1];

	  if (parse_and_set_opt (name, val))
	    fprintf (stderr, "%s: set %s failed: %s\n", 
		     argv[0], name, emcast_strerror(rv));
	  i++;
	}
    }

  while (1)
    {
      FD_ZERO (&read_fdset);
      if (!quiet)
	FD_SET (STDIN_FILENO, &read_fdset);
      FD_SET (emcastfd, &read_fdset);

      rv = select (MAX(STDIN_FILENO, emcastfd) + 1, 
		   &read_fdset, NULL, NULL, NULL);

      if (rv == -1)
	PERROR ("select failed");

      if (FD_ISSET(STDIN_FILENO, &read_fdset))
	{
	  assert (!quiet);

	  /* Read from stdin */
	  rv = read (STDIN_FILENO, buf, buflen);
	  if (rv == -1)
	    PERROR ("read failed");
	  else if (rv == 0)
	    break;

	  /* Send to group */
	  rv = emcast_send (emcast, buf, rv);
	  if (rv < 0)
	    EERROR ("send", rv);
	}

      if (FD_ISSET(emcastfd, &read_fdset))
	{
	  int read;
	  int written;

	  /* Read from emcast */
	  read = emcast_recv (emcast, buf, buflen);
	  if (read < 0)
	    {
	      if (read == EMCAST_EBADARG)
		fprintf (stderr, "emcast warning: buffer too small for recv\n");
	      else
		EERROR ("recv", read);
	    }

	  /* Write to stdout if not quiet */
	  if (!quiet)
	    {
	      written = writen (STDOUT_FILENO, buf, read);
	      if (written < 0)
		PERROR ("write failed");
	    }
	}
    }

  /* Leave the group */
  rv = emcast_leave (emcast);
  emcast = NULL;
  if (rv != 0)
    EERROR ("leave", rv);

  free (buf);

  exit (EXIT_SUCCESS);
  return 0;
}


void
usage (char* prog, int exitval)
{
  fprintf (stderr, "Usage: %s [OPTION]... <URL>\n", prog);
  fprintf (stderr, "Endhost multicast utility\n");
  fprintf (stderr, "\t-b <size>         buffer size (default is %d)\n", BUFLEN);
  fprintf (stderr, "\t-h                display help\n");
  fprintf (stderr, "\t-l                turn loopback on\n");
  fprintf (stderr, "\t-o<name> <value>  set option <name> to <value>\n");
  fprintf (stderr, "\t-O<name> <value>  set option <name> to <value> before joining\n");
  fprintf (stderr, "\t-q                quiet (no input/output)\n");
  fprintf (stderr, "\t-t <ttl>          time-to-live\n");
  exit (exitval);
}


static void
sig_int_cb (int ignore)
{
  /* Reset signal handler */
  signal (SIGINT, SIG_DFL);

  if (emcast)
    emcast_leave (emcast);

  exit (EXIT_FAILURE);
}


static int
parse_and_set_opt (char* name, char* val)
{
  int   ival;
  unsigned int xval;
  int rv;

  ival = atoi (val);
  if (ival && val[0] != '0')
    {
      ival = htonl (ival);
      rv = emcast_setopt (emcast, name, &ival, sizeof(ival));
    }
  else if (sscanf(val, "%x", &xval) == 1)
    {
      xval = htonl (xval);
      rv = emcast_setopt (emcast, name, &xval, sizeof(xval));
    }
  else
    {
      rv = emcast_setopt (emcast, name, val, strlen(val));
    }

  return rv;
}
