#include <glib.h>
#include "entity.h"

/* Deleting of nodes is real fun.. basically, when the user calls delete()
 * method, we notify the renderers to destroy itself and all children.  It
 * goes through and unref's them all, and free the node only if the refcount
 * is 0.  Nodes with positive
 * refcount are unlinked from the tree, but not freed. */


#define ENODE_REFCOUNT_DEBUG 0


static void
enode_unlink (ENode * node)
{
    ENode *parent;

#ifdef ENODE_REFCOUNT_DEBUG
    EDEBUG (("refcounting", "unlinking node '%s' (refcount %d)",
	     node->element->str, node->refcount));
#endif				/* ENODE_REFCOUNT_DEBUG */

    parent = enode_parent (node, NULL);

    if (parent) {
	if (parent->children) {
	    parent->children = g_slist_remove_tail (parent->children, node,
						    &parent->children_tail);
	    node->parent = NULL;
	}
    }

    if (node->children) {
	g_slist_free (node->children);
	node->children = NULL;
	node->children_tail = NULL;
    }
}


/* unref only deals with freeing the memory associated with the node. The
 * actual notification to the renderers etc. occurs in enode_destroy_list (), 
 * which is the single point of entry for the enode delete functions */
void
enode_unref (ENode * node)
{
    ECHECK_RET (node != NULL);

    node->refcount--;

#ifdef ENODE_REFCOUNT_DEBUG
    EDEBUG (("refcounting", "-1 refcount to node '%s' (refcount now %d)",
	     node->element->str, node->refcount));
#endif				/* ENODE_REFCOUNT_DEBUG */

    if (node->refcount <= 0) {
	ECHECK (node->refcount >= 0);

	/* insure it's been destroyed */
	if (ENODE_FLAG_ISSET (node, ENODE_DELETED)) {
#ifdef ENODE_REFCOUNT_DEBUG
	    EDEBUG (("refcounting", "freeing node %s.%s (refcount %d)",
		     node->element->str, 
		     enode_attrib_str (node, "name", NULL),
		     node->refcount));
#endif				/* ENODE_REFCOUNT_DEBUG */

	    enode_free (node);
	} else {
	    EBuf *basename = enode_basename (node);
	    g_warning ("attempt to free an undeleted node '%s'!",
		       basename->str);
	    ebuf_free (basename);
	}
    }
}


void
enode_ref (ENode * node)
{
    ECHECK_RET (node != NULL);
    ECHECK (node->refcount >= 0);

#ifdef ENODE_REFCOUNT_DEBUG
    do {
	EBuf *path = enode_path (node);
	EDEBUG (("refcounting", "+1 refcount to node '%s' (refcount now %d)",
		 path->str, node->refcount + 1));
	ebuf_free (path);
    } while (0);
#endif				/* ENODE_REFCOUNT_DEBUG */

    node->refcount++;
}

/* All deletions go through here, which insures the node is fully destroyed */
static void
enode_destroy_real (ENode * node)
{
#ifdef ENODE_REFCOUNT_DEBUG
    EBuf *path = enode_path (node);
    EDEBUG (("refcounting", "enode_destroy_real called for '%s' (refcount %d)",
	     path->str, node->refcount));
    ebuf_free (path);
#endif				/* ENODE_REFCOUNT_DEBUG */

    /* Don't allow double-delete's */
    if (!ENODE_FLAG_ISSET (node, ENODE_DELETED)) {
	ENODE_SET_FLAG (node, ENODE_DELETED);
	enode_event_delete (node);

        {
        EBuf * path;

        path = enode_path (node);
        EDEBUG (("enode-destroy", "freeing %s", path->str));
        ebuf_free (path);
        }

	enode_unlink (node);
	enode_unref (node);
    }
}

static void
enode_destroy_list (GSList * list)
{
    GSList *tmp = list;

    while (tmp) {
	ENode *node = tmp->data;

	enode_destroy_real (node);

	tmp = tmp->next;
    }
}

/* Delete node and all children */
void
enode_destroy (ENode * node)
{
    GSList *list;
    ENode *parent;

    ECHECK_RET (node != NULL);

    parent = enode_parent (node, NULL);

    list = enode_child_list (node, FALSE);
    enode_destroy_list (list);
    g_slist_free (list);

    /* last but not least, destroy the current node too */
    /* Note that this one MUST be destroyed last, so things
     * like widget trees don't get bent out of shape. */
    enode_destroy_real (node);
    
    /* Now check if parent was an instance tag. */
    if (parent && ENODE_FLAG_ISSET (parent, ENODE_INSTANCE_PLACEHOLDER)) {
	enode_destroy_real (parent);
    }
}

/* Delete child nodes */
void
enode_destroy_children (ENode * node)
{
    GSList *list = NULL;

    ECHECK_RET (node != NULL);

    list = enode_child_list (node, FALSE);
    enode_destroy_list (list);
    g_slist_free (list);

    if (node->children)
	g_slist_free (node->children);

    /* Santity checks */
    ECHECK (node->children == NULL);
    ECHECK (node->children_tail == NULL);
}




