#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"


static gint rendgtk_spinner_attr_set (ENode * node, EBuf * attr, EBuf * value);

static void
rendgtk_spinner_onenter_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    gchar *text;
    GtkWidget *spinner;

    spinner = enode_get_kv (node, "top-widget");
    if (!spinner)
	return;

    function = enode_attrib_str (node, "onenter", NULL);

    text = gtk_entry_get_text (GTK_ENTRY (GTK_SPIN_BUTTON (spinner)));
    enode_attrib_quiet (node, "value", ebuf_new_with_str (text));

    enode_call_ignore_return (node, function, "");
}

static void
rendgtk_spinner_onchange_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;
    gchar *text;
    GtkWidget *spinner;

    spinner = enode_get_kv (node, "top-widget");
    if (!spinner)
	return;

    text = gtk_entry_get_text (GTK_ENTRY (GTK_SPIN_BUTTON (spinner)));
    enode_attrib_quiet (node, "value", ebuf_new_with_str (text));

    function = enode_attrib_str (node, "onchange", NULL);

    enode_call_ignore_return (node, function, "");
}


static gint
rendgtk_spinner_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *spinner;		/* The spinner. */
    GtkAdjustment *spinner_adj;	/* How to act. */
    float opt;


    spinner = enode_get_kv (node, "top-widget");
    if (!spinner)
	return TRUE;

    spinner_adj = gtk_spin_button_get_adjustment (GTK_SPIN_BUTTON (spinner));

    opt = erend_get_float (value);

    if (ebuf_equal_str (attr, "max")) {
	spinner_adj->upper = opt;
    } else if (ebuf_equal_str (attr, "min")) {
	spinner_adj->lower = opt;
    } else if (ebuf_equal_str (attr, "value")) {
	spinner_adj->value = opt;
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spinner), opt);
	EDEBUG (("spinner-renderer", "value's opt = %f", opt));
    } else if (ebuf_equal_str (attr, "step")) {
	spinner_adj->step_increment = opt;
    } else if (ebuf_equal_str (attr, "wrap")) {
	gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner),
				  erend_value_is_true (value));
	return TRUE;
    } else if (ebuf_equal_str (attr, "places")) {
	gtk_spin_button_set_digits (GTK_SPIN_BUTTON (spinner),
				    erend_get_integer (value));
    } else if (ebuf_equal_str (attr, "onchange")) {
	return TRUE;
    } else if (ebuf_equal_str (attr, "onenter")) {
	return TRUE;
    } else {
	return FALSE;
    }

    gtk_spin_button_set_adjustment (GTK_SPIN_BUTTON (spinner), spinner_adj);
    return TRUE;
}

/*--element

Element Name: <spinner>

This creates a spin button for int and floating point input.

%widget%

*/

/*--attr

Attribute: "onchange"
If this attribute is set, then whenever a change is made 
to the spinner, the function specified as it's value will
be called.  The first argument is the full path to the entry that
generated the event.

Attribute: "max"
The largest value that the spinner can be.

Attribute: "min"
The smallest value the spinner can be.

Attribute: "step"
How much to add or subtract when a up or down button is pressed.
How much to add or subtract when a up or down key is pressed.

Attribute: "places"
The number of significan't places past the decimal point.

Attribute: "wrap"
If true then the spinner will wrap around at both min and max.

Default: none n(no action)

*/

static void
rendgtk_spinner_render (ENode * node)
{
    GtkWidget *spinner;		/* The new spinner. */
    GtkAdjustment *spinner_adj;	/* How to act. */

    gfloat stepf = 1.0;
    gfloat minf = 0.0;
    EBuf *step;
    EBuf *min;

    step = enode_attrib (node, "step", NULL);
    if (ebuf_not_empty (step))
	stepf = erend_get_float (step);

    min = enode_attrib (node, "min", NULL);
    if (ebuf_not_empty (min))
	minf = erend_get_float (min);

    EDEBUG (("spinner-renderer", "step = %f, min = %f\n", stepf, minf));

    spinner_adj = (GtkAdjustment *)
	gtk_adjustment_new (minf, 0.0, 0.0, stepf, 0.0, 0.0);
    spinner = gtk_spin_button_new (spinner_adj, stepf, 0);

    enode_set_kv (node, "top-widget", spinner);
    enode_set_kv (node, "bottom-widget", spinner);

    enode_attribs_sync (node);

    gtk_signal_connect_after (GTK_OBJECT (spinner), "changed",
			      rendgtk_spinner_onchange_callback, node);
    gtk_signal_connect_after (GTK_OBJECT (spinner), "activate",
			      rendgtk_spinner_onenter_callback, node);

    rendgtk_show_cond (node, spinner);
}

void
spinner_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);

    element->render_func = rendgtk_spinner_render;
    element->destroy_func = rendgtk_element_destroy;
    element->parent_func = rendgtk_box_pack;
    element->tag = "spinner";
    element->description =
	"Create a fancy widget for selecting an integer or floating point value";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onenter";
    e_attr->description =
	"Sets the function to call when the enter is pressed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(event_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onchange";
    e_attr->description =
	"Sets the function to call when the contents is changed.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(event_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "value";
    e_attr->description = "Sets the value in the spinner.";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "min";
    e_attr->description = "Sets the minimum value the spinner can be.";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "max";
    e_attr->description = "Sets the maximum value the spinner can be.";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "step";
    e_attr->description = "Sets the increment size of the spinner.";
    e_attr->value_desc = "float";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "places";
    e_attr->description =
	"Sets the number of decimal places to show in the spinner.";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "wrap";
    e_attr->description =
	"If set the spinner should wrap around at min-1 and max+1";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_spinner_attr_set;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_SPIN_BUTTON);
}
