#include <math.h>
#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "entity.h"
#include "rendgtkgl.h"

static void
rendgtkgl_area_redraw (ENode *glarea);

static gint
rendgtkgl_area_realize (GtkWidget *widget, gpointer data)
{
    ENode *node = data;

   /* OpenGL functions can be called only if make_current returns true */
    if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    {
       	glViewport(0,0, widget->allocation.width, widget->allocation.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0,100, 100,0, -100,100);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glEnable(GL_DEPTH_TEST);
	//glEnable(GL_LIGHTING);
	//glEnable(GL_LIGHT0);
	//glEnable(GL_LIGHT1);
	//glEnable(GL_LIGHT2);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_NORMALIZE);

                glEnable( GL_DEPTH_TEST );
                glShadeModel( GL_SMOOTH );
                glEnable( GL_TEXTURE_2D );
                glAlphaFunc( GL_GREATER, 0.5 );
                glDepthMask( GL_TRUE );
                glDepthFunc( GL_LEQUAL );
                glBlendFunc( GL_ONE, GL_ZERO );
                glEnable( GL_BLEND );
                glDisable( GL_DITHER );
                glPolygonOffset( -1.0f, -1.0 );
              
    }
    EDEBUG (("rendgtkgl", "GTK GL realize called"));

    rendgtkgl_area_redraw (node);

    return TRUE;
}

/* When widget is exposed it's contents are redrawn. */
static gint
rendgtkgl_area_expose (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
    ENode *node = data;

    if (event->count > 0) 
    {
	rendgtkgl_area_redraw (node);
    }
    return TRUE;
}

/* When glarea widget size changes, viewport size is set to match the new size */
static gint
rendgtkgl_area_configure (GtkWidget *widget, GdkEventConfigure *event)
{
  /* OpenGL functions can be called only if make_current returns true */
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    {
      glViewport(0,0, widget->allocation.width, widget->allocation.height);
    }
  return TRUE;
}



static void
rendgtkgl_render (ENode * node)
{
    GtkWidget *glarea;
    EBuf *value;

    /* Attribute list for gtkglarea widget. Specifies a
       list of Boolean attributes and enum/integer
       attribute/value pairs. The last attribute must be
       GDK_GL_NONE. See glXChooseVisual manpage for further
       explanation.
     */
    int attrlist[] = {
	GDK_GL_RGBA,
	GDK_GL_RED_SIZE,1,
	GDK_GL_GREEN_SIZE,1,
	GDK_GL_BLUE_SIZE,1,
	GDK_GL_DOUBLEBUFFER,
	GDK_GL_NONE
    };
    
    /* Check if OpenGL is supported. */
    if (gdk_gl_query() == FALSE) {
	g_warning("OpenGL not known to be supported on this system\n");
	return;
    }

#if 0
    /* This only works in 1.2.2 and greater, so I'll just leave it out. */
    /* vendor dependent version info string */
    info_str = gdk_gl_get_info();
    g_print(info_str);
    g_free(info_str);
#endif
    
    /* Create new OpenGL widget. */
    glarea = GTK_WIDGET(gtk_gl_area_new(attrlist));
    /* Events for widget must be set before X Window is created */
    gtk_widget_set_events(GTK_WIDGET(glarea),
			  GDK_EXPOSURE_MASK|
			  GDK_BUTTON_PRESS_MASK);
    
    /* Redraw image when exposed. */
    gtk_signal_connect(GTK_OBJECT(glarea), "expose_event",
	GTK_SIGNAL_FUNC(rendgtkgl_area_expose), node);
    /* When window is resized viewport needs to be resized also. */
    gtk_signal_connect(GTK_OBJECT(glarea), "configure_event",
	GTK_SIGNAL_FUNC(rendgtkgl_area_configure), node);
    /* Do initialization when widget has been realized. */
    gtk_signal_connect(GTK_OBJECT(glarea), "realize",
	GTK_SIGNAL_FUNC(rendgtkgl_area_realize), node);

  
    /* gtk_widget_set_usize(GTK_WIDGET(glarea), 100,100); */
  
    value = enode_attrib (node, "visible", NULL);

    if ((ebuf_empty (value)) || erend_value_is_true (value))
	gtk_widget_show (glarea);
   
    enode_set_kv (node, "top-widget", glarea);

    EDEBUG (("rendgtkgl", "GTK GL render called"));
}


static int
rendgtkgl_redraw_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    rendgtkgl_area_redraw (node);
    return TRUE;
}


static void
rendgtkgl_parent (ENode *parent_node, ENode *child_node)
{
    EDEBUG (("rendgtkgl", "rendgtkgl_parent called, looking for parent"));
}


static void
rendgtkgl_destroy (ENode * node)
{
    GtkWidget *top = enode_get_kv (node, "top-widget");

    if (top)
	gtk_widget_destroy (top);
}

void
rendgtkgl_line_render (ENode *node)
{
    EBuf *val;
    float x1, y1, z1, x2, y2, z2;
    float width;
    
    
    val = enode_attrib (node, "x1", NULL);
    x1 = erend_get_float (val);

    val = enode_attrib (node, "y1", NULL);
    y1 = erend_get_float (val);

    val = enode_attrib (node, "z1", NULL);
    z1 = erend_get_float (val);

    val = enode_attrib (node, "x2", NULL);
    x2 = erend_get_float (val);

    val = enode_attrib (node, "y2", NULL);
    y2 = erend_get_float (val);

    val = enode_attrib (node, "z2", NULL);
    z2 = erend_get_float (val);

    EDEBUG (("rendgtkgl", "rendering line, %f %f %f -> %f %f %f",x1,y1,z1,x2,y2,z2));

    val = enode_attrib (node, "width", NULL);
    if (ebuf_not_empty (val))
    {
	width = erend_get_float (val);
	EDEBUG (("rendgtkgl", "setting line width to %f", width));
	glLineWidth(width);
    }
    else
    {
	glLineWidth(1.0);
    }

    glBegin(GL_LINES);
    glVertex3f(x1, y1, z1);
    glVertex3f(x2, y2, z2);
    glEnd();  
}

void
rendgtkgl_point_render (ENode *node)
{
    EBuf *val;
    float x, y;
    
    EDEBUG (("rendgtkgl", "rendering point"));
    
    val = enode_attrib (node, "x", NULL);
    x = erend_get_float (val);

    val = enode_attrib (node, "y", NULL);
    y = erend_get_float (val);

    glBegin(GL_POINTS);
    glVertex2f(x, y);
    glEnd();  
}

void
rendgtkgl_matrix_render (ENode *node)
{
    EBuf *val;
    float x, y, z;
    float angle;

    EDEBUG (("rendgtkgl", "rendering matrix"));
    
    glPushMatrix();
    
    val = enode_attrib (node, "translate-x", NULL);
    if (ebuf_not_empty (val))
    {
	x = erend_get_float (val);
	
    	val = enode_attrib (node, "translate-y", NULL);
	y = erend_get_float (val);
	
	val = enode_attrib (node, "translate-z", NULL);
	z = erend_get_float (val);

	glTranslatef (x, y, z);
    }

    val = enode_attrib (node, "rotate-angle", NULL);
    if (ebuf_not_empty (val))
    {
	angle = erend_get_float (val);
	
	val = enode_attrib (node, "rotate-x", NULL);
	x = erend_get_float (val);
	
    	val = enode_attrib (node, "rotate-y", NULL);
	y = erend_get_float (val);
	
	val = enode_attrib (node, "rotate-z", NULL);
	z = erend_get_float (val);

	EDEBUG (("rendgtkgl", "doing gl-matrix rotation, %f, %f, %f, %f",
		angle, x, y, z));
	glRotatef (angle, x, y, z);
    }

    val = enode_attrib (node, "scale-x", NULL);
    if (ebuf_not_empty (val))
    {
	x = erend_get_float (val);
	
    	val = enode_attrib (node, "scale-y", NULL);
	y = erend_get_float (val);
	
	val = enode_attrib (node, "scale-z", NULL);
	z = erend_get_float (val);

	glScalef (x, y, z);
    }
}

void
rendgtkgl_text_render (ENode *node)
{
   EBuf *val;
   float x, y;  
   char *p, *text = NULL;
   
   text = enode_attrib_str (node, "text", NULL);
   
   EDEBUG (("rendgtkgl", "rendering text"));
    
   val = enode_attrib (node, "x", NULL);
   x = erend_get_float (val);
   
   val = enode_attrib (node, "y", NULL);
   y = erend_get_float (val);

   glRasterPos2f(x, y);
      
   for ( p = text; *p; p++ ) {
      glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, *p);
   }
   
}

static void
rendgtkgl_rectangle_render (ENode *node)
{
   /* change values when you split this up okay? */
   EBuf *val;
   float x1, y1, x2, y2;
   
   EDEBUG (("rendgtkgl", "rendering a rectangle" ));
   
   val = enode_attrib (node, "x1", NULL);
   x1 = erend_get_float (val);
   
   val = enode_attrib (node, "y1", NULL);
   y1 = erend_get_float (val);
   
   val = enode_attrib (node, "x2", NULL);
   x2 = erend_get_float (val);
     
   val = enode_attrib (node, "y2", NULL);
   y2 = erend_get_float (val);
   
   EDEBUG (("rendgtkgl", "rendering a rectangle %f, %f, %f, %f",
	    x1, y1, x2, y2));
   
   glRectf (x1, y1, x2, y2);
}


static void
rendgtkgl_cube_render (ENode *node)
{
   EBuf *val;
   float size;

   EDEBUG (("rendgtkgl", "rendering cube"));

   val  = enode_attrib (node, "size", NULL);
   size = erend_get_float (val);   
   glutSolidCube(size);
}

static void
rendgtkgl_sphere_render (ENode *node)
{
   EBuf *val;
   float radius, slices, stacks;
   
   val = enode_attrib (node, "radius", NULL);
   radius = erend_get_float (val);
   
   val = enode_attrib (node, "slices", NULL);
   slices = erend_get_float (val);
   
   val = enode_attrib (node, "stacks", NULL);
   stacks = erend_get_float (val);
   
   glutSolidSphere(radius, slices, stacks);
}

static gint
rendgtkgl_state_render (EBuf *attrib, EBuf *val) 
{
  int symbol=0;

  if (ebuf_equal_str (attrib, "lighting")) {
    symbol=GL_LIGHTING;
  } else 
  if (ebuf_equal_str (attrib, "blend")) {
    symbol=GL_BLEND;
  } else 
  if (ebuf_equal_str (attrib, "depth_test")) {
    symbol=GL_DEPTH_TEST;
  } else 
  if (ebuf_equal_str (attrib, "color_logic_op")) {
    symbol=GL_COLOR_LOGIC_OP;
  } else 
  if (ebuf_equal_str (attrib, "color_material")) {
    symbol=GL_COLOR_MATERIAL;
  } else 
  if (ebuf_equal_str (attrib, "fog")) {
    symbol=GL_FOG;
  } else 
  if (ebuf_equal_str (attrib, "light0")) {
    symbol=GL_LIGHT0;
  } else 
  if (ebuf_equal_str (attrib, "light1")) {
    symbol=GL_LIGHT1;
  } else 
  if (ebuf_equal_str (attrib, "light2")) {
    symbol=GL_LIGHT2;
  } else 
  if (ebuf_equal_str (attrib, "light3")) {
    symbol=GL_LIGHT3;
  } else 
  if (ebuf_equal_str (attrib, "light4")) {
    symbol=GL_LIGHT4;
  } else 
  if (ebuf_equal_str (attrib, "light5")) {
    symbol=GL_LIGHT5;
  } else 
  if (ebuf_equal_str (attrib, "light6")) {
    symbol=GL_LIGHT6;
  } else 
  if (ebuf_equal_str (attrib, "light7")) {
    symbol=GL_LIGHT7;
  } else 
  if (ebuf_equal_str (attrib, "line_smooth")) {
    symbol=GL_LINE_SMOOTH;
  } else 
  if (ebuf_equal_str (attrib, "line_stipple")) {
    symbol=GL_LINE_STIPPLE;
  } else 
  if (ebuf_equal_str (attrib, "normalize")) {
    symbol=GL_NORMALIZE;
  } else 
  if (ebuf_equal_str (attrib, "point_smooth")) {
    symbol=GL_POINT_SMOOTH;
  } else 
  if (ebuf_equal_str (attrib, "polygon_stipple")) {
    symbol=GL_POLYGON_STIPPLE;
  } else 
  if (ebuf_equal_str (attrib, "texture_1d")) {
    symbol=GL_TEXTURE_1D;
  } else 
  if (ebuf_equal_str (attrib, "texture_2d")) {
    symbol=GL_TEXTURE_2D;
  } else 
    {
      return FALSE;
    }

  if (ebuf_equal_strcase (val, "true") ||
        ebuf_equal_strcase (val, "yes") ||
        ebuf_equal_strcase (val, "on") || atoi (val->str)) {
    glEnable(symbol);
  } else {
    glDisable(symbol);
  }

  return TRUE;

/*
 *  todo: add all on/off symbols as in the documentation
 *  
 */

    /* 

 GL_ALPHA_TEST If enabled, do alpha	testing. See
				   glAlphaFunc.

	  GL_AUTO_NORMAL If enabled, generate	normal vectors
				   when	either GL_MAP2_VERTEX_3 or
				   GL_MAP2_VERTEX_4 is used to
				   generate vertices. See glMap2.

	  GL_BLEND If enabled, blend the incoming RGBA
				   color values	with the values	in the
				   color buffers. See glBlendFunc.

	  GL_CLIP_PLANEi If enabled, clip geometry against
				   user-defined	clipping plane i.  See
				   glClipPlane.

	  GL_COLOR_LOGIC_OP If enabled, apply the currently
				   selected logical operation to the
				   incoming RGBA color and color
				   buffer values. See glLogicOp.

	  GL_COLOR_MATERIAL If enabled, have one	or more
				   material parameters track the
				   current color.  See
				   glColorMaterial.

	  GL_CULL_FACE If enabled, cull polygons based on
				   their winding in window
				   coordinates.	See glCullFace.

	  GL_DEPTH_TEST If enabled, do depth	comparisons
				   and update the depth	buffer.	Note
				   that	even if	the depth buffer
				   exists and the depth	mask is	non-
				   zero, the depth buffer is not
				   updated if the depth	test is
				   disabled. See glDepthFunc and
				   glDepthRange.

	  GL_DITHER If enabled, dither color components
				   or indices before they are written
				   to the color	buffer.

	  GL_FOG If enabled, blend a fog color into
				   the posttexturing color.  See
				   glFog.

	  GL_INDEX_LOGIC_OP If enabled, apply the currently
				   selected logical operation to the
				   incoming index and color buffer
				   indices. See
				   glLogicOp.

	  GL_LIGHTi If enabled, include light i in the
				   evaluation of the lighting
				   equation. See glLightModel and
				   glLight.

	  GL_LIGHTING If enabled, use the current
				   lighting parameters to compute the
				   vertex color	or index.  Otherwise,
				   simply associate the	current	color
				   or index with each vertex. See
				   glMaterial, glLightModel, and
				   glLight.

	  GL_LINE_SMOOTH If enabled, draw lines with correct
				   filtering.  Otherwise, draw aliased
				   lines.  See glLineWidth.

	  GL_LINE_STIPPLE If enabled, use the current line
				   stipple pattern when	drawing	lines.
				   See glLineStipple.

	  GL_MAP1_COLOR_4 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate RGBA values.  See glMap1.

	  GL_MAP1_INDEX If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate color indices.  See
				   glMap1.

	  GL_MAP1_NORMAL If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate normals.  See glMap1.

	  GL_MAP1_TEXTURE_COORD_1 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate s texture coordinates.
				   See glMap1.

	  GL_MAP1_TEXTURE_COORD_2 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate s and t texture
				   coordinates.	 See glMap1.

	  GL_MAP1_TEXTURE_COORD_3 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate s, t, and r texture
				   coordinates.	 See glMap1.

	  GL_MAP1_TEXTURE_COORD_4 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate s, t, r, and q texture
				   coordinates.	 See glMap1.

	  GL_MAP1_VERTEX_3 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate x, y, and z vertex
				   coordinates.	 See glMap1.

	  GL_MAP1_VERTEX_4 If enabled, calls to	glEvalCoord1,
				   glEvalMesh1,	and glEvalPoint1
generate homogeneous	x, y, z, and w
vertex coordinates.	See glMap1.

	  GL_MAP2_COLOR_4 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate RGBA values.  See glMap2.

	  GL_MAP2_INDEX If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate color indices.  See
				   glMap2.

	  GL_MAP2_NORMAL If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate normals.  See glMap2.

	  GL_MAP2_TEXTURE_COORD_1 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate s texture coordinates.
				   See glMap2.

	  GL_MAP2_TEXTURE_COORD_2 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate s and t texture
				   coordinates.	 See glMap2.

	  GL_MAP2_TEXTURE_COORD_3 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate s, t, and r texture
				   coordinates.	 See glMap2.

	  GL_MAP2_TEXTURE_COORD_4 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate s, t, r, and q texture
				   coordinates.	 See glMap2.

	  GL_MAP2_VERTEX_3 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate x, y, and z vertex
				   coordinates.	 See glMap2.

	  GL_MAP2_VERTEX_4 If enabled, calls to	glEvalCoord2,
				   glEvalMesh2,	and glEvalPoint2
generate homogeneous	x, y, z, and w
vertex coordinates.	See glMap2.

	  GL_NORMALIZE If enabled, normal vectors
				   specified with glNormal are scaled
				   to unit length after
				   transformation. See glNormal.

	  GL_POINT_SMOOTH If enabled, draw points with	proper
				   filtering.  Otherwise, draw aliased
				   points.  See	glPointSize.

	  GL_POLYGON_OFFSET_FILL If enabled, and if the polygon is
				   rendered in GL_FILL mode, an	offset
				   is added to depth values of a
				   polygon's fragments before the
				   depth comparison is performed. See
				   glPolygonOffset.

	  GL_POLYGON_OFFSET_LINE If enabled, and if the polygon is
				   rendered in GL_LINE mode, an	offset
				   is added to depth values of a
				   polygon's fragments before the
				   depth comparison is performed. See
				   glPolygonOffset.

	  GL_POLYGON_OFFSET_POINT If enabled, an offset is added to
				   depth values	of a polygon's
				   fragments before the	depth
				   comparison is performed, if the
				   polygon is rendered in GL_POINT
mode. See
				   glPolygonOffset.

	  GL_POLYGON_SMOOTH If enabled, draw polygons with
				   proper filtering.  Otherwise, draw
				   aliased polygons. For correct
				   anti-aliased	polygons, an alpha
				   buffer is needed and	the polygons
				   must	be sorted front	to back.

	  GL_POLYGON_STIPPLE If enabled, use the current polygon
				   stipple pattern when	rendering
				   polygons. See glPolygonStipple.

	  GL_SCISSOR_TEST If enabled, discard fragments that
				   are outside the scissor rectangle.
				   See glScissor.

	  GL_STENCIL_TEST If enabled, do stencil testing and
				   update the stencil buffer. See
				   glStencilFunc and glStencilOp.

	  GL_TEXTURE_1D If enabled, one-dimensional
				   texturing is	performed (unless
				   two-dimensional texturing is	also
				   enabled). See glTexImage1D.

	  GL_TEXTURE_2D If enabled, two-dimensional
				   texturing is	performed. See
				   glTexImage2D.

	  GL_TEXTURE_GEN_Q If enabled, the q texture
				   coordinate is computed using	the
				   texture generation function defined
				   with	glTexGen.  Otherwise, the
				   current q texture coordinate	is
				   used.  See glTexGen.

	  GL_TEXTURE_GEN_R If enabled, the r texture
				   coordinate is computed using	the
				   texture generation function defined
				   with	glTexGen.  Otherwise, the
				   current r texture coordinate	is
				   used.  See glTexGen.

	  GL_TEXTURE_GEN_S If enabled, the s texture
				   coordinate is computed using	the
				   texture generation function defined
				   with	glTexGen.  Otherwise, the
				   current s texture coordinate	is
				   used. See glTexGen.

	  GL_TEXTURE_GEN_T If enabled, the t texture
				   coordinate is computed using	the
				   texture generation function defined
				   with	glTexGen.  Otherwise, the
				   current t texture coordinate	is
				   used.  See glTexGen.



    */


}



// IMAGE LOADING

static GdkPixbuf *
rendgtkgl_load_pixbuf (ENode * node, gchar * filename)
{
    GdkPixbuf *pixbuf;
    gchar *realfile;

    realfile = eutils_file_search (node, filename);

    if (!realfile) {
	/* FIXME: Should use onerror handler */
	g_warning ("Unable to locate file '%s'", filename);
	return (NULL);
    }

    pixbuf = gdk_pixbuf_new_from_file (realfile);

    if (!pixbuf)
	return (NULL);

    /* 
     * This could be useful later.. ? enode_set_kv (node, "pixbuf-width",
     * GUINT_TO_POINTER (width)); enode_set_kv (node, "pixbuf-height",
     * GUINT_TO_POINTER (height)); */
    g_free (realfile);
    return (pixbuf);

}

//
//  main stuff
//
//

static gint
rendgtkgl_object_render (ENodeTreeWalk *walker)
{
    EBufConst *type;
    EBufConst *val;
    int ival;
    char *sval;
    GdkColor color;
    ENode *node = walker->curnode;
    float fa,fb,fc,fd;
    char *sa,*sb,*sc,*sd;

    type = enode_type (node);
    
    EDEBUG (("rendgtkgl", "rendering %s", walker->curnode->element->str));
    EDEBUG (("rendgtkgl", "type is |%s|", *type));


    #define E_FLOAT_ATTR(target,daattr,default) val=enode_attrib(node,daattr,NULL); if (ebuf_not_empty(val)) { target=erend_get_float(val); } else { target=default; } 
    #define E_INT_ATTR(target,daattr,default) val=enode_attrib(node,daattr,NULL); if (ebuf_not_empty(val)) { target=erend_get_integer(val); } else { target=default; } 

    #define E_STRING_ATTR(target,daattr,default) sval=enode_attrib_str(node,daattr,NULL); if (sval) { target=sval; } else { target=default; } 



    if (ebuf_equal_str (type, "gl-vertex")) 
    {

      E_FLOAT_ATTR(fa,"x",0)
      E_FLOAT_ATTR(fb,"y",0)
      E_FLOAT_ATTR(fc,"z",0)

      EDEBUG (("rendgtkgl", "vertex3f: %f %f %f", fa,fb,fc));

      glVertex3f(fa,fb,fc);
    } else 
    if (ebuf_equal_str (type, "gl-texcoord")) 
    {

      E_FLOAT_ATTR(fa,"s",0)
      E_FLOAT_ATTR(fb,"t",0)
      E_FLOAT_ATTR(fc,"r",0)

      EDEBUG (("rendgtkgl", "texcoord3f: %f %f %f", fa,fb,fc));

      glTexCoord3f(fa,fb,fc);
    } else 
    if (ebuf_equal_str (type, "gl-color")) 
    {

      E_FLOAT_ATTR(fa,"r",0)
      E_FLOAT_ATTR(fb,"g",0)
      E_FLOAT_ATTR(fc,"b",0)
      E_FLOAT_ATTR(fd,"a",1)

      EDEBUG (("rendgtkgl", "color4f: %f %f %f %f", fa,fb,fc,fd));

      glColor4f(fa,fb,fc,fd);
    } else 
    if (ebuf_equal_str (type, "gl-displaylist")) 
    {
      GLuint mode=GL_COMPILE;
      GLuint num;

      E_STRING_ATTR(sa,"mode","compile")    
      E_INT_ATTR(num,"num",1)    
	
	if (!strcasecmp(sa, "compile_and_execute")) 
	  { mode=GL_COMPILE_AND_EXECUTE; }
	    
  	EDEBUG (("rendgtkgl", "Starting Display List %d %d %s",num,mode,sa));
      glNewList(num,mode);     
    } else 
    if (ebuf_equal_str (type, "gl-calllist")) 
    {
      GLuint num;
      E_INT_ATTR(num,"num",1)    

      EDEBUG (("rendgtkgl", "Calling Display List"));
      glCallList(num);
    } else 
    if (ebuf_equal_str (type, "gl-bindtexture")) 
    {
      static GLuint previous=-1;	
      GLuint num;
      
      E_INT_ATTR(num,"num",1)    

      if (num!=previous) {	

      EDEBUG (("rendgtkgl", "Binding 2D Texture"));
      glBindTexture(GL_TEXTURE_2D,num);
      previous = num;
      
      EDEBUG (("rendgtkgl", "bindtexture error: %d", glGetError()));

      } else {
      
      EDEBUG (("rendgtkgl", "2D Texture already bound"));
      
      }
    } else 
    if (ebuf_equal_str (type, "gl-teximage2d")) 
    {
      int done;
      int width,height;
      GdkPixbuf *image;
      
      E_INT_ATTR(done,"done",FALSE)    
 
      if (!done) {   
 
      E_STRING_ATTR(sa,"src","")    

 
     EDEBUG (("rendgtkgl", "attempting to load image '%s'", sa));
    image = rendgtkgl_load_pixbuf (node, sa);

    if (image) {
    
	width = gdk_pixbuf_get_width (image);
 	height = gdk_pixbuf_get_height (image);

	EDEBUG (("rendgtkgl", "image '%s' (%dx%d), %d channels - loaded", sa,width,height,gdk_pixbuf_get_n_channels(image)));

//	enode_set_kv (node, "rendgtkgl-image-pixbuf", image);
	
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	
      EDEBUG (("rendgtkgl", "Specifying 2D Texture"));
      glTexImage2D(GL_TEXTURE_2D,0,GL_RGB8,width,height,0,GL_RGB,GL_UNSIGNED_BYTE,gdk_pixbuf_get_pixels(image));

      EDEBUG (("rendgtkgl", "teximage2d error: %d", glGetError()));
      
     // release the pixbuf
      
      gdk_pixbuf_unref(image);

    } else {
	EDEBUG (("rendgtkgl", "failed to load '%s'", sa));
    }
       	    
	enode_attrib_str(node,"done","1");
      
     }  
    else {
      
      EDEBUG (("rendgtkgl", "2D Texture already done"));
      
      }
    
    } else 
    if (ebuf_equal_str (type, "gl-light-ambient")) 
    {

      float params[4];
      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting GL_AMBIENT on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(params[0],"r",0)
      E_FLOAT_ATTR(params[1],"g",0)
      E_FLOAT_ATTR(params[2],"b",0)
      E_FLOAT_ATTR(params[3],"a",1)


      glLightfv(num,GL_AMBIENT,params);

    } else 
    if (ebuf_equal_str (type, "gl-light-diffuse")) 
    {

      float params[4];
      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting GL_DIFFUSE on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(params[0],"r",0)
      E_FLOAT_ATTR(params[1],"g",0)
      E_FLOAT_ATTR(params[2],"b",0)
      E_FLOAT_ATTR(params[3],"a",1)


      glLightfv(num,GL_DIFFUSE,params);

    } else 
    if (ebuf_equal_str (type, "gl-light-specular")) 
    {

      float params[4];
      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting GL_SPECULAR on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(params[0],"r",0)
      E_FLOAT_ATTR(params[1],"g",0)
      E_FLOAT_ATTR(params[2],"b",0)
      E_FLOAT_ATTR(params[3],"a",1)


      glLightfv(num,GL_SPECULAR,params);

    } else 
    if (ebuf_equal_str (type, "gl-light-spot-direction")) 
    {

      float params[3];
      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting GL_SPOT_DIRECTION on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(params[0],"x",0)
      E_FLOAT_ATTR(params[1],"y",0)
      E_FLOAT_ATTR(params[2],"z",-1)

      glLightfv(num,GL_SPOT_DIRECTION,params);

    } else 
    if (ebuf_equal_str (type, "gl-light-position")) 
    {

      float params[4];
      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting GL_POSITION on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(params[0],"x",0)
      E_FLOAT_ATTR(params[1],"y",0)
      E_FLOAT_ATTR(params[2],"z",1)
      E_FLOAT_ATTR(params[2],"w",1)

      glLightfv(num,GL_POSITION,params);

    } else 
    if (ebuf_equal_str (type, "gl-light-attenuation")) 
    {

      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting attenuation on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(fa,"linear",-1)
      E_FLOAT_ATTR(fb,"quadratic",-1)
      E_FLOAT_ATTR(fc,"constant",-1)

      if (fa>=0)	
	glLightf(num,GL_LINEAR_ATTENUATION,fa);
      if (fb>=0)	
	glLightf(num,GL_QUADRATIC_ATTENUATION,fb);
      if (fc>=0)	
	glLightf(num,GL_CONSTANT_ATTENUATION,fc);

    } else 
    if (ebuf_equal_str (type, "gl-light-spot-shape")) 
    {

      GLuint num;

      E_INT_ATTR(num,"num",0)    
      if (num>7) num=7;	
      if (num<0) num=0;	

      EDEBUG (("rendgtkgl", "setting attenuation on light %u", num));
      num+=GL_LIGHT0;

      E_FLOAT_ATTR(fa,"exponent",-1)
      E_FLOAT_ATTR(fb,"cutoff",-1)

      if (fa>=0)	
	glLightf(num,GL_SPOT_EXPONENT,fa);
      if (fb>=0)	
	glLightf(num,GL_SPOT_CUTOFF,fb);

    } else 
    if (ebuf_equal_str (type, "gl-lines")) 
    {
      E_FLOAT_ATTR(fa,"width",0)    
      if (fa>0) { glLineWidth(fa); }     

  	EDEBUG (("rendgtkgl", "Starting Lines Primitive"));
      glBegin(GL_LINES);     
    } else 
    if (ebuf_equal_str (type, "gl-trianglestrip")) 
    {
      glBegin(GL_TRIANGLE_STRIP);     
    } else
    if (ebuf_equal_str (type, "gl-quads")) 
    {
  	EDEBUG (("rendgtkgl", "Starting Quads Primitive"));
      glBegin(GL_QUADS);     
    } else
    if (ebuf_equal_str (type, "gl-points")) 
    {
      glBegin(GL_POINTS);     
    } else
    if (ebuf_equal_str (type, "gl-linestrip")) 
    {
      glBegin(GL_LINE_STRIP);     
    } else
    if (ebuf_equal_str (type, "gl-lineloop")) 
    {
      glBegin(GL_LINE_LOOP);     
    } else
    if (ebuf_equal_str (type, "gl-polygon")) 
    {
      glBegin(GL_POLYGON);     
    } else
    if (ebuf_equal_str (type, "gl-trianglefan")) 
    {
      glBegin(GL_TRIANGLE_FAN);     
    } else
    if (ebuf_equal_str (type, "gl-triangles")) 
    {
      glBegin(GL_TRIANGLES);     
    } else
    if (ebuf_equal_str (type, "gl-quadstrip")) 
    {
      glBegin(GL_QUAD_STRIP);     
    } else
    if (ebuf_equal_str (type, "gl-state")) 
      {
	GSList *tmp;
     
	tmp=node->attribs;
	while (tmp) {
	  EBuf *attrib;
	  EBuf *value;

	  attrib = tmp->data;
	  tmp = tmp->next;
	  value = tmp->data;
	  tmp = tmp->next;

	  if (ebuf_not_empty (value))
	    rendgtkgl_state_render(attrib,value);
	}

	return TRUE;
      } 





    /* Common stuff */
    val = enode_attrib (node, "color", NULL);
    if (ebuf_not_empty (val)) 
    {
	if (gdk_color_parse (val->str, &color) )
	{
	    glColor3us (color.red, color.green, color.blue);
        }
	else
	{
	    glColor3us (255, 255, 255);
	}
    }
    
    if (ebuf_equal_str (type, "gl-point")) 
    {
	rendgtkgl_point_render (node);
    } 
    else if (ebuf_equal_str (type, "gl-line")) 
    {
	rendgtkgl_line_render (node);
    } 
    else if (ebuf_equal_str (type, "gl-matrix")) 
    {
	rendgtkgl_matrix_render (node);
    }
    else if (ebuf_equal_str (type, "gl-rectangle"))
    {
        rendgtkgl_rectangle_render (node);
    }
    else if (ebuf_equal_str (type, "glut-cube"))
    { 
        rendgtkgl_cube_render (node);  
    } 
    else if (ebuf_equal_str (type, "glut-sphere"))
    {
        rendgtkgl_sphere_render (node);	
    }
    else if (ebuf_equal_str (type, "glut-text"))
    {
        rendgtkgl_text_render (node);
    }
      
    return (TRUE);
}

static gint
rendgtkgl_object_parent (ENodeTreeWalk *walker)
{
    EBuf *type;
    
    EDEBUG (("rendgtkgl", "Parenting %s to parent %s", walker->curnode->element->str,
    walker->parentnode->element->str));

    type = enode_type (walker->curnode);

    
    if (
	(ebuf_equal_str (type, "gl-lines"))
	||(ebuf_equal_str (type, "gl-trianglestrip"))
	||(ebuf_equal_str (type, "gl-quads"))
	||(ebuf_equal_str (type, "gl-points"))
	||(ebuf_equal_str (type, "gl-linestrip"))
	||(ebuf_equal_str (type, "gl-lineloop"))
	||(ebuf_equal_str (type, "gl-polygon"))
	||(ebuf_equal_str (type, "gl-trianglefan"))
	||(ebuf_equal_str (type, "gl-triangles"))
	||(ebuf_equal_str (type, "gl-quadstrip"))
	) 
    {
    	EDEBUG (("rendgtkgl", "Ending Primitive"));
      glEnd();     
    } else 
    if (ebuf_equal_str(type, "gl-matrix"))
    {
    	EDEBUG (("rendgtkgl", "Popping matrix"));
	glPopMatrix();
    } else 
    if (ebuf_equal_str (type, "gl-displaylist")) 
    {
      EDEBUG(("rendgtkgl","Ending displaylist"));
      glEndList();
      
    }
    return (TRUE);
}
		
static void
rendgtkgl_area_redraw (ENode *glarea)
{
    GtkWidget *widget;

    widget = enode_get_kv (glarea, "top-widget");
    if (!widget)
	return;
    
    /* OpenGL functions can be called only if make_current returns true */
    if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    {
    	ENodeTreeWalk *walker;
 	glClearColor(0,0,0,1);
    	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	
	walker = enode_treewalk_new (glarea);
	enode_treewalk (walker, rendgtkgl_object_render, rendgtkgl_object_parent);
	enode_treewalk_free (walker);

	/* Swap backbuffer to front */
	gtk_gl_area_swapbuffers(GTK_GL_AREA(widget));
    }

}


void
renderer_init (RendererFlags flags)
{
    Element *element;
    ElementAttr *e_attr;

    EDEBUG (("rendgtkgl", "registering rendgtkgl"));
    
    if (flags & RENDERER_REGISTER)
    {
      	EDEBUG (("rendgtkgl", "Registering glarea element"));
     	element = g_new0 (Element, 1);
	element->render_func = rendgtkgl_render;
	element->parent_func = rendgtkgl_parent;
	element->destroy_func = rendgtkgl_destroy;
	element->description = "Create a gtk GL area for doing GL based graphics.";
	element->tag = "glarea";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "redraw";
	e_attr->description = "Cause a redraw of the GL Area.";
	e_attr->value_desc = "*";
	e_attr->set_attr_func = rendgtkgl_redraw_attr_set;
	element_register_attrib (element, e_attr);


	/* GtkGLArea Point Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-point";
	element->description = "Create a single point on the gl area.";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "x";
	e_attr->description = "Horizontal point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "y";
	e_attr->description = "Vertical point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);

        /* GtkGLArea Rectangle Renderer, change atribs when you split this up please */
        element = g_new0 (Element, 1);
        element->tag = "gl-rectangle";
	element->description = "Create a rectangle in the gl area.";
        element_register (element);
       
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "x1";
        e_attr->description = "First horizontal point location.";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
        
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "y1";
        e_attr->description = "First vertical point location.";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);   

        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "x2";
        e_attr->description = "Second horizontal point location.";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
       
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "y2";
        e_attr->description = "Second vertical point location.";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr); 
       
        /* GtkGLArea Line Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-line";
	element->description = "Create a line in the gl area.";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "x1";
	e_attr->description = "First horizontal point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "y1";
	e_attr->description = "First vertical point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
   
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "z1";
	e_attr->description = "First virtual point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
   
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "x2";
	e_attr->description = "Second horizontal point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "y2";
	e_attr->description = "Second vertical point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);

	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "z2";
	e_attr->description = "Second virtual point location.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
        /* GtkGLArea Cube Renderer */
        element = g_new0 (Element, 1);
        element->tag = "glut-cube";
	element->description = "Draw a cube in the gl area.";
        element_register (element);

        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "size";
        e_attr->description = "Size of cube.";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
        element = g_new0 (Element, 1);

        /* GtkGLArea Sphere Renderer */
        element->tag = "glut-sphere";
	element->description = "Create a sphere in the gl area.";
        element_register (element);

        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "radius";
        e_attr->description = "radius of sphere";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);

        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "slices";
        e_attr->description = "number of subdivisions around X axis";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);

        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "stacks";
        e_attr->description = "number of subdivisions along Z axis";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
       
        /* font text render */
        element = g_new0 (Element, 1);
        element->tag = "glut-text";
	element->description = "Draw text in a gl area.";
        element_register (element);
               
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "text";
        e_attr->description = "text in widget";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
              
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "x";
        e_attr->description = "x co-ordinate";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);
       
        e_attr = g_new0 (ElementAttr, 1);
        e_attr->attribute = "y";
        e_attr->description = "y coordinate";
        e_attr->value_desc = "0,*";
        element_register_attrib (element, e_attr);       

	// NEW STUFFS

/*  
 *  todo: add all new elements as in the implementation
 *  missing: a lot of them
 */

        /* GtkGLArea Lines Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-lines";
	element->description = "Draw separate lines in the gl area.";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "width";
	e_attr->description = "Width of the drawn lines.";
	e_attr->value_desc = "1,*";
	element_register_attrib (element, e_attr);
    
        /* GtkGLArea Points Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-points";
	element->description = "Draw points in the gl area.";
	element_register (element);
	
        /* GtkGLArea Triangles Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-triangles";
	element->description = "Draw separate triangles in the gl area.";
	element_register (element);
	
        /* GtkGLArea Quads Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-quads";
	element->description = "Draw separate quads in the gl area.";
	element_register (element);
	
        /* GtkGLArea Triangle Strip Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-trianglestrip";
	element->description = "Draw a strip of connected triangles in the gl area.";
	element_register (element);
	
        /* GtkGLArea Line Strip Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-linestrip";
	element->description = "Draw a strip of connected lines in the gl area.";
	element_register (element);
	
        /* GtkGLArea Vertex3f Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-vertex";
	element->description = "Specify a single vertex with 3 float values.";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "x";
	e_attr->description = "Vertex X coordinate.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "y";
	e_attr->description = "Vertex Y coordinate.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "z";
	e_attr->description = "Vertex Z coordinate.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	
        /* GtkGLArea Color4f Renderer */
	element = g_new0 (Element, 1);
	element->tag = "gl-color";
	element->description = "Specify the current color with 4 float values.";
	element_register (element);
	
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "r";
	e_attr->description = "Color R component.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "g";
	e_attr->description = "Color G component.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "b";
	e_attr->description = "Color B component.";
	e_attr->value_desc = "0,*";
	element_register_attrib (element, e_attr);
    
	e_attr = g_new0 (ElementAttr, 1);
	e_attr->attribute = "a";
	e_attr->description = "Color A component.";
	e_attr->value_desc = "1,*";
	element_register_attrib (element, e_attr);
    
       
    }
}


