# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the UMLWidget base class.
"""

from qt import *
from qtcanvas import *

class UMLWidget(QCanvasRectangle):
    """
    Class implementing the UMLWidget base class.
    """
    def __init__(self, canvas, x=0, y=0, rounded=0):
        """
        Constructor
        
        @param canvas canvas containing the class widget (QCanvas)
        @param x x-coordinate (integer)
        @param y y-coordinate (integer)
        @param rounded flag indicating a rounded corner (boolean)
        """
        QCanvasRectangle.__init__(self, canvas)
        self.font = qApp.font()
        self.font.setPointSize(10)
        self.margin = 5
        self.associations = []
        
        self.setSize(60, 30)
        
        if rounded:
            p = self.pen()
            p.setCapStyle(Qt.RoundCap)
            p.setJoinStyle(Qt.RoundJoin)
            self.setPen(p)
            
        self.setX(x)
        self.setY(y)
        
        self.update()
        
    def addAssociation(self, assoc):
        """
        Method to add an association to this widget.
        
        @param assoc association to be added (AssociationWidget)
        """
        if assoc and not assoc in self.associations:
            self.associations.append(assoc)
            
    def adjustAssociations(self):
        """
        Method to adjust the associations to widget movements.
        """
        for assoc in self.associations:
            assoc.widgetMoved()
            
    def moveBy(self, dx, dy):
        """
        Overriden method to move the widget relative.
        
        @param dx relative movement in x-direction (float)
        @param dy relative movement in y-direction (float)
        """
        QCanvasRectangle.moveBy(self, dx, dy)
        self.adjustAssociations()
        
    def move(self, x, y):
        """
        Overriden method to move the widget absolute.
        
        @param dx absolute x-position (float)
        @param dy absolute y-position (float)
        """
        QCanvasRectangle.move(self, x, y)
        self.adjustAssociations()
        
    def drawSelected(self, painter, x, y):
        """
        Method to draw the widget as selected.
        
        @param painter painter to draw on (QPainter)
        @param x x-position (integer)
        @param y y-position (integer)
        """
        w = self.width()
        h = self.height()
        s = 6
        brush = QBrush(Qt.red)
        painter.fillRect(x, y, s, s, brush)
        painter.fillRect(x, y + h - s, s, s, brush)
        painter.fillRect(x + w - s, y, s, s, brush)
        painter.fillRect(x + w - s, y + h - s, s, s, brush)
