# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing interfaces to source code documentation tools.

It implements the high level interface and all the supporting dialogs
for the supported source code documentors.
"""

import sys
import os.path
import shutil
import copy

from qt import *

import Utilities
from EricdocConfigDialog import EricdocConfigDialog
from EricapiConfigDialog import EricapiConfigDialog
from DocGeneratorExecDialog import DocGeneratorExecDialog

from UI.E3Action import E3Action, E3ActionGroup

class DocumentationTools(QObject):
    """ 
    Class implementing the high level interface.
    """
    def __init__(self, project, parent):
        """
        Constructor
        
        @param project project object of the current project
        @param parent parent object of this class (QObject)
        """
        QObject.__init__(self, parent)
        
        self.project = project
        
    def initActions(self):
        """
        Private method to initialize the API documentation actions.
        """
        self.actions = []
        
        self.ericapiAct = E3Action(self.trUtf8('Use eric3-api'),
            self.trUtf8('Use eric3-&api'),0,0,self,'doc_eric3_api')
        self.ericapiAct.setStatusTip(\
            self.trUtf8('Generate an API file using eric3-api'))
        self.ericapiAct.setWhatsThis(self.trUtf8(
            """<b>Use eric3-api</b>"""
            """<p>Generate an API file using eric3-api.</p>"""
                        ))
        self.ericapiAct.connectIt(SIGNAL('activated()'),self.handleEricapi)
        self.actions.append(self.ericapiAct)
        
        self.ericdocAct = E3Action(self.trUtf8('Use eric3-doc'),
            self.trUtf8('Use eric3-&doc'),0,0,self,'doc_eric3_doc')
        self.ericdocAct.setStatusTip(\
            self.trUtf8('Generate API documentation using eric3-doc'))
        self.ericdocAct.setWhatsThis(self.trUtf8(
            """<b>Use eric3-doc</b>"""
            """<p>Generate API documentation using eric3-doc.</p>"""
                        ))
        self.ericdocAct.connectIt(SIGNAL('activated()'),self.handleEricdoc)
        self.actions.append(self.ericdocAct)
        
    def initMenu(self):
        """
        Private method called to build the project API documentation submenu.
        
        @return the menu or None
        """
        menu = QPopupMenu(self.parent())
        menu.insertTearOffHandle()
        
        self.ericdocAct.addTo(menu)
        self.ericapiAct.addTo(menu)
        
        return menu
        
    def handleEricdoc(self):
        """
        Private slot to perform the eric3-doc api documentation generation.
        """
        if len(self.project.pdata['ERIC3DOCPARMS']):
            parms = copy.deepcopy(self.project.pdata['ERIC3DOCPARMS'][0])
        else:
            parms = None
        dlg = EricdocConfigDialog(self.project.ppath, parms)
        if dlg.exec_loop() == QDialog.Accepted:
            args, parms = dlg.generateParameters()
            
            # test for changes of selected parameters and save the in the project
            # 1. there were none, now there are
            if len(self.project.pdata['ERIC3DOCPARMS']) == 0 and len(parms) > 0:
                self.project.pdata['ERIC3DOCPARMS'] = [copy.deepcopy(parms)]
                self.project.setDirty(1)
            # 2. there were some, now there aren't
            elif len(self.project.pdata['ERIC3DOCPARMS']) > 0 and len(parms) == 0:
                self.project.pdata['ERIC3DOCPARMS'] = []
                self.project.setDirty(1)
            # 3. there were some and still are
            else:
                if parms != self.project.pdata['ERIC3DOCPARMS'][0]:
                    self.project.pdata['ERIC3DOCPARMS'] = [copy.deepcopy(parms)]
                    self.project.setDirty(1)
                    
            # now do the call
            dia = DocGeneratorExecDialog("Ericdoc")
            res = dia.start(args, self.project.ppath)
            if res:
                dia.exec_loop()
                
            outdir = parms['outputDirectory']
            if outdir == '':
                outdir = 'doc'      # that is eric3-docs default output dir
                
            # add it to the project data, if it isn't in already
            outdir = outdir.replace(self.project.ppath+os.sep, '')
            if outdir not in self.project.pdata['OTHERS']:
                self.project.pdata['OTHERS'].append(outdir)
                self.project.setDirty(1)
                self.project.othersAdded(outdir)
        
    def handleEricapi(self):
        """
        Private slot to perform the eric3-api api generation.
        """
        if len(self.project.pdata['ERIC3APIPARMS']):
            parms = copy.deepcopy(self.project.pdata['ERIC3APIPARMS'][0])
        else:
            parms = None
        dlg = EricapiConfigDialog(self.project.ppath, parms)
        if dlg.exec_loop() == QDialog.Accepted:
            args, parms = dlg.generateParameters()
            
            # test for changes of selected parameters and save the in the project
            # 1. there were none, now there are
            if len(self.project.pdata['ERIC3APIPARMS']) == 0 and len(parms) > 0:
                self.project.pdata['ERIC3APIPARMS'] = [copy.deepcopy(parms)]
                self.project.setDirty(1)
            # 2. there were some, now there aren't
            elif len(self.project.pdata['ERIC3APIPARMS']) > 0 and len(parms) == 0:
                self.project.pdata['ERIC3APIPARMS'] = []
                self.project.setDirty(1)
            # 3. there were some and still are
            else:
                if parms != self.project.pdata['ERIC3APIPARMS'][0]:
                    self.project.pdata['ERIC3APIPARMS'] = [copy.deepcopy(parms)]
                    self.project.setDirty(1)
                    
            # now do the call
            dia = DocGeneratorExecDialog("Ericapi")
            res = dia.start(args, self.project.ppath)
            if res:
                dia.exec_loop()
                
            outfile = parms['outputFile']
                
            # add it to the project data, if it isn't in already
            outfile = outfile.replace(self.project.ppath+os.sep, '')
            if outfile not in self.project.pdata['OTHERS']:
                self.project.pdata['OTHERS'].append(outfile)
                self.project.setDirty(1)
                self.project.othersAdded(outfile)
        
    def getActions(self):
        """
        Public method to get a list of all actions.
        
        @return list of all actions (list of E3Action)
        """
        return self.actions[:]
