/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////


#include "Foundation/StringUtil.h"
#include "Parallel/CheckPointInfo.h"
#include "Geometry/GeometryInfo.h"
#include "MainBE/version.h"

namespace esys
{
  namespace lsm
  {
    class CheckPointInfo::Impl
    {
    public:
      Impl()
        : m_numTimeSteps(0),
          m_timeStepSize(0.0),
          m_timeStep(0),
          m_geoInfo(),
          m_fileNames()
      {
      }

      ~Impl()
      {
      }

      bool operator==(const Impl &impl) const
      {
        return
          (
            (m_numTimeSteps == impl.m_numTimeSteps)
            &&
            (m_timeStepSize == impl.m_timeStepSize)
            &&
            (m_timeStep     == impl.m_timeStep)
            &&
            (m_geoInfo      == impl.m_geoInfo)
            &&
            (m_fileNames    == impl.m_fileNames)
          );
      }
      
      void read(std::istream &iStream)
      {
	string magic;
	int version;
        iStream	>> magic >> version; 
	iStream 
          >> m_numTimeSteps
          >> m_timeStepSize
          >> m_timeStep;
          m_geoInfo.read(iStream);

          std::string fileNames;
          /*
           * Skip over any blank lines.
           */
          while (fileNames.size() <= 0) {
            std::getline(iStream, fileNames);
            fileNames = StringUtil::trim(fileNames);
          }
          m_fileNames = StringUtil::splitStrings(fileNames, " ");
      }

      void write(std::ostream &oStream) const
      {
        const char delim = '\n';
        oStream
	  << "V " << lsm_version_info::CheckPointVersion << delim
          << m_numTimeSteps << delim
          << m_timeStepSize << delim
          << m_timeStep     << delim;
        m_geoInfo.write(oStream);
        oStream
          << delim
          << StringUtil::join(
              m_fileNames.begin(),
              m_fileNames.end(),
              std::string(" "),
              StringUtil::StdOStreamOp<StringVector::const_iterator>()
            );
      }

      int          m_numTimeSteps;
      double       m_timeStepSize;
      int          m_timeStep;
      GeometryInfo m_geoInfo;
      StringVector m_fileNames;
    };

    CheckPointInfo::CheckPointInfo() : m_pImpl(new CheckPointInfo::Impl)
    {
    }

    CheckPointInfo::~CheckPointInfo()
    {
      delete m_pImpl;
    }
    
    bool CheckPointInfo::operator==(const CheckPointInfo &cpInfo) const
    {
      return (*m_pImpl == *(cpInfo.m_pImpl));
    }

    const GeometryInfo &CheckPointInfo::getGeometryInfo() const
    {
      return m_pImpl->m_geoInfo;
    }

    void CheckPointInfo::setGeometryInfo(const GeometryInfo &geoInfo)
    {
      m_pImpl->m_geoInfo = geoInfo;
    }
    
    const StringVector &CheckPointInfo::getLatticeDataFiles() const
    {
      return m_pImpl->m_fileNames;
    }

    void CheckPointInfo::setLatticeDataFiles(const StringVector &fileNames)
    {
      m_pImpl->m_fileNames = fileNames;
    }
    
    void CheckPointInfo::setNumTimeSteps(int numTimeSteps)
    {
      m_pImpl->m_numTimeSteps = numTimeSteps;
    }

    void CheckPointInfo::setTimeStep(int timeStep)
    {
      m_pImpl->m_timeStep = timeStep;
    }

    int CheckPointInfo::getTimeStep() const
    {
      return m_pImpl->m_timeStep;
    }
    
    void CheckPointInfo::setTimeStepSize(double timeStepSize)
    {
      m_pImpl->m_timeStepSize = timeStepSize;
    }

    void CheckPointInfo::read(std::istream &iStream)
    {
      m_pImpl->read(iStream);
    }

    void CheckPointInfo::write(std::ostream &oStream) const
    {
      m_pImpl->write(oStream);
    }
  };
};
