#############################################################
##                                                         ##
## Copyright (c) 2003-2011 by The University of Queensland ##
## Earth Systems Science Computational Centre (ESSCC)      ##
## http://www.uq.edu.au/esscc                              ##
##                                                         ##
## Primary Business: Brisbane, Queensland, Australia       ##
## Licensed under the Open Software License version 3.0    ##
## http://www.opensource.org/licenses/osl-3.0.php          ##
##                                                         ##
#############################################################

"""Defines the L{Capsule} base class"""
from esys.lsm.vis import core
from Modifier     import Modifiable
from Csg          import Union
from Box          import Box
from Cylinder     import Disk

class Capsule(core.Capsule, Modifiable):
    """
    Objects of this class represent capsules, a capsule capped with
    a hemisphere at each end.
    """
    def __init__(self, radius, endPt1=None, endPt2=None, segEndPt1=None, segEndPt2=None):
        """
        Initialises capsule with center-end-point coordinates and radius.
        @type radius: float
        @param radius: Radius of capsule.
        @type endPt1: sequence of 3 floats
        @param endPt1: Coordinate of one end ("apex" of hemisphere).
        @type endPt2: sequence of 3 floats
        @param endPt2: Center coordinate of other end ("apex" of hemisphere).
        @type segEndPt1: sequence of 3 floats
        @param segEndPt1: Coordinate of one end ("apex" of hemisphere).
        @type segEndPt2: sequence of 3 floats
        @param segEndPt2: Center coordinate of other end ("apex" of hemisphere).
        """
        core.Capsule.__init__(self,radius,endPt1,endPt2,segEndPt1,segEndPt2)
        Modifiable.__init__(self)

    def writeSdl(self, f):
        radius = self.getRadius()
        f.write("\nsphere_sweep {\nlinear_spline\n2\n")
        f.write("<%s,0,0> %s\n" % (-self.getSegmentLength()*0.5,radius))
        f.write("<%s,0,0> %s\n" % (self.getSegmentLength()*0.5,radius))
        Modifiable.writeSdl(self, f)
        f.write("\ntranslate <%s,%s,%s>\n" % tuple(self.getCenter()))
        f.write("}")

class CapsuleDisk(core.CapsuleDisk, Modifiable):
    """
    Objects of this class represent a cylinder with a capsule cross-section.
    """
    def __init__(
        self,
        center,
        length,
        radius,
        height = None
    ):
        core.CapsuleDisk.__init__(self,center,length,radius,height)
        Modifiable.__init__(self)

    def writeSdl(self, f):
        csg = Union()
        halfSegLength = 0.5*self.getSegmentLength()
        box = \
          Box(
            core.Vec3(-halfSegLength, -self.getRadius(), -0.5*self.getHeight()),
            core.Vec3(halfSegLength, self.getRadius(), 0.5*self.getHeight())
          )
        csg.append(box)
        csg.append(
          Disk(
            center = core.Vec3(-halfSegLength, 0, 0),
            radius = self.getRadius(),
            height = self.getHeight(),
            direction = (0,0,1)
          )
        )
        csg.append(
          Disk(
            center = core.Vec3(halfSegLength, 0, 0),
            radius = self.getRadius(),
            height = self.getHeight(),
            direction = (0,0,1)
          )
        )
        csg.writeBegin(f)
        csg.writeObjects(f)
        Modifiable.writeSdl(self,f)
        f.write("\ntranslate <%s,%s,%s>\n" % tuple(self.getCenter()))
        csg.writeEnd(f)
