/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aixregmgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the AIX region manager. This is also where all AIX global
 *              data is declared.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"


// Global variables
static engine_mode_t		open_mode;
engine_functions_t		* aix_engine;		// The Engine's internal API set.
plugin_record_t			* aix_plugin;		// Used to identify containers and regions as owned by AIX.
dlist_t				aix_container_list;	// List of real containers.


/* aix_setup_evms_plugin
 *
 *	This function gets called shortly after the plugin is loaded by the
 *	Engine. It performs all tasks that are necessary before the initial
 *	discovery pass.
 */
static int aix_setup_evms_plugin(engine_mode_t		mode,
				engine_functions_t	* functions)
{
	open_mode = mode;
	aix_engine = functions;
	aix_plugin = &AIX_Plugin;

	// Allocate the global container list.
	aix_container_list = CreateList();
	if ( ! aix_container_list ) {
		LOG_CRITICAL("Error creating AIX container list.\n");
		return EINVAL;
	}

	// Register the AIX namespace so nobody tries to steal it.
	aix_engine->register_name(EVMS_DEV_NODE_PATH "aix");

	return 0;
}


/* aix_cleanup_evms_plugin
 *
 *	Memory cleanup before closing the engine.
 */
static void aix_cleanup_evms_plugin( void )
{
	storage_container_t	* container;
	TAG			tag;
	unsigned int		size;

	LOG_ENTRY;

	GoToStartOfList(aix_container_list);
	while ( ! BlindExtractObject(aix_container_list, &size, &tag, NULL, (void**)&container) ) {
		aix_deallocate_container(container);
	}

	DestroyList(&aix_container_list, FALSE);

	LOG_EXIT(0);
}


/****** Region Checking Functions ******/

/* All of the following aix_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */



/* aix_can_delete
 *
 *	Can we remove the specified AIX region, and consolidate the
 *	space with the freespace region?
 */
static int aix_can_delete( storage_object_t * region )
{
	LOG_ENTRY;
	LOG("Not yet implemented.\n");
	RETURN(ENOSYS);
}


/* aix_can_expand
 *
 *	Can this region be expanded, using freespace from its container? If so,
 *	add the region to the expansion_points output list.
 */
static int aix_can_expand(storage_object_t	* region,
			u_int64_t		* expand_limit,
			dlist_t			expansion_points )
{
	LOG_ENTRY;
	LOG("Not yet implemented.\n");
	RETURN(ENOSYS);
}


/* aix_can_expand_by
 *
 *	AIX does not allow its child objects to expand.
 */
static int aix_can_expand_by(	storage_object_t	* child_object,
				u_int64_t		* size )
{
	LOG_ENTRY;
	RETURN(EINVAL);
}


/* aix_can_shrink
 */
static int aix_can_shrink(storage_object_t	* region,
			u_int64_t		* shrink_limit,
			dlist_t			shrink_points )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_can_shrink_by
 */
static int aix_can_shrink_by(	storage_object_t	* region,
				u_int64_t		* size )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_can_move
 */
static int aix_can_move( storage_object_t * region )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_can_set_volume
 *
 *	Any non-freespace AIX region can be converted into an EVMS
 *	compatibility volume.
 */
static int aix_can_set_volume(	storage_object_t	* region,
				boolean			flag )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_discover
 *
 *	Examine all objects and find AIX PVs. Assemble containers
 *	and export all AIX LVs as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	objects from the input list must either be claimed or moved to the
 *	output list.
 */
static int aix_discover(dlist_t	objects,
			dlist_t	regions,
			boolean	final_call )
{
	int rc;

	LOG_ENTRY;

	aix_discover_containers(objects, regions);

//	aix_check_containers();

	aix_read_vgdas();

	aix_discover_regions();

//	aix_check_regions();

	rc = aix_export_regions(regions);

	RETURN(rc);
}



/****** Region Functions ******/



/* aix_create
 *
 *	Create a new AIX region using the specified freespace.
 */
static int aix_create(	dlist_t		freespace_region_list,
			option_array_t	* options,
			dlist_t		new_region_list )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_delete
 *
 *	Remove the specified region and consolidate all of its space into
 *	the appropriate freespace region.
 */
static int aix_delete(	storage_object_t	* region,
			dlist_t			children )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_expand
 */
static int aix_expand(	storage_object_t	* region,
			storage_object_t	* expand_object,
			dlist_t			input_objects,
			option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_shrink
 */
static int aix_shrink(	storage_object_t	* region,
			storage_object_t	* shrink_object,
			dlist_t			input_objects,
			option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_move
 */
static int aix_move(	storage_object_t	* source,
			storage_object_t	* target,
			option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_set_volume
 */
static void aix_set_volume(	storage_object_t	* region,
				boolean			flag )
{
	LOG_ENTRY;
	LOG("That's nice. :)\n");
	LOG_EXIT(0);
}


/* aix_add_sectors_to_kill_list
 *
 *	The kill_sectors list contains a list of sectors that need to be zeroed
 *	during the next commit. This function is very similar to read/write.
 */
static int aix_add_sectors_to_kill_list( storage_object_t	* region,
					lsn_t			lsn,
					sector_count_t		count )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_commit_changes
 */
static int aix_commit_changes(	storage_object_t	* region,
				unsigned int		phase )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(0);
}


/* aix_get_option_count
 *
 *	Determine the type of Task that is being performed, and return
 *	the number of options that are available for that Task.
 */
static int aix_get_option_count( task_context_t * task )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_init_task
 *
 *	Determine the type of Task that is being performed, and set up the
 *	context structure with the appropriate initial values.
 */
static int aix_init_task( task_context_t * context )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_set_option
 *
 *	Determine the type of Task that is being performed. Then examine the
 *	desired option (using the index), and verify that the given value is
 *	appropriate. Reset the value if necessary and possible. Adjust other
 *	options as appropriate.
 */
static int aix_set_option(task_context_t	* context,
			u_int32_t		index,
			value_t			* value,
			task_effect_t		* effect )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_set_objects
 *
 *	Determine the type of task, and then validate that the objects on the
 *	"selected" list are valid for that task. If so, adjust the option
 *	descriptor as appropriate.
 */
static int aix_set_objects(	task_context_t	* context,
				dlist_t		declined_objects,
				task_effect_t	* effect )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_get_info
 *
 *	Return AIX-specific information about the specified region. If the
 *	name field is set, only return the "extra" information pertaining
 *	to that name.
 */
static int aix_get_info(	storage_object_t	* region,
				char			* name,
				extended_info_array_t	** info_array )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_set_info
 */
static int aix_set_info(	storage_object_t	* region,
				option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_get_plugin_info
 *
 *	Return information about the AIX plugin. There is no "extra"
 *	information about AIX, so "name" should always be NULL.
 */
static int aix_get_plugin_info(	char			* name,
				extended_info_array_t	** info_array )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_read
 *
 *	Perform a logical-to-physical remapping, and send the read down to
 *	the next plugin.
 */
static int aix_read(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	LOG_ENTRY;

	memset(buffer, 0, count << EVMS_VSECTOR_SIZE_SHIFT);

	RETURN(0);
}


/* aix_write
 *
 *	Perform a logical-to-physical remapping, and send the write down to
 *	the next plugin.
 */
static int aix_write(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_direct_plugin_communication
 */
static int aix_direct_plugin_communication(	void	* thing,
						boolean	target_kernel_plugin,
						void	* arg )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}



/****** Container Functions ******/



/* aix_can_create_container
 *
 *	Can we create a new container from this list of objects?
 */
static int aix_can_create_container( dlist_t objects )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* : aix_can_delete_container
 *
 *	Can we remove the specified container?
 */
static int aix_can_delete_container( storage_container_t * container )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_can_add_object
 */
static int aix_can_add_object(	storage_object_t	* object,
				storage_container_t	* container )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_can_remove_object
 */
static int aix_can_remove_object( storage_object_t * object )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_create_container
 *
 *	Create a new AIX container, using the specified objects.
 */
static int aix_create_container( dlist_t			objects,
				option_array_t			* options,
				storage_container_t		** container )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_add_object
 *
 *	Add the specified object to the specified (existing) container. If
 *	container is NULL, add the object to the unassigned container. The
 *	object must have already been removed from its previous location
 *	before this function is called. This function is not used to move an
 *	object from one AIX container to another AIX container. The function
 *	aix_transfer_object will be used for that purpose.
 */
static int aix_add_object(	storage_object_t	* object,
				storage_container_t	* container,
				option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_transfer_object
 *
 *	Move the specified object from its current container to the specified
 *	container. Unlike add_object, this function is used to move objects
 *	around internally to AIX.
 */
static int aix_transfer_object( storage_object_t	* object,
				storage_container_t	* target_container,
				option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_remove_object
 *
 *	Remove the specified object from its current container. Make sure the
 * 	object does not contain any allocated regions. In addition to removing
 *	it from the container, we want to kill any metadata that might have
 *	been on the object.
 */
static int aix_remove_object( storage_object_t * object )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_delete_container
 *
 *	Remove the specified container.
 */
static int aix_delete_container(storage_container_t	* container,
				dlist_t			objects_consumed )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_commit_container_changes
 *
 *	Write all of the metadata for this container to the underlying objects.
 */
static int aix_commit_container_changes( storage_container_t	* container,
					unsigned int		phase )
{
	storage_object_t	* object;
	int			copy = 0;
	int			rc, err = 0;

	LOG_ENTRY;

	if ( ! container ) {
		BUG();
		RETURN(EFAULT);
	}

	// AIX will commit all metadata during phase 1.
	if ( phase != 1 ) {
		RETURN(0);
	}

	// Does this container belong to AIX.
	if ( container->plugin != aix_plugin ) {
		LOG_WARNING("Container %s does not belong to the AIX region manager.\n", container->name);
		RETURN(EINVAL);
	}

	// Is this container dirty?
	if ( ! (container->flags & SCFLAG_DIRTY) ) {
		LOG_WARNING("Container %s is not dirty.\n", container->name);
		RETURN(0);
	}

	// Two copies of metadata to write to each PV in the container. Stop
	// the commit on any error.
	for ( copy = 0; copy < 2; copy++ ) {
		FOR_EACH(object, container->objects_consumed) {
			if ( copy == 0 ) {
				// Only one copy of the IPL and LVM.
				if (!err) err = aix_write_ipl(object);
				if (!err) err = aix_write_lvm(object);
			}

			// Write the VGSA and VGDA.
			if (!err) err = aix_write_vgsa(object, copy);
			if (!err) err = aix_write_vg_header(object, copy);
			if (!err) err = aix_write_lv_array(object, copy);
			if (!err) err = aix_write_pvs(object, copy);
			if (!err) err = aix_write_lv_names(object, copy);
			if (!err) err = aix_write_vg_trailer(object, copy);
		}
	}

	if (!err) {
		container->flags &= ~SCFLAG_DIRTY;
		LOG_DETAILS("Wrote container %s to disk.\n", container->name);
	}

	RETURN(err);
}


/* aix_get_container_info
 *
 *	Return an information array for the specified container. If the name
 *	field is set, only return the "extra" information pertaining to that
 *	name.
 */
static int aix_get_container_info(	storage_container_t	* container,
					char			* name,
					extended_info_array_t	** info_array )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


/* aix_set_container_info
 *
 *	Apply the settings of the options to the container.
 */
static int aix_set_container_info(	storage_container_t	* container,
					option_array_t		* options )
{
	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}



/* Function tables for the AIX Region Manager */
static plugin_functions_t aix_functions = {
	setup_evms_plugin		: aix_setup_evms_plugin,
	cleanup_evms_plugin		: aix_cleanup_evms_plugin,
	can_delete			: aix_can_delete,
	can_expand			: aix_can_expand,
	can_expand_by			: aix_can_expand_by,
	can_shrink			: aix_can_shrink,
	can_shrink_by			: aix_can_shrink_by,
	can_move			: aix_can_move,
	can_set_volume			: aix_can_set_volume,
	discover			: aix_discover,
	create				: aix_create,
	delete				: aix_delete,
	expand				: aix_expand,
	shrink				: aix_shrink,
	move				: aix_move,
	set_volume			: aix_set_volume,
	add_sectors_to_kill_list	: aix_add_sectors_to_kill_list,
	commit_changes			: aix_commit_changes,
	get_option_count		: aix_get_option_count,
	init_task			: aix_init_task,
	set_option			: aix_set_option,
	set_objects			: aix_set_objects,
	get_info			: aix_get_info,
	set_info			: aix_set_info,
	get_plugin_info			: aix_get_plugin_info,
	read				: aix_read,
	write				: aix_write,
	direct_plugin_communication	: aix_direct_plugin_communication
};


static container_functions_t aix_container_functions = {
	can_create_container		: aix_can_create_container,
	can_delete_container		: aix_can_delete_container,
	can_add_object			: aix_can_add_object,
	can_remove_object		: aix_can_remove_object,
	create_container		: aix_create_container,
	add_object			: aix_add_object,
	transfer_object			: aix_transfer_object,
	remove_object			: aix_remove_object,
	delete_container		: aix_delete_container,
	commit_container_changes	: aix_commit_container_changes,
	get_container_info		: aix_get_container_info,
	set_container_info		: aix_set_container_info
};


plugin_record_t AIX_Plugin = {
	id				: SetPluginID( EVMS_OEM_IBM, EVMS_REGION_MANAGER, 0x03),
	version				: {
		major			: MAJOR_VERSION,
		minor			: MINOR_VERSION,
		patchlevel		: PATCH_LEVEL
	},
	required_api_version		: {
		major			: 3,
		minor			: 0,
		patchlevel		: 0
	},
	short_name			: "AixRegMgr",
	long_name			: "AIX Region Manager",
	oem_name			: "IBM",
	functions			: {
		plugin			: &aix_functions
	},
	container_functions 		: &aix_container_functions
};

plugin_record_t * evms_plugin_records[] = {
	&AIX_Plugin,
	NULL
};

