/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: linear_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Linear MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#define MY_PLUGIN linear_plugin
#include "md.h"
#include "linear_discover.h"



// Global variables
static engine_mode_t		open_mode;		// Still don't know what this is for. :)



/* Function: linear_setup_evms_plugin
 *
 *	This function gets called shortly after the plugin is loaded by the
 *	Engine. It performs all tasks that are necessary before the initial
 *	discovery pass.
 */
static int linear_setup_evms_plugin(engine_mode_t		mode,
				engine_functions_t	* functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	open_mode = mode;
	EngFncs = functions;

	LOG_ENTRY;
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	RETURN(rc);
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: linear_can_delete
 *
 *	Can we remove the specified MD logical volume, and consolidate the
 *	space with the freespace volume?
 */
static int linear_can_delete( storage_object_t * region )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: linear_can_expand
 *
 *	Can this region be expanded? If so,
 *	add the region to the expansion_points output list. else, just
 *      return 0 to allow those above us to do as they please
 */
static int linear_can_expand(storage_object_t	* region,
			     u_int64_t		* expand_limit,
			     dlist_t		expansion_points )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: linear_can_expand_by
 *
 *	Can we expand the specified region, by the specified size, using space
 *	from the freespace volume in the same group?
 */

static int linear_can_expand_by(storage_object_t	* child_object,
				u_int64_t		* size )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_can_shrink
 *
 *	Just like can_expand, but in the opposite direction.
 */
static int linear_can_shrink(storage_object_t	* region,
			u_int64_t		* shrink_limit,
			dlist_t			shrink_points )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: linear_can_shrink_by
 *
 *	Can we shrink the specified logical volume and consolidate the removed
 *	space into the freespace volume? If it can be shrunk, but not to
 *	exactly new_size, reset new_size to the next lower possible size.
 */
static int linear_can_shrink_by(storage_object_t	* region,
				u_int64_t		* size )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_can_move
 *
 */
static int linear_can_move( storage_object_t * region )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_can_set_volume
 *
 * Always yes
 */
static int linear_can_set_volume(storage_object_t	* region,
				boolean			flag )
{
	LOG_ENTRY;
	RETURN(0);
}


/* Function: linear_discover
 *
 *	Examine all disk segments and find MD PVs. Assemble volume groups
 *	and export all MD logical volumes as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	segments from the input list must either be claimed or moved to the
 *	output list.
 */
static int linear_discover(dlist_t	input_list,
           		   dlist_t	output_list,
			   boolean	final_call )
{
	int count = 0;

	LOG_ENTRY;

	md_discover_volumes(input_list, output_list);
	LOG_DETAILS("Object discovery complete.\n");

	// LV discovery and exporting
	linear_discover_regions(output_list, &count, final_call);
	LOG_DETAILS("Object creation complete.\n");

	RETURN(count);
}



/****** Region Functions ******/



/* Function: linear_create
 *
 *	Create a new MD volume.
 */
static int linear_create(dlist_t       	objects,
			option_array_t	* options,
			dlist_t		new_region_list )
{
	md_volume_t	* volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int tag, waste;
	int i, index = 0;
	int rc = 0;
	mdp_disk_t disk;

	LOG_ENTRY;

	if ( md_allocate_memory((void**)&volume, sizeof(md_volume_t) ) ){
		LOG_CRITICAL("Memory error new volume structure.\n");
		RETURN(ENOMEM);
	}

	GetListSize(objects, &nr_disks);
	if (nr_disks > MAX_MD_DEVICES) {
		LOG_CRITICAL("Too many objects for MD Linear create %d.\n",nr_disks);
		RETURN(EINVAL);
	}
	while (!(rc = BlindExtractObject(objects, &waste, (TAG *)&tag, NULL, (void *)&object))) {
		size = min(size, object->size);  // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
	}
	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_NEW_SIZE_BLOCKS(size/2); // first convert sectors to blocks

	md_create_first_superblock(volume, disk, pers_to_level(LINEAR), 4, size, nr_disks, 0, (1 << MD_SB_CLEAN));

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
	}

	volume->personality = LINEAR;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;

	rc = linear_create_region(volume, new_region_list, TRUE);
	RETURN(rc);
}


/* Function: linear_delete
 *
 *	Remove the specified region
 */
static int linear_delete(storage_object_t	* region,
			 dlist_t		children )
{
	md_volume_t	* volume;
	int		rc = 0;

	LOG_ENTRY;

	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume);
	EngFncs->free_region(region);

	RETURN(rc);
}


/* Function: linear_expand
 */

static int linear_expand(storage_object_t	* region,
			 storage_object_t	* expand_object,
			 dlist_t       		input_objects,
			 option_array_t		* options )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_shrink
 */
static int linear_shrink(storage_object_t	* region,
			storage_object_t	* shrink_object,
			dlist_t			input_objects,
			option_array_t		* options )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_move
 */
static int linear_move(	storage_object_t	* source,
			storage_object_t	* target,
			option_array_t		* options )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: linear_set_volume
 *
 *	MD doesn't really care if its regions are made into volumes.
 */
static void linear_set_volume(	storage_object_t	* region,
				boolean			flag )
{
	LOG_ENTRY;
	LOG_EXIT(0);
}


/* Function: linear_add_sectors_to_kill_list
 *
 *	The kill_sectors list contains a list of sectors that need to be zeroed
 *	during the next commit. This function is very similar to read/write.
 */
static int linear_add_sectors_to_kill_list( storage_object_t	* region,
					lsn_t			lsn,
					sector_count_t		count )
{

	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_kill = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 			i;
	int			rc = 0;

	LOG_ENTRY;


	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect \n ",volume->name);
		RETURN(EIO);
	}

        for (i=0; i < volume->nr_disks; i++) {

		current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
                if ( current_end >= io_lsn ) {

                    max_sector_count = current_end - io_lsn + 1;

                    if ( max_sector_count >= sectors_left_to_kill ) {
                        io_sector_count = sectors_left_to_kill;
                    }
                    else {
                        io_sector_count = max_sector_count;
                    }

                    link_lsn = io_lsn - current_start;

                    rc = KILL_SECTORS( volume->child_object[i],
                               link_lsn,
                               io_sector_count);

                    io_lsn               += io_sector_count;
                    sectors_left_to_kill -= io_sector_count;

                    if ((sectors_left_to_kill == 0) || (rc)) break;
                }
		current_start = current_end;

        }
	RETURN(rc);
}


/* Function: linear_commit_changes
 *
 */
static int linear_commit_changes( storage_object_t * region, uint phase )
{
	md_volume_t	* volume = (md_volume_t *)region->private_data;
	int		rc = 0;

	LOG_ENTRY;

	if ( ! (region->flags & SOFLAG_DIRTY) ) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		RETURN(0);
	}

	switch (phase) {
	case 1:
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;  // mark clean
	default:
	}

	RETURN(rc);
}


/* Function: linear_get_option_count
 *
 *	Determine the type of Task that is being performed, and return
 *	the number of options that are available for that Task.
 */
static int linear_get_option_count( task_context_t * task )
{
	int count = 0;

	LOG_ENTRY;

	switch(task->action) {
	case EVMS_Task_Create:
		count = 0;
		break;
	default:
		count = 0;
		break;
	}

	RETURN(count);
}


/* Function: linear_init_task
 *
 *	Determine the type of Task that is being performed, and set up the
 *	context structure with the appropriate initial values.
 */
static int linear_init_task( task_context_t * context )
{
	int rc = 0;
	dlist_t tmp_list;

	LOG_ENTRY;

	switch(context->action) {

	case EVMS_Task_Create:

		context->option_descriptors->count = 0;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					DATA_TYPE,
					NULL,
					VALID_INPUT_OBJECT,
					&tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		DestroyList(&tmp_list, FALSE);
       		context->min_selected_objects = 1;
       		context->max_selected_objects = MAX_MD_DEVICES;
		break;

	default:
		break;
	}

	RETURN(rc);
}


/* Function: linear_set_option
 *
 *	Determine the type of Task that is being performed. Then examine the
 *	desired option (using the index), and verify that the given value is
 *	appropriate. Reset the value if necessary and possible. Adjust other
 *	options as appropriate.
 */
static int linear_set_option(task_context_t	* context,
			u_int32_t		index,
			value_t			* value,
			task_effect_t		* effect )
{
	int rc = 0;

	LOG_ENTRY;

	*effect = 0;

	switch (context->action) {

	case EVMS_Task_Create:
		// no options, just return 0
		break;

	default:
		break;
	}
	RETURN(rc);
}


/* Function: linear_set_objects
 *
 *	Determine the type of task, and then validate that the objects on the
 *	"selected" list are valid for that task. If so, adjust the option
 *	descriptor as appropriate.
 */
static int linear_set_objects(	task_context_t	* context,
				dlist_t		declined_objects,
				task_effect_t	* effect )
{
	int rc = 0;

	LOG_ENTRY;

	switch(context->action) {

	case EVMS_Task_Create:
		// since this is within task context, and we provided the initial list
		// do we need any further validation here?
		rc = 0;
		break;

	default:
		break;
	}
	RETURN(rc);
}


/* Function: linear_get_info
 *
 *	Return MD-specific information about the specified region. If the
 *	name field is set, only return the "extra" information pertaining
 *	to that name.
 */
static int linear_get_info(	storage_object_t	* region,
				char			* name,
				extended_info_array_t	** info_array )
{
	md_volume_t * volume = NULL;
	int           rc= 0;

	LOG_ENTRY;

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

        RETURN(rc);
}


/* Function: linear_get_plugin_info
 *
 *	Return information about the MD plugin. There is no "extra"
 *	information about MD, so "name" should always be NULL.
 */
static int linear_get_plugin_info(char			* name,
				  extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[50] = {0};
	int			i = 0;
	int rc = 0;
	
	LOG_ENTRY;

	// Parameter check
	if ( ! info_array ) {
		RETURN(EFAULT);
	}

	if ( ! name ) {
		// Get memory for the info array
		if ( ! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*4)) ) {
			LOG_ERROR("Error allocating memory for info array\n");
			RETURN(ENOMEM);
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, linear_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, linear_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required API Version
		SET_STRING(info->info[i].name, "Required_Version");
		SET_STRING(info->info[i].title, "Required Plugin API Version");
		SET_STRING(info->info[i].desc, "This is the version of the engine that the plugin requires.It will not run on older versions of the Engine.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", linear_plugin->required_api_version.major, linear_plugin->required_api_version.minor, linear_plugin->required_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	}
	else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	info->count = i;
	*info_array = info;
	RETURN(0);
}


/* Function: linear_read
 *
 *	Perform a logical-to-physical remapping, and send the read down to
 *	the next plugin.
 */
static int linear_read(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_read = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 			i;
	int			rc = 0;

	LOG_ENTRY;

	// to be in sync with the kernel MD driver we are not supporting
	// partial exports at this time.  Use EVMS Drive linking instead.
	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n ",volume->name);
		RETURN(0);
	}

        for (i=0; i < volume->nr_disks; i++) {

		current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
                if ( current_end >= io_lsn ) {

                    max_sector_count = current_end - io_lsn + 1;

                    if ( max_sector_count >= sectors_left_to_read ) {
                        io_sector_count = sectors_left_to_read;
                    }
                    else {
                        io_sector_count = max_sector_count;
                    }

                    link_lsn = io_lsn - current_start;

                    rc = READ( volume->child_object[i],
                               link_lsn,
                               io_sector_count,
                               (void *) io_buffer_ptr );

                    io_lsn               += io_sector_count;
                    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
                    sectors_left_to_read -= io_sector_count;

                    if ((sectors_left_to_read == 0) || (rc)) break;
                }
		current_start = current_end;

        }
	RETURN(rc);
}


/* Function: linear_write
 *
 *	Perform a logical-to-physical remapping, and send the write down to
 *	the next plugin.
 */
static int linear_write(storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_write = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 		 	  i;
	int			  rc = 0;

	LOG_ENTRY;

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD region %s is corrupt, writing data is not allowed\n ",volume->name);
		RETURN(EIO);
	}

        for (i=0; i < volume->nr_disks; i++) {

		current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
                if ( current_end >= io_lsn ) {

                    max_sector_count = current_end - io_lsn + 1;

                    if ( max_sector_count >= sectors_left_to_write ) {
                        io_sector_count = sectors_left_to_write;
                    }
                    else {
                        io_sector_count = max_sector_count;
                    }

                    link_lsn = io_lsn - current_start;

                    rc = WRITE( volume->child_object[i],
                               link_lsn,
                               io_sector_count,
                               (void *) io_buffer_ptr );

                    io_lsn               += io_sector_count;
                    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
                    sectors_left_to_write -= io_sector_count;

                    if ((sectors_left_to_write == 0) || (rc)) break;
                }
		current_start = current_end;

        }
	RETURN(rc);
}


/* Function: linear_direct_plugin_communication
 */
static int linear_direct_plugin_communication(	void	* thing,
						boolean	target_kernel_plugin,
						void	* arg )
{
	LOG_ENTRY;
	RETURN(ENOSYS);
}


static int free_region (ADDRESS object,
			TAG     object_tag,
			uint    object_size,
			ADDRESS object_handle,
			ADDRESS parameters) {

	int i;
	int nr_disks = 0;
	storage_object_t * region = (storage_object_t *) object;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	for (i = 0; (i < MAX_MD_DEVICES) && (nr_disks < volume->nr_disks); i++) {
		if (volume->child_object[i]) {
			nr_disks++;
			md_deallocate_memory(volume->super_array[i]);
		}
	}
	md_remove_volume_from_list(volume);
	md_deallocate_memory(volume);

	RETURN(DLIST_SUCCESS);
}



void linear_plugin_cleanup(void) {

	dlist_t regions_list;

	LOG_ENTRY;

	EngFncs->get_object_list(REGION, DATA_TYPE, linear_plugin, 0, &regions_list);

	ForEachItem(regions_list, free_region, NULL, TRUE);

	DestroyList(&regions_list, FALSE);

	LOG_EXIT(0);
	return;
}


/* Function tables for the MD Region Manager */
static plugin_functions_t linear_functions = {
	cleanup_evms_plugin		: linear_plugin_cleanup,
	setup_evms_plugin		: linear_setup_evms_plugin,
	can_delete			: linear_can_delete,
	can_expand			: linear_can_expand,
	can_expand_by			: linear_can_expand_by,
	can_shrink			: linear_can_shrink,
	can_shrink_by			: linear_can_shrink_by,
	can_move			: linear_can_move,
	can_set_volume			: linear_can_set_volume,
	discover			: linear_discover,
	create				: linear_create,
	delete				: linear_delete,
	expand				: linear_expand,
	shrink				: linear_shrink,
	move				: linear_move,
	set_volume			: linear_set_volume,
	add_sectors_to_kill_list	: linear_add_sectors_to_kill_list,
	commit_changes			: linear_commit_changes,
	get_option_count		: linear_get_option_count,
	init_task			: linear_init_task,
	set_option			: linear_set_option,
	set_objects			: linear_set_objects,
	get_info			: linear_get_info,
	get_plugin_info			: linear_get_plugin_info,
	read				: linear_read,
	write				: linear_write,
	direct_plugin_communication	: linear_direct_plugin_communication
};



/* Function: PluginInit
 *
 *	Initializes the local plugin record
 */

plugin_record_t linear_plugin_record = {
    id:                    SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 4),

    version:              {major:      MAJOR_VERSION,
                           minor:      MINOR_VERSION,
                           patchlevel: PATCH_LEVEL},

    required_api_version: {major:      3,
                           minor:      0,
                           patchlevel: 0},

    short_name:            "MDLinearRegMgr",
    long_name:             "MD Linear Raid Region Manager",
    oem_name:              "IBM",

    functions:             {plugin:   &linear_functions},

    container_functions:   NULL
};
					
