/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid0_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of raid0 MD physical volumes and logical
 *              volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#define MY_PLUGIN raid0_plugin
#include "md.h"
#include "raid0_mgr.h"

static int create_strip_zones (md_volume_t * volume)
{
	int rc;
	int i, c, j, j1, j2, cur;
	u_int64_t current_offset;
	u_int64_t curr_zone_offset;
	u_int64_t size;
	u_int64_t zone0_size;
	raid0_conf_t * conf = mdvol_to_conf(volume);
	storage_object_t * child_object;
	storage_object_t * child_object1;
	storage_object_t * child_object2;
	storage_object_t * smallest_child;
	unsigned int chunk_size_in_sectors;

	LOG_ENTRY;

	// if this volume is corrupt, can't build the stripes correctly, so just return
	if (volume->flags & MD_CORRUPT) {
		RETURN(0);
	}

	chunk_size_in_sectors = volume->super_block->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
	/*
	 * The number of 'same size groups'
	 */

	conf->nr_strip_zones = 0;

	for (j1 = 0; j1 < volume->nr_disks; j1++) {
		child_object1 = volume->child_object[j1];
		LOG_DEBUG("Looking at %s\n", child_object1->name);
		c = 0;

		for (j2 = 0; j2 < volume->nr_disks; j2++) {
			child_object2 = volume->child_object[j2];
			LOG_DEBUG("Comparing %s(%lld) with %s(%lld)\n", child_object1->name, MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object1->size), child_object2->name, MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object2->size));
			if (child_object2 == child_object1) {
				LOG_DEBUG("  END\n");
				break;
			}
			if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object2->size) == MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object1->size)) {
				/*
				 * Not unique, dont count it as a new
				 * group
				 */
				LOG_DEBUG("  EQUAL\n");
				c = 1;
				break;
			}
			LOG_DEBUG("  NOT EQUAL\n");
		}
		if (!c) {
			LOG_DEBUG("  ==> UNIQUE\n");
			conf->nr_strip_zones++;
			LOG_DEBUG("  %d zones\n", conf->nr_strip_zones);
		}
	}
	LOG_DEBUG("FINAL %d zones\n", conf->nr_strip_zones);

	rc = md_allocate_memory((void **) &conf->strip_zone, sizeof(struct strip_zone) * conf->nr_strip_zones);

	if (rc != 0) {
		LOG_CRITICAL("Error %d allocating memory for strip zone structures.\n", rc);
		RETURN(rc);
	}

	conf->smallest_zone = NULL;
	current_offset = 0;
	curr_zone_offset = 0;

	for (i = 0; i < conf->nr_strip_zones; i++) {
		struct strip_zone *zone = conf->strip_zone + i;

		LOG_DEBUG("Zone %d\n", i);
		zone->dev_offset = current_offset;
		smallest_child = NULL;
		c = 0;

		for (j = 0; j < volume->nr_disks; j++) {

			child_object = volume->child_object[j];
			LOG_DEBUG("  checking %s ...\n", child_object->name);
			if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object->size) > current_offset) {
				LOG_DEBUG("  contained as device %d\n", c);
				zone->dev[c] = child_object;
				c++;
				if (!smallest_child || (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object->size) < MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, smallest_child->size))) {
					smallest_child = child_object;
					LOG_DEBUG("  (%lld) is smallest!.\n", MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, child_object->size));
				}
			} else {
				LOG_DEBUG("  nope.\n");
			}
		}

		zone->nb_dev = c;
		zone->size = (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, smallest_child->size) - current_offset) * c;
		LOG_DEBUG("zone->nb_dev: %d, size: %lld\n", zone->nb_dev, zone->size);

		if (!conf->smallest_zone || (zone->size < conf->smallest_zone->size))
			conf->smallest_zone = zone;

		zone->zone_offset = curr_zone_offset;
		curr_zone_offset += zone->size;

		current_offset = MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, smallest_child->size);
		LOG_DEBUG("current zone offset: %lld\n", current_offset);
	}

	conf->nr_zones = (volume->region->size + conf->smallest_zone->size - 1) / conf->smallest_zone->size;
	LOG_DEBUG("Number of zones is %d.\n", conf->nr_zones);

	/* Set up the hash tables. */
	cur = 0;

	rc = md_allocate_memory((void **) &conf->hash_table, sizeof (struct raid0_hash) * conf->nr_zones);
	if (rc != 0) {
		LOG_CRITICAL("Error %d allocating memory for zone hash table.\n", rc);
		RETURN(rc);
	}
	size = conf->strip_zone[cur].size;

	i = 0;
	while (cur < conf->nr_strip_zones) {
		conf->hash_table[i].zone0 = conf->strip_zone + cur;

		/*
		 * If we completely fill the slot
		 */
		if (size >= conf->smallest_zone->size) {
			conf->hash_table[i++].zone1 = NULL;
			size -= conf->smallest_zone->size;

			/*
			 * If there is nothing left in the strip zone,
			 * move to the next srip zone.  Else, the
			 * next iteration of the loop will hit the
			 * code below where zone1 is filled in for this
			 * hash entry.
			 */
			if (!size) {
				if (++cur == conf->nr_strip_zones)
					continue;
				size = conf->strip_zone[cur].size;
			}
			continue;
		}
		if (++cur == conf->nr_strip_zones) {
			/*
			 * Last dev, set unit1 as NULL
			 */
			conf->hash_table[i].zone1=NULL;
			continue;
		}

		/*
		 * Here we use a 2nd dev to fill the slot
		 */
		zone0_size = size;
		size = conf->strip_zone[cur].size;
		conf->hash_table[i++].zone1 = conf->strip_zone + cur;
		size -= conf->smallest_zone->size - zone0_size;
	}

	RETURN(0);
}


int raid0_create_region(md_volume_t * volume, dlist_t output_list, boolean final_call){
	int rc = 0;
	storage_object_t * region;
	int found = 0;
	int i, j = -1;
	unsigned int chunk_size_in_sectors;

	LOG_ENTRY;

	if ((!volume->super_array[0] || (volume->nr_disks !=  volume->super_array[0]->nr_disks)) &&
	    !final_call) {
		LOG_DETAILS("Region %s is missing members, delaying discovery\n",volume->name);
		RETURN(0);
	}

	LOG_DETAILS("Discovered region %s.\n",volume->name);
	if ((rc = EngFncs->allocate_region(volume->name, &region))) {
		for (j = MAX_MD_DEVICES -1;(rc != 0) && (j >=0) ; j--) {
			sprintf(volume->name, "md/md%d",j);
			rc = EngFncs->allocate_evms_object(volume->name, &region);
		}
		if (j<0) {
			LOG_ERROR("No more names for MD ");
			RETURN(ENOMEM);
		}
	}
	volume->region = region;
	region->size = 0;  // initialize for size calculation
	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++) {
		// check for null object, if missing, skip and set corrupt flag
		if (volume->child_object[i]) {
			chunk_size_in_sectors = volume->super_array[i]->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
			// if name registration failed and we changed the name, fix up all the minor numbers
			if (j >= 0) {
				volume->super_array[i]->md_minor = j;
			}
			md_append_region_to_object(region, volume->child_object[i]);
			LOG_DETAILS("Adding Object %s to %s\n",volume->child_object[i]->name,volume->name);
			region->size += MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, volume->child_object[i]->size);
			found++;
		} else {
			LOG_ERROR("Volume %s is corrupt, missing member object %d\n",volume->name,i);
			MESSAGE("Volume %s is corrupt, missing member object %d\n",volume->name,i);
			volume->flags |= MD_CORRUPT;
		}
	}

	rc = md_allocate_memory(&volume->private_data, sizeof (raid0_conf_t));
	if (rc == 0) {
		rc = create_strip_zones(volume);
	} else {
		LOG_CRITICAL("Error %d allocating memory for raid 0 configuration structure.\n", rc);
		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;
	}
	region->data_type = DATA_TYPE;
	region->plugin = raid0_plugin;
	region->private_data = (void *)volume;
	volume->flags |= MD_DISCOVERED;
	volume->region = region;
	md_add_object_to_list(region, output_list);
	RETURN(rc);
}




/* Function: discover_regions
 *
 *	run the global list of regions and pirce them together.
 */
int raid0_discover_regions( dlist_t output_list, int *count, boolean final_call )
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	LOG_ENTRY;

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == RAID0)) {
			rc = raid0_create_region(volume, output_list, final_call);
			if (volume->flags & MD_DISCOVERED) {
				*count = *count + 1;
			}
		}
		volume = volume->next;
	}



	RETURN(rc);
}
