/*
 *
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: copy.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <pthread.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <malloc.h>

#include "fullengine.h"
#include "engine.h"
#include "copy.h"
#include "internalAPI.h"
#include "memman.h"
#include "message.h"
#include "dm.h"
#include "dm-ioctl.h"
#include "lists.h"


#define COPY_NAME_PREFIX        "Copy_"
#define COPY_NAME_PREFIX_LEN    (sizeof(COPY_NAME_PREFIX) - 1)
#define COPY_NAME_INFIX         "_to_"
#define COPY_NAME_INFIX_LEN     (sizeof(COPY_NAME_INFIX) - 1)


/*********************************\
* Functions for handling progress *
\*********************************/


static int copy_progress_update(copy_job_t * copy_job) {

	int rc;
	char * info;
	int count;
	int nr_mirrors;
	int src_major;
	int src_minor;
	int trg_major;
	int trg_minor;
	unsigned long sync_count;
	unsigned long nr_regions;

	LOG_PROC_ENTRY();

	if (copy_job->mirror == NULL) {
		LOG_ERROR("Copy job \"%s\" does not have a mirror object.\n", copy_job->title);
		LOG_PROC_EXIT_INT(ENOENT);
		return ENOENT;
	}

	if (copy_job->flags & COPY_FINISHED) {
		/* Progress doesn't change after it's finished. */
		LOG_PROC_EXIT_INT(0);
		return 0;
	}

	rc = dm_get_info_v4(copy_job->mirror->name, &info);

	if (rc == 0) {
		count = sscanf(info, "%d %d:%d %d:%d %lu/%lu",
			       &nr_mirrors,
			       &src_major,  &src_minor,
			       &trg_major,  &trg_minor,
			       &sync_count, &nr_regions);

		if (count < 7) {
			LOG_WARNING("Scanned %d variables from string \"%s\".  Need 7 variables.\n", count, info);
			rc = ENODATA;
		}

		engine_free(info);

	} else {
		LOG_DETAILS("dm_get_info_v4() returned error code %d: %s\n", rc, evms_strerror(rc));
	}

	pthread_mutex_lock(&copy_job->progress_mutex);
	if (rc == 0) {
		copy_job->progress.total_count = nr_regions;
		copy_job->progress.count = sync_count;

		if (sync_count > 0) {
			copy_job->flags |= COPY_STARTED;
		}

		if (sync_count >= nr_regions) {
			copy_job->flags |= COPY_FINISHED;
		}
	}

//	calculate_time_estimate(&copy_job->progress);
	pthread_mutex_unlock(&copy_job->progress_mutex);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void initialize_copy_progress(copy_job_t * copy_job,
				     u_int64_t    total_count) {

	progress_t * progress = &copy_job->progress;

	LOG_PROC_ENTRY();

	memset(progress, 0, sizeof(*progress));

	progress->title       = copy_job->title;
	progress->description = copy_job->description;
	progress->type        = DISPLAY_PERCENT;
	progress->total_count = total_count;
	pthread_mutex_init(&copy_job->progress_mutex, NULL);

	LOG_PROC_EXIT_VOID();
	return;
}


static void report_copy_progress(copy_job_t * copy_job) {

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&copy_job->progress_mutex);
	if (copy_job->progress.type != INDETERMINATE) {

		/*
		 * If the copy_progress_thread() is not running, then we must
		 * update the time estimate ourselves.
		 */
		if (copy_job->progress_tid == 0) {
//			calculate_time_estimate(&copy_job->progress);
		}
	}

	plugin_progress(&copy_job->progress);
	pthread_mutex_unlock(&copy_job->progress_mutex);

	LOG_PROC_EXIT_VOID();
	return;
}


static void final_copy_progress(copy_job_t * copy_job) {

	progress_t * progress = &copy_job->progress;

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&copy_job->progress_mutex);
	if (progress->count < progress->total_count) {
		progress->count = progress->total_count;
		progress->remaining_seconds = 0;

		plugin_progress(&copy_job->progress);
	}
	pthread_mutex_unlock(&copy_job->progress_mutex);

	engine_free(progress->plugin_private_data);
	progress->plugin_private_data = NULL;

	LOG_PROC_EXIT_VOID();
	return;
}


/**********************************\
* Functions for doing offline copy *
\**********************************/


static int local_copy(copy_job_t * copy_job) {

	int rc = 0;
	void * copy_buffer;
	lsn_t src_lsn = copy_job->src.start;
	lsn_t trg_lsn = copy_job->trg.start;
	sector_count_t len = min(copy_job->src.len, copy_job->trg.len);
	u_int64_t chunk_size = min(len, 65536 >> EVMS_VSECTOR_SIZE_SHIFT);
	sector_count_t sectors_copied = 0;

	LOG_PROC_ENTRY();
	LOG_DEBUG("Source:	  %s\n", copy_job->src.obj->name);
	LOG_DEBUG("Source offset: %"PRIu64"\n", src_lsn);
	LOG_DEBUG("Target:	  %s\n", copy_job->trg.obj->name);
	LOG_DEBUG("Target offset: %"PRIu64"\n", trg_lsn);
	LOG_DEBUG("Length:        %"PRIu64"\n", len);

	/*
	 * We don't use engine_alloc() to get the copy buffer because we want
	 * to try and get an aligned buffer.  engine_alloc() can't align a
	 * buffer.
	 *
	 * The largest alignment factor is 4KB.  Get a buffer that is aligned on
	 * a 4KB boundary in hopes that the Local Disk Manager can use the
	 * buffer directly instead of having to double buffer.
	 */
	copy_buffer = memalign(4096, chunk_size << EVMS_VSECTOR_SIZE_SHIFT);
	if (copy_buffer == NULL) {
		/* Try plain old unaligned malloc(). */
		copy_buffer = malloc(chunk_size << EVMS_VSECTOR_SIZE_SHIFT);

		if (copy_buffer == NULL) {
			LOG_CRITICAL("Unable to get memory for a copy buffer.\n");
			LOG_PROC_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}

	if (copy_job->title != NULL) {
		initialize_copy_progress(copy_job, (len + chunk_size - 1) / chunk_size);
		report_copy_progress(copy_job);
	}

	while ((rc == 0) && (sectors_copied < len)) {
		sector_count_t sectors_to_copy = min(chunk_size, len - sectors_copied);

		rc = copy_job->src.obj->plugin->functions.plugin->read(copy_job->src.obj, src_lsn, sectors_to_copy, copy_buffer);

		if (rc == 0) {
			rc = copy_job->trg.obj->plugin->functions.plugin->write(copy_job->trg.obj, trg_lsn, sectors_to_copy, copy_buffer);
		}

		src_lsn += sectors_to_copy;
		trg_lsn += sectors_to_copy;
		sectors_copied += sectors_to_copy;
		copy_job->progress.count ++;

		if ((rc == 0) && (copy_job->title != NULL)) {
			report_copy_progress(copy_job);
		}
	}

	if (copy_job->title != NULL) {
		final_copy_progress(copy_job);
	}

	LOG_DEBUG("Total sectors copied: %"PRIu64"\n", sectors_copied);

	/* Don't use engine_free() since the buffer wasn't engine_alloc()ed. */
	free(copy_buffer);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_offline_copy(copy_job_t * copy_job) {

	int rc = 0;

	struct timezone tz;
	struct timeval copy_start_time;
	struct timeval copy_stop_time;
	struct timeval copy_time;
	int copy_hours;
	int copy_minutes;

	LOG_PROC_ENTRY();

	gettimeofday(&copy_start_time, &tz);

        rc = local_copy(copy_job);

	gettimeofday(&copy_stop_time, &tz);

	copy_time.tv_sec = copy_stop_time.tv_sec - copy_start_time.tv_sec;
	copy_time.tv_usec = copy_stop_time.tv_usec - copy_start_time.tv_usec;
	if (copy_time.tv_usec < 0) {
		copy_time.tv_sec --;
		copy_time.tv_usec += 1000000;
	}

	copy_hours = copy_time.tv_sec / 3600;
	copy_time.tv_sec %= 3600;
	copy_minutes = copy_time.tv_sec / 60;
	copy_time.tv_sec %= 60;

	LOG_DEFAULT("Copy time: %02d:%02d:%02ld.%06ld\n", copy_hours, copy_minutes, copy_time.tv_sec, copy_time.tv_usec);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*********************************\
* Functions for doing online copy *
\*********************************/


static int online_copy = -1;

boolean engine_can_online_copy(void) {

	LOG_PROC_ENTRY();

	/* Only check once.  The answer will never change. */
	if (online_copy == -1) {
		int rc;
		storage_object_t * disk;
		storage_object_t * mirror;

		online_copy = 0;

		if (dm_get_version() != 4) {
			LOG_DETAILS("Online copy requires version 4 of the device-mapper ioctl.\n");
			return online_copy;
		}

		disk = first_thing(&disks_list, NULL);
		if (disk == NULL) {
			LOG_DETAILS("Can't get a disk for testing if device-mapper raid 1 is available.\n");
			return online_copy;
		}

		rc = allocate_new_storage_object(&mirror);
		if (rc == 0) {
			strcpy(mirror->name, "Test-mirror");
			mirror->size = 512;
			mirror->object_type = SEGMENT;

			rc = dm_create(mirror);
			if (rc == 0) {
				dm_target_t * target;
				

				target = dm_allocate_target(DM_TARGET_MIRROR, 0, mirror->size, 2, 0);

				if (target != NULL) {
					target->data.mirror->num_mirrors = 2;
					target->data.mirror->chunk_size = 32;
					target->data.mirror->persistent = FALSE;
					target->data.mirror->devices[0].major = disk->dev_major;
					target->data.mirror->devices[0].minor = disk->dev_minor;
					target->data.mirror->devices[0].start = 0;
					target->data.mirror->devices[1].major = disk->dev_major;
					target->data.mirror->devices[1].minor = disk->dev_minor;
					target->data.mirror->devices[1].start = 0;

					rc = dm_load_targets(mirror, target);
					if (rc == 0) {
						online_copy = 1;
					}

					dm_deallocate_targets(target);

				} else {
					LOG_SERIOUS("Error allocating a target list.\n");
				}

				dm_deactivate(mirror);

			} else {
				LOG_SERIOUS("Failed to create device-mapper mapping for mirror.  Error code is %d: %s.\n", rc, evms_strerror(rc));
			}

			free_old_storage_object(mirror);

		} else {
			LOG_SERIOUS("Error allocating a new storage object for the mirror: %d: %s\n", rc, evms_strerror(rc));
		}

		LOG_PROC_EXIT_BOOLEAN(online_copy);
		return online_copy;
	}

	LOG_PROC_EXIT_BOOLEAN(online_copy);
	return online_copy;
}


/*
 * This thread is launched when a copy job is started or when a copy job is
 * setup and already has an active mirror.
 * The thread wakes up every 1/10 second, and updates the progress in the
 * copy job.
 */
static void * copy_progress_thread(void * arg) {

	copy_job_t * copy_job = (copy_job_t *) arg;
	int old_cancel_state;
	int rc;

	/*
	 * Don't allow the thread to be canceled unless it is sleeping.  There
	 * is a very real risk that the thread can be canceled in the middle
	 * of writing to the log file while the log mutex lock is held.  Bad
	 * things happen if the thread leaves while holding the log lock.
	 */
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_cancel_state);

	LOG_PROC_ENTRY();

	do {
		rc = copy_progress_update(copy_job);

		if (rc == 0) {

			if (!(copy_job->flags & COPY_FINISHED)) {

				/*
				 * The thread can handle being canceled at this point.
				 */
				pthread_setcancelstate(old_cancel_state, NULL);
				pthread_testcancel();

				usleep(100000);

				/* Here, too. */
				pthread_testcancel();
				pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_cancel_state);
			}
		}

	} while ((rc == 0) && (!(copy_job->flags & COPY_FINISHED)));

	/* This thread is going away. */
	copy_job->progress_tid = 0;

	LOG_PROC_EXIT_PTR(NULL);

	pthread_setcancelstate(old_cancel_state, NULL);

	return NULL;
}


static int find_copy_mirror_device(copy_job_t * copy_job, storage_object_t * mirror) {

	int rc = 0;
	dm_device_list_t * device_list;
	dm_device_list_t * device;
	dm_target_t * target;

	LOG_PROC_ENTRY();

	rc = dm_get_devices(&device_list);

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	for (rc = ENOENT, device = device_list;
	     (device != NULL) && rc != 0;
	     device = device ->next) {
		if (strncmp(device->name, COPY_NAME_PREFIX, COPY_NAME_PREFIX_LEN) != 0) {
			continue;
		}

		strcpy(mirror->name, device->name);

		if (dm_get_targets(mirror, &target) == 0) {
			if (target->type == DM_TARGET_MIRROR) {

				if ((target->data.mirror->devices[0].major == copy_job->src.obj->dev_major) &&
				    (target->data.mirror->devices[0].minor == copy_job->src.obj->dev_minor) &&
				    (target->data.mirror->devices[0].start == copy_job->src.start) &&
				    (target->data.mirror->devices[1].major == copy_job->trg.obj->dev_major) &&
				    (target->data.mirror->devices[1].minor == copy_job->trg.obj->dev_minor) &&
				    (target->data.mirror->devices[1].start == copy_job->trg.start)) {
					
					dm_update_status(mirror);
					
					copy_job->mirror = mirror;

					rc = 0;
				}
			}

			dm_deallocate_targets(target);
		}
	}

	dm_deallocate_device_list(device_list);

	if (rc != 0) {
		memset(mirror->name, '\0', sizeof(mirror->name));
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_copy_setup(copy_job_t * copy_job) {

	int rc = 0;
	storage_object_t * mirror;
	dm_target_t * target;

	LOG_PROC_ENTRY();

	rc = allocate_new_storage_object(&mirror);
	if (rc != 0) {
		LOG_SERIOUS("Error allocating a new storage object for the mirror: %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	mirror->size = min(copy_job->src.len, copy_job->trg.len);
	mirror->object_type = copy_job->src.obj->object_type;

	insert_thing(mirror->child_objects,
		     copy_job->src.obj,
		     INSERT_BEFORE,
		     NULL);

	insert_thing(mirror->child_objects,
		     copy_job->trg.obj,
		     INSERT_AFTER,
		     NULL);

	rc = find_copy_mirror_device(copy_job, mirror);

	if (rc == 0) {
		/*
		 * We're going multi-threaded. Turn on the logging of the PID
		 * in the log timestamp.
		 */
		log_pid = TRUE;

		/*
		 * The mirror is running.  Start the copy_progress_thread() to
		 * keep the progress in the copy_job up to date.
		 */
		initialize_copy_progress(copy_job, 1);

		pthread_create(&copy_job->progress_tid,
			       &pthread_attr_detached,
			       copy_progress_thread,
			       copy_job);

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	snprintf(mirror->name, EVMS_NAME_SIZE, COPY_NAME_PREFIX "%s" COPY_NAME_INFIX "%s",
		 copy_job->src.obj->name, copy_job->trg.obj->name);

	/*
	 * The device must have a size if it is to be used as the target of
	 * another device (i.e., the device that will point to the mirror
	 * instead of the source object).  The device must have an active map in
	 * order to have a size.  But the mirror isn't ready to run until
	 * copy_start() is called.  So, load the mirror device with an error map
	 * and suspend the device.
	 */
	target = dm_allocate_target(DM_TARGET_ERROR, 0, mirror->size, 0, 0);

	if (target != NULL) {
		/*
		 * 2.4 kernels keep device sizes in number of 1KB blocks, not in
		 * sectors.  An odd size will be rounded down by the kernel.  If
		 * the source has an odd size, then when the caller of the copy
		 * service remaps to the mirror device the device can end up
		 * being too small causing an error when loading the map.  Round
		 * up the error target size so that it is big enough for the
		 * device that will remap to the mirror.
		 */
		if (is_2_4_kernel) {
			target->length += 1;
			target->length &= ~(1);
		}

		rc = dm_activate(mirror, target);

		if (rc == 0) {
			rc = dm_suspend(mirror, TRUE);

			if (rc != 0) {
				LOG_SERIOUS("dm_suspend() to suspend the error map for the mirror returned %d: %s\n", rc, evms_strerror(rc));
			}

		} else {
			LOG_SERIOUS("dm_activate() to activate the error map for the mirror returned %d: %s\n", rc, evms_strerror(rc));
		}

		dm_deallocate_targets(target);

	} else {
		LOG_CRITICAL("Error allocating target.\n");
		rc = ENOMEM;
	}

	if (rc != 0) {
		dm_deactivate(mirror);
		free_old_storage_object(mirror);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Now load the mirror device's inactive table with the mapping for
	 * the mirror.  When the device is resumed in copy_start(), the
	 * error map will be replaced with the mirror map.
	 */
	target = dm_allocate_target(DM_TARGET_MIRROR, 0, mirror->size, 2, 0);

	if (target != 0) {
		/*
		 * 2.4 kernels keep device sizes in number of 1KB blocks, not in
		 * sectors.  An odd size will be rounded down by the kernel.  If
		 * the source has an odd size it will be rounded down by the
		 * kernel.  When the mirror map is created the same size as the
		 * source, the source can end up being too small causing an
		 * error when loading the map for the mirror.  Round down the
		 * mirror target size so that it is not bigger than the source.
		 */
		if (is_2_4_kernel) {
			target->length &= ~(1);
		}

		target->data.mirror->num_mirrors = 2;
		target->data.mirror->chunk_size = 128;
		target->data.mirror->persistent = FALSE;
		target->data.mirror->devices[0].major = copy_job->src.obj->dev_major;
		target->data.mirror->devices[0].minor = copy_job->src.obj->dev_minor;
		target->data.mirror->devices[0].start = copy_job->src.start;
		target->data.mirror->devices[1].major = copy_job->trg.obj->dev_major;
		target->data.mirror->devices[1].minor = copy_job->trg.obj->dev_minor;
		target->data.mirror->devices[1].start = copy_job->trg.start;

		rc = dm_load_targets(mirror, target);

		dm_deallocate_targets(target);

	} else {
		LOG_CRITICAL("Error allocating target.\n");
		rc = ENOMEM;
	}

	if (rc == 0) {
		copy_job->mirror = mirror;

	} else {
		LOG_SERIOUS("dm_load_targets() to load the mirror target returned %d: %s\n", rc, evms_strerror(rc));
		dm_deactivate(mirror);
		free_old_storage_object(mirror);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_copy_start(copy_job_t * copy_job) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = dm_suspend(copy_job->mirror, FALSE);

	if (rc == 0) {
		copy_job->flags |= COPY_STARTED;
	}

	/* Start the copy_progress_thread() if it hasn't been started yet. */
	if (copy_job->progress_tid == 0) {
		initialize_copy_progress(copy_job, 1);

		pthread_create(&copy_job->progress_tid,
			       &pthread_attr_detached,
			       copy_progress_thread,
			       copy_job);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * This thread is launched when engine_copy_wait() is called.
 * The thread wakes up every 1/10 second, and reports the progress of the
 * copy to the UI.  It relies on the copy_progress_thread() to keep the
 * progress in the copy job up to date.
 */
static void * copy_wait_progress_thread(void * arg) {

	copy_job_t * copy_job = (copy_job_t *) arg;
	int old_cancel_state;

	/*
	 * Don't allow the thread to be canceled unless it is sleeping.  There
	 * is a very real risk that the thread can be canceled in the middle
	 * of writing to the log file while the log mutex lock is held.  Bad
	 * things happen if the thread leaves while holding the log lock.
	 */
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_cancel_state);

	LOG_PROC_ENTRY();

	do {
		report_copy_progress(copy_job);

		if (!(copy_job->flags & COPY_FINISHED)) {

			/*
			 * The thread can handle being canceled at this point.
			 */
			pthread_setcancelstate(old_cancel_state, NULL);
			pthread_testcancel();

			usleep(100000);

			/* Here, too. */
			pthread_testcancel();
			pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &old_cancel_state);
		}

	} while (!(copy_job->flags & COPY_FINISHED));

	LOG_PROC_EXIT_PTR(NULL);

	pthread_setcancelstate(old_cancel_state, NULL);

	return NULL;
}


static unsigned char sector_buf[EVMS_VSECTOR_SIZE];

int engine_copy_wait(copy_job_t * copy_job) {

	int rc = 0;
	unsigned int event_nr = 0;
	char * info = NULL;
	pthread_t copy_wait_progress_tid = 0;
	boolean orig_log_pid = log_pid;

	LOG_PROC_ENTRY();

	if (copy_job->title != NULL) {
		/*
		 * Since progress is reported on a separate thread,
		 * turn on the logging of the PID in the log timestamp.
		 */
		log_pid = TRUE;

		pthread_create(&copy_wait_progress_tid,
			       &pthread_attr_detached,
			       copy_wait_progress_thread,
			       copy_job);

		if (copy_wait_progress_tid == 0) {
			log_pid = orig_log_pid;
			copy_job->progress.type = INDETERMINATE;
			report_copy_progress(copy_job);
		}
	}

	rc = dm_wait(copy_job->mirror, &event_nr, &info);

	if (rc == 0) {
		/*
		 * 2.4 kernels use block sizes of 1KB.  They don't handle
		 * any odd sector at the end of the device.  If we are
		 * running on a 2.4 kernel and the mirror has an odd
		 * number of sectors, copy the las sector by hand.
		 */
		if (is_2_4_kernel && (copy_job->mirror->size & 1)) {
			rc = copy_job->src.obj->plugin->functions.plugin->read(copy_job->src.obj, copy_job->src.start + copy_job->mirror->size - 1, 1, sector_buf);
			if (rc == 0) {
				rc = copy_job->trg.obj->plugin->functions.plugin->write(copy_job->trg.obj, copy_job->trg.start + copy_job->mirror->size - 1, 1, sector_buf);

				if (rc != 0) {
					LOG_SERIOUS("Error code %d when writing the last sector to target object %s: %s\n", rc, copy_job->trg.obj->name, evms_strerror(rc));
				}

			} else {
				LOG_SERIOUS("Error code %d when reading the last sector from source object %s: %s\n", rc, copy_job->src.obj->name, evms_strerror(rc));
			}
		}
	}

	if (copy_wait_progress_tid != 0) {
		pthread_cancel(copy_wait_progress_tid);

		/* Wait for the thread to exit. */
		pthread_join(copy_wait_progress_tid, NULL);

		log_pid = orig_log_pid;
	}

	final_copy_progress(copy_job);

	engine_free(info);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_copy_cleanup(copy_job_t * copy_job) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (copy_job->mirror != NULL) {
		dm_deactivate(copy_job->mirror);

		free_old_storage_object(copy_job->mirror);

		copy_job->mirror = NULL;
	}

	if (copy_job->progress_tid != 0) {
		pthread_cancel(copy_job->progress_tid);
		pthread_join(copy_job->progress_tid, NULL);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

