/*
 *
 *   (C) Copyright IBM Corp. 2002, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "readable.h"
#include "enum.h"
#include "logging.h"

/**
 *	filter_thing_to_activate - a clist_filter_func that keeps things that can be activated
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be activated.
 */
int filter_thing_to_activate(object_handle_t handle, void *user_data)
{
	return evms_can_activate(handle);
}

/**
 *	activate_items - activates one or more selected items
 *	@item: the menu item/button that was activated
 *
 *	This routine activates all selected items in the clist of the selection
 *	dialog window. Once all items have been activated, we set the dialog->status
 *	flag to indicate we can now close the dialog.
 */
int activate_items(struct menu_item *item)
{
	GList *element;
	int rc = 0, list_length = 0;
	gboolean error_occurred = FALSE;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	element = selwin->clist->selections;

	while (element != NULL) {
		struct clist_item *row;
		object_handle_t handle;

		row = element->data;
		handle = GPOINTER_TO_UINT(row->user_data);

		rc = evms_activate(handle);

		if (rc != 0 && rc != E_CANCELED)
			error_occurred = TRUE;

		list_length++;
		element = g_list_next(element);
	}

	if (list_length == 1) {
			update_status(_("Activate"), rc);
	} else {
		if (error_occurred)
			print_statusbar_text("One or more Activate commands completed with errors.");
		else
			print_statusbar_text("The Activate command(s) completed successfully.");
	}

	refresh_views();
	dialog->status = DLG_STATUS_CLOSING;

	return 0;
}

/**
 *	show_activate_dialog - allow deletion of one or more objects from selection list
 *	@handle: handle of object when called for a context-sensitive activate, zero otherwise
 *	@type: the object type (always given even if handle is zero)
 *
 *	This routine is called to display a list of objects of the given type to allow
 *	the user to activate one or more of them.
 */
int show_activate_dialog(object_handle_t handle)
{
	char *title, *column_title;
	struct selwin *selwin;
	struct dialog_window *dialog;

	column_title = _("Logical Volume and/or Storage Object");
	title = _("Activate Logical Volume and/or Storage Object");
	
        selwin = create_selection_window(title, NULL, NULL,
					_("_Activate"),
					(menuitem_activate_cb)activate_items,
					NULL,
					NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, column_title);
	print_clist_column_title(selwin->clist, 2, _("Size"));

	set_menu_item_visibility(dialog->prev_button, FALSE);

	if (handle == 0) {
		clist_populate(selwin->clist, enum_volumes, filter_thing_to_activate,
				format_standard_item, NULL, NULL);
		clist_populate(selwin->clist, enum_data_objects, filter_thing_to_activate,
				format_standard_item, NULL, NULL);
		set_selection_mode(selwin->clist, CLIST_MODE_MULTI);
		set_menu_item_sensitivity(dialog->prev_button, FALSE);
	} else {
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);
	}

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_activate_menuitem_activated - activate initiated from context popup menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to activate an object selected from the context popup
 *	menu if a view.
 */
int context_activate_menuitem_activated(struct menu_item *item)
{
	object_type_t type;
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (evms_get_handle_object_type(handle, &type) == 0)
		show_activate_dialog(handle);

	return 0;
}

/**
 *	actions_activate_menuitem_activated - activate initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by one of the Activate Actions pulldown
 *	menu items. The object type is passed as the item->user_data;
 */
int actions_activate_menuitem_activated(struct menu_item *item)
{
	show_activate_dialog(0);
	return 0;
}


/**
 *	filter_thing_to_deactivate - a clist_filter_func that keeps things that can be deactivated
 *	@handle: the thing's handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be deactivated.
 */
int filter_thing_to_deactivate(object_handle_t handle, void *user_data)
{
	return evms_can_deactivate(handle);
}

/**
 *	deactivate_items - deactivates one or more selected items
 *	@item: the menu item/button that was deactivated
 *
 *	This routine deactivates all selected items in the clist of the selection
 *	dialog window. Once all items have been deactivated, we set the dialog->status
 *	flag to indicate we can now close the dialog.
 */
int deactivate_items(struct menu_item *item)
{
	GList *element;
	int rc = 0, list_length = 0;
	gboolean error_occurred = FALSE;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	element = selwin->clist->selections;

	while (element != NULL) {
		struct clist_item *row;
		object_handle_t handle;

		row = element->data;
		handle = GPOINTER_TO_UINT(row->user_data);

		rc = evms_deactivate(handle);

		if (rc != 0 && rc != E_CANCELED)
			error_occurred = TRUE;

		list_length++;
		element = g_list_next(element);
	}

	if (list_length == 1) {
			update_status(_("Deactivate"), rc);
	} else {
		if (error_occurred)
			print_statusbar_text("One or more Deactivate commands completed with errors.");
		else
			print_statusbar_text("The Deactivate command(s) completed successfully.");
	}

	refresh_views();
	dialog->status = DLG_STATUS_CLOSING;

	return 0;
}

/**
 *	show_deactivate_dialog - allow deletion of one or more objects from selection list
 *	@handle: handle of object when called for a context-sensitive deactivate, zero otherwise
 *	@type: the object type (always given even if handle is zero)
 *
 *	This routine is called to display a list of objects of the given type to allow
 *	the user to deactivate one or more of them.
 */
int show_deactivate_dialog(object_handle_t handle)
{
	char *title, *column_title;
	struct selwin *selwin;
	struct dialog_window *dialog;

	column_title = _("Logical Volume and/or Storage Object");
	title = _("Deactivate Logical Volume and/or Storage Object");
	
        selwin = create_selection_window(title, NULL, NULL,
					_("_Deactivate"),
					(menuitem_activate_cb)deactivate_items,
					NULL,
					NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, column_title);
	print_clist_column_title(selwin->clist, 2, _("Size"));

	set_menu_item_visibility(dialog->prev_button, FALSE);

	if (handle == 0) {
		clist_populate(selwin->clist, enum_volumes, filter_thing_to_deactivate,
				format_standard_item, NULL, NULL);
		clist_populate(selwin->clist, enum_data_objects, filter_thing_to_deactivate,
				format_standard_item, NULL, NULL);
		set_selection_mode(selwin->clist, CLIST_MODE_MULTI);
		set_menu_item_sensitivity(dialog->prev_button, FALSE);
	} else {
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);
	}

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_deactivate_menuitem_deactivated - deactivate initiated from context popup menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to deactivate an object selected from the context popup
 *	menu if a view.
 */
int context_deactivate_menuitem_activated(struct menu_item *item)
{
	object_type_t type;
	object_handle_t handle = GPOINTER_TO_UINT(item->user_data);

	if (evms_get_handle_object_type(handle, &type) == 0)
		show_deactivate_dialog(handle);

	return 0;
}

/**
 *	actions_deactivate_menuitem_deactivated - deactivate initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by one of the Deactivate Actions pulldown
 *	menu items. The object type is passed as the item->user_data;
 */
int actions_deactivate_menuitem_activated(struct menu_item *item)
{
	show_deactivate_dialog(0);
	return 0;
}

