/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*****************************************************************
*
*  File: filgraph.c
*
*  Purpose:  Triangle list file output.
*             Format: One triangle per line,
*                   x1 y1 x2 y2 x3 y3 w1 w2 w3 d
*             vertex coordinates, edge widths, and face density.
*/

#include "include.h"

static FILE *fd;

/*****************************************************************
*
*  Function: fil_init()
*
*  Purpose:  Get file name from user.
*/

void fil_init()
{
  char file_name[200];

  for (;;)
  {
     prompt("Enter file name: ",file_name,sizeof(file_name));
     if ( file_name[0] == 0 )
     { outstring("File aborted.\n");
       longjmp(jumpbuf,1); /*abort*/
     }

     fd = fopen(file_name,"w");
     if ( fd == NULL )
        { perror(file_name);
          kb_error(1034,"",RECOVERABLE);

        }
     else return;
  }

}

/************************************************************
*
*  Function: fil_edge()
*
*  Purpose: Graphs one edge, already transformed.
*/

void fil_edge(t)
struct tsort *t;
{
  fprintf(fd,"  %f %f  ",(DOUBLE)t->x[0][0],(DOUBLE)t->x[0][1]);
  fprintf(fd,"  %f %f  ",(DOUBLE)t->x[1][0],(DOUBLE)t->x[1][1]);
  fprintf(fd,"  %f %f  ",(DOUBLE)t->x[1][0],(DOUBLE)t->x[1][1]);
  fprintf(fd,"  0.03  0.03  0.03  0.0 \n");
}


/************************************************************
*
*  Function: fil_facet()
*
*  Purpose: Graphs one facet, already transformed.
*/

void fil_facet(t)
struct tsort *t;
{
  REAL cosine;
  edge_id e_id;
  facetedge_id fe_id;
  int type;
  int i;
  
  if ( t->color == CLEAR ) return;

  fprintf(fd,"%f %f",(DOUBLE)t->x[0][0],(DOUBLE)t->x[0][1]);
  fprintf(fd," %f %f",(DOUBLE)t->x[1][0],(DOUBLE)t->x[1][1]);
  fprintf(fd," %f %f",(DOUBLE)t->x[2][0],(DOUBLE)t->x[2][1]);
  fe_id = get_facet_fe(t->f_id);
  for ( i = 0 ; i < 3 ; i++ )
     { 
        e_id = get_fe_edge(fe_id);
        if ( get_eattr(e_id) & FIXED ) type = 3;
        else if ( equal_id(get_next_facet(fe_id),fe_id) )
          type = 1;  /* edge of some sort */
        else if ( !equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) )
          type = 4;  /* triple line at least */
        else type = 0;  /* ordinary internal grid line */

        fprintf(fd," %1d",type);
        fe_id = get_next_edge(fe_id);
     }
  
  if ( valid_id(t->f_id) )
     {
        cosine = t->normal[1]/sqrt(dotf(t->normal,t->normal,3));
        if ( (REAL)t->normal[2] < 0.0 ) cosine = -cosine;
     }
  else cosine = 0.0;
  fprintf(fd," %f\n",(DOUBLE)cosine);
}

/*************************************************************
*
*  Function:  fil_finish()
*
*  Purpose:    End output.
*/

void fil_finish()
{
  fclose(fd);
}
