/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*************************************************************
*
*     file:        lexinit2.c
*
*     Purpose:    Routines used by initialize() in lexinit.c
*                 Split because lexinit.c got too big for Mac
*
*/

#include "include.h"
#include "lex.h"
#include "ytab.h"
#define yylex kb_yylex

/****************************************************************
*
*  Function: read_periods()
*
*  Purpose:  Reads torus periods.
*/

void read_periods()
{
  int i,j;

  /* read in torus periods */
  web.inverse_periods = dmatrix(0,SDIM-1,0,SDIM-1);
  for ( i = 0 ; i < SDIM ; i++ )
    for ( j = 0 ; j < SDIM ; j++ )
    { int esize;
      esize = exparse(0,&torus_period_expr[i][j],USERCOPY);
      if ( esize <= 0 )
      { sprintf(errmsg,
            "Bad torus_period[%d][%d] definition.  Check space dimension\n",
                   i+1,j+1);
        kb_error(1670,errmsg,DATAFILE_ERROR);
        return;
      }
    }
  calc_periods(ADJUST_VOLUMES);

}

/****************************************************************
*
*  Function: read_parameter()
*
*  Purpose:  Reads one adjustable parameter.
*/

void read_parameter()
{
  int n;
  struct global *p;
  int oldtok = tok; /* whether variable */

  tok = yylex(); /* dispose of PARAMETER */
  if ( tok == IDENT_ )
     { p = globals + int_val;
        if ( !(p->flags & LEFTOVER) ) 
        { sprintf(errmsg,"Redefinition of identifier '%s'.\n",yytext);
          kb_error(1671,errmsg,DATAFILE_ERROR);
        }
        p->flags &= ~LEFTOVER;
     }
  else 
  { if ( tok != NEWIDENT_ )
     { kb_error(1672,"Need PARAMETER identifier.\n",WARNING);
        return;
     }
     if ( strlen(yytext) < 2 )
        kb_error(1867,"Identifiers must be at least two characters long.\n",
            DATAFILE_ERROR);
     n = add_global(yytext);
     if ( n < 0 )
         kb_error(1673,"Duplicate parameter name.\n",DATAFILE_ERROR);
     p = globals+ n;
  }
  p->value.real = 0.0;  /* default */
  p->flags |= ORDINARY_PARAM | SURFACE_PARAMETER;
  p->delta = OPTPARAM_DELTA;
  p->scale = 1.0;
  if ( oldtok == OPTIMIZING_PARAMETER_ )
  { p->flags |= OPTIMIZING_PARAMETER;
     if ( optparamcount >= MAXOPTPARAM-1 )
        kb_error(1674,"Too many optimizing parameters. Change MAXOPTPARAM in extern.h if you really need more.\n",DATAFILE_ERROR);
     else optparam[optparamcount++].pnum = n;
  }

  tok = yylex(); /* dispose of IDENT_ */
  if ( (tok == '=') || (tok == ASSIGN_) ) /* have initial value */
  { if ( read_const(&p->value.real) < 0 )
       kb_error(1675,"Need constant expression for initial value.\n",DATAFILE_ERROR);
    else tok = yylex();  /* get back token exparse pushed back */
    if ( tok == ';' ){ verb_flag = 0; tok = yylex();  /* permit ; */ }
  }
  else if ( tok == PARAMETER_FILE_ )
  {
    tok = yylex();
    if ( tok == QUOTATION_ )
    { FILE *pfd = path_open(yytext);
      int k;
      REAL val;

      if ( pfd == NULL )
      { 
        sprintf(errmsg, "Cannot open parameter file %s.\n",yytext);
        kb_error(1676,errmsg, DATAFILE_ERROR);
        return;
      }
      p->value.file.value_file = mycalloc(strlen(yytext)+1,1);
      strcpy(p->value.file.value_file,yytext);
      p->value.file.values = (REAL *)mycalloc(1000,sizeof(REAL));
      while ( fgets(msg,msgmax,pfd) )
      {
#ifdef LONGDOUBLE
        sscanf(msg,"%d %Lf",&k,&val);
#else
        sscanf(msg,"%d %lf",&k,&val);
#endif
        if ( k >= 1000 )
        { kb_error(1677,"Too high element number in parameter file.\n",WARNING);
          break;
        }
        p->value.file.values[k] = val;
      }
      fclose(pfd);
      p->flags |= FILE_VALUES;
      tok = yylex();
    }
    else
      kb_error(1678,"Parameter file name missing.\n",DATAFILE_ERROR);

  }
  if ( tok == DELTA_ )
  { tok = yylex(); /* dispose of IDENT_ */
     if ( (tok == '=') || (tok == ASSIGN_) ) /* have initial value */
     { if ( read_const(&p->delta) < 0 )
            kb_error(1675,"Need constant expression for delta.\n",DATAFILE_ERROR);
        else tok = yylex();  /* get back token exparse pushed back */
     }
  }
  if ( tok == SCALE_ )
  { tok = yylex(); /* dispose of IDENT_ */
     if ( (tok == '=') || (tok == ASSIGN_) ) /* have initial value */
     { if ( read_const(&p->scale) < 0 )
            kb_error(1676,"Need constant expression for scale.\n",DATAFILE_ERROR);
        else tok = yylex();  /* get back token exparse pushed back */
     }
  }
  recovery_flag = 0;
}
         
      
/*************************************************************************
*
* Reads and parses information for one free boundary specification.
* Current line starts with BOUNDARY keyword.
*/

void read_boundary()
{
  int bnum;  /* boundary number */
  int pcount; /* number of parameters */
  int i;
  int esize;
  struct boundary *bdry;

  tok = yylex();  /* eat BOUNDARY */
  if ( tok != INTEGER_ ) 
     { kb_error(1679,"Need boundary number.\n",DATAFILE_ERROR);
        return;
     }
  bnum = int_val;     /* boundary number */
  if ( (bnum < 0) || (tok != INTEGER_) ) 
     { sprintf(errmsg,"Bad boundary number: %d.\n",bnum);
        kb_error(1680,errmsg,DATAFILE_ERROR);
        tok = yylex();
        return;
     }
  if ( bnum >= BDRYMAX )
     { sprintf(errmsg,"Boundary number exceeds maximum (%d): %d. Change BDRYMAX in skeleton.h if you really need more.\n",
                                                  BDRYMAX-1,bnum);
        kb_error(1681,errmsg,DATAFILE_ERROR);
        tok = yylex();
        return;
     }
  bdry = web.boundaries + bnum;
  bdry->num = bnum;
  if ( bdry->pcount > 0 )
     { sprintf(errmsg,"Boundary number %d already defined.\n",bnum);
        kb_error(1682,errmsg,DATAFILE_ERROR);
     }

  tok = yylex();
  if ( tok != PARAMETERS_ )
     { sprintf(errmsg,"Expecting PARAMETERS keyword for boundary %d.\n",bnum);
        kb_error(1683,errmsg,DATAFILE_ERROR);
        pcount = bdry->pcount = 1; 
        /* try to continue */
     }
  else
  { tok = gettok(INTEGER_);
    bdry->pcount = pcount = int_val;
  }
  if ( pcount > web.maxparam ) web.maxparam = pcount;
  if ( (pcount < 0) || (tok != INTEGER_) ) 
     { sprintf(errmsg,"Bad parameter count %d for boundary %d. Assuming 1.\n",
          pcount,bnum);
       kb_error(1684,errmsg,DATAFILE_ERROR);
       pcount = bdry->pcount = 1;  /* try to continue */
     }
  if ( pcount > MAXPARAM )
     { sprintf(errmsg,
            "Parameter count for boundary %d exceeds %d. Assuming %d.\n",
                                          bnum,MAXPARAM,MAXPARAM);
       kb_error(1685,errmsg,DATAFILE_ERROR);
       bdry->pcount = pcount = MAXPARAM; /* try to continue */
     }

  tok = yylex();
  if ( tok == CONVEX_ ) 
     { web.convex_flag = 1;
       bdry->attr |= B_CONVEX;
       tok = yylex();
     }

  /* read and parse coordinate functions */
  bdry->coordf[0] = (struct expnode *)mycalloc(SDIM+2*MAXCOORD,sizeof(struct expnode));
  for ( i = 1 ; i < SDIM ; i++ )
    bdry->coordf[i] = bdry->coordf[0] + i;

  for ( i = 0 ; i < SDIM ; i++ )
  {
    if ( (tok != COORD_ ) || ( int_val != 1 + i ))
    { sprintf(errmsg,
          "Bad coordinate %d definition for boundary %d.\n",i+1,bnum);
      goto berr;
    }
    boundary_expr_flag = 1;
    esize = exparse(pcount,bdry->coordf[i],USERCOPY);
    boundary_expr_flag = 0;
    tok = yylex();
    if ( esize <= 0 )
    { sprintf(errmsg,
       "Bad coordinate %d definition for boundary %d.\n",i+1,bnum);
      goto berr;
    }
  }

  /* various integrands */
  for ( i = 0 ; i < MAXCOORD ; i++ )
     { bdry->envect[i] = bdry->coordf[0] + SDIM + i;
        bdry->convect[i] = bdry->coordf[0] + SDIM + MAXCOORD + i;
     }
  for (;;)
    switch ( tok )
    { 
      case ENERGY_:
      /* read and parse energy function */
      tok = yylex();
      for ( i = 0 ; i < MAXCOORD ; i++ )
       {
         if ( (tok != ENVECT_) || (int_val != 1 + i ))
            break;
         esize = exparse(SDIM,bdry->envect[i],USERCOPY);
         tok = yylex();  
         if ( esize <= 0 )
          { sprintf(errmsg,
                "Bad energy component %d definition for boundary %d.\n",i+1,bnum);
             kb_error(1693,errmsg,DATAFILE_ERROR);
             return;
          }
        }
      bdry->compcount = i;
      bdry->attr |= CON_ENERGY;
      break;

     case CONTENT_:
      /* read and parse content vector potential */
      tok = yylex();
      bdry->attr |= CON_CONTENT;
      for ( i = 0 ; i < SDIM ; i++ )
       {
         if ( (tok != CONVECT_) || (int_val != 1 + i ))
            break;
         esize = exparse(SDIM,bdry->convect[i],USERCOPY);
         tok = yylex();
         if ( esize <= 0 )
          { sprintf(errmsg,
                 "Bad content component %d definition for boundary %d.\n",i+1,bnum);
             kb_error(1694,errmsg,DATAFILE_ERROR);
             return;
          }
        }
        /* check consistency of number of components */
        if ( bdry->compcount )
        { if ( bdry->compcount != i )
            kb_error(1695,"Inconsistent number of components in content integrand.\n",
                   WARNING);
        }
        else
        { if ( (i != 1) && (i != SDIM) )
            kb_error(1696,"Illegal number of components in content integrand.\n",
                   DATAFILE_ERROR);
          bdry->compcount = i;
        }
      bdry->attr |= CON_CONTENT;
      break;
    
     default: return; 
    } /* end switch for integrands */

  return;

berr:
  kb_error(1686,errmsg,DATAFILE_ERROR);

}

/*************************************************************************
*
* Reads and parses information for one free constraint specification.
* Current line starts with constraint keyword.
*/

void read_constraint()
{
  int cnum;  /* constraint number */
  int i;
  int esize;
  int more_attr;
  struct constraint *con;
  struct quantity *quan;
  int snum;

  tok = yylex();  /* eat CONSTRAINT */
  if ( tok != INTEGER_ ) 
     { kb_error(1687,"Need constraint number.\n",DATAFILE_ERROR);
        return;
     }
  cnum = int_val;     /* constraint number */
  if ( cnum < 0 ) 
     { kb_error(1688,"Constraint number must be positive.\n",DATAFILE_ERROR);
        tok = yylex();
        return;
     }
  if ( cnum >= MAXCON )
     { sprintf(errmsg,"Constraint number exceeds maximum (%d): %d.\n",
                                         MAXCON-1,cnum);
        kb_error(1689,errmsg,DATAFILE_ERROR);
        tok = yylex();
        return;
     }
  if ( web.constraint_addr[cnum] )
     { sprintf(errmsg,"Constraint number %d already defined.\n",cnum);
        kb_error(1690,errmsg,DATAFILE_ERROR);
     }
  web.constraint_addr[cnum] = dy_calloc(sizeof(struct constraint),1);
  con = GETCONSTR(cnum);
  if ( cnum >= web.concount ) web.concount = cnum+1;

  tok = yylex();
  for ( more_attr = 1 ; more_attr ; )
    switch ( tok )
    {
      case CONVEX_:
          web.convex_flag = 1;
          con->attr |= B_CONVEX;
          tok = yylex();
          break;

      case NONNEGATIVE_:
          web.constr_flag = 1;
          con->attr |= NONNEGATIVE;
          tok = yylex();
          break;

      case NONPOSITIVE_:
          web.constr_flag = 1;
          con->attr |= NONPOSITIVE;
          tok = yylex();
          break;

      case GLOBAL_:
          web.constr_flag = 1;
          con->attr |= GLOBAL;
          web.con_global_map[web.con_global_count++] = (conmap_t)cnum;
          tok = yylex();
          break;

      default: more_attr = 0;
    }
     

  /* read and parse defining function */
  constraint_init(con);
  if ( tok != FUNCTION_ )
     { sprintf(errmsg,
          "Expected function definition for constraint %d.\n",cnum);
        kb_error(1691,errmsg,DATAFILE_ERROR);
        return;
     }
  esize = exparse(SDIM,con->formula,USERCOPY);
  tok = yylex();
  if ( esize <= 0 )
     { sprintf(errmsg,"Bad function definition for constraint %d.\n",cnum);
        kb_error(1692,errmsg,DATAFILE_ERROR);
        return;
     }

  /* various integrands */
  for (;;)
    switch ( tok )
    { 
     case ENERGY_:
      /* read and parse energy function */
      tok = yylex();
      for ( i = 0 ; i < MAXCONCOMP ; i++ )
        {
         if ( (tok != ENVECT_) || (int_val != 1 + i ))
            break;
         esize = exparse(SDIM,con->envect[i],USERCOPY);
         tok = yylex();  
         if ( esize <= 0 )
          { sprintf(errmsg,
                "Bad energy component %d definition for constraint %d.\n",i+1,cnum);
             kb_error(1693,errmsg,DATAFILE_ERROR);
             return;
          }
        }
      con->compcount = i;
      con->attr |= CON_ENERGY;
      break;

     case CONTENT_:
      /* read and parse content vector potential */
      tok = yylex();
      con->attr |= CON_CONTENT;
      for ( i = 0 ; i < SDIM ; i++ )
        {
         if ( (tok != CONVECT_) || (int_val != 1 + i ))
            break;
         esize = exparse(SDIM,con->convect[i],USERCOPY);
         tok = yylex();
         if ( esize <= 0 )
          { sprintf(errmsg,
                 "Bad content component %d definition for constraint %d.\n",i+1,cnum);
             kb_error(1694,errmsg,DATAFILE_ERROR);
             return;
          }
        }
        /* check consistency of number of components */
        if ( con->compcount )
        { if ( con->compcount != i )
             kb_error(1695,"Inconsistent number of components in content integrand.\n",
               WARNING);
        }
        else
        { if ( (i != 1) && (i != SDIM) )
             kb_error(1696,"Illegal number of components in content integrand.\n",
                  DATAFILE_ERROR);
          con->compcount = i;
        }
      con->attr |= CON_CONTENT;
      break;
    
     case QUANTITY_:
        /* read and parse a quantity integrand */
      tok = yylex();
      if ( tok != INTEGER_ ) 
         { read_named_quantity();  /* oops, got fooled */
            return;
         }
        snum = int_val;     /* spec number */
        if ( snum >= QUANTMAX )
          { sprintf(errmsg,
                "Quantity number exceeds maximum (%d): %d.\n",QUANTMAX-1,snum);
             kb_error(1697,errmsg,DATAFILE_ERROR);
             tok = yylex();
             return;
          }
        quan = web.quants + snum;
        if ( !quan->quanvect[0] )
         { sprintf(errmsg,"Quantity number %d not defined.\n",snum);
            kb_error(1698,errmsg,DATAFILE_ERROR);
            return;
         }
        con->quanvect[snum][0] = 
          (struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
        tok = yylex();
        for ( i = 0 ; i < SDIM ; i++ )
         {
          con->quanvect[snum][i] = con->quanvect[snum][0] + i;
          if ( (tok !=  QVECT_) || (int_val != 1 + i ))
             break;
          esize = exparse(SDIM,con->quanvect[snum][i],USERCOPY);
          tok = yylex();
          if ( esize <= 0 )
            { sprintf(errmsg,
                 "Bad integrand %d definition for quantity %d.\n",i+1,snum);
              kb_error(1699,errmsg,DATAFILE_ERROR);
              return;
            }
         }
        /* check consistency of number of components */
        if ( con->compcount )
        { if ( con->compcount != i )
            kb_error(1700,"Inconsistent number of components in quantity integrand.\n",
                WARNING);
        }
        else 
        { if ( (i != 1) && (i != SDIM) )
             kb_error(1701,"Illegal number of components in quantity integrand.\n",
                 DATAFILE_ERROR);
          con->compcount = i;
        }
      con->quantity_map |= (1<<snum);
      break;

     default: return; 
    } /* end switch for integrands */
}


/*************************************************************************
*
* Reads and parses information for one surface energy  specification.
* Current line starts with SURFACE ENERGY keyword.
* Optional GLOBAL keyword.
*/

void read_surface_energy()
{
  int snum;  /* spec number */
  int i;
  int esize;
  int more_attr;
  struct surf_energy *surfen;

  kb_error(1702,"Numbered surface energies soon obsolete. Please use named quantity.\n",
      WARNING);

  tok = gettok(INTEGER_);
  if ( tok != INTEGER_ ) 
     { kb_error(1703,"Need surface energy number.\n",DATAFILE_ERROR);
        return;
     }
  tok = yylex();
  snum = int_val;     /* spec number */
  if ( snum >= SURFENMAX )
     { sprintf(errmsg,
          "Surface energy number exceeds maximum (%d): %d.\n",SURFENMAX-1,snum);
        kb_error(1704,errmsg,DATAFILE_ERROR);
        return;
     }
  surfen = web.surfen + snum;
  if ( surfen->envect[0] )
     { sprintf(errmsg,"Surface energy number %d already defined.\n",snum);
        kb_error(1705,errmsg,DATAFILE_ERROR);
        return; 
     }
  if ( snum >= web.surfen_count ) web.surfen_count = snum+1;

  for ( more_attr = 1 ; more_attr ; )
    switch ( tok )
    {
      case GLOBAL_:
          surfen->attr |= GLOBAL;
          web.surf_global_map |= (1 << snum);
          tok = yylex();
          break;

      default: more_attr = 0;
    }
     


  /* read and parse energy integrand vector */
  surfen->envect[0]=(struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
  for ( i = 0 ; i < SDIM ; i++ )
     {
        surfen->envect[i] = surfen->envect[0] + i;
        if ( (tok != ENVECT_) || (int_val != 1 + i ))
          { sprintf(errmsg,
                "Expected component %d definition for surface energy %d.\n",
                        i+1,snum);
             kb_error(1706,errmsg,DATAFILE_ERROR);
             return;
          }
        esize = exparse(SDIM,surfen->envect[i],USERCOPY);
        tok = yylex();
        if ( esize <= 0 )
          { sprintf(errmsg,
                "Bad component %d definition for surface energy %d.\n",i+1,snum);
             kb_error(1707,errmsg,DATAFILE_ERROR);
          }
     }
}

/*************************************************************************
*
*  function: read_method_instance()
*
*  purpose: reads information for definition of one method instance.
*/

void read_method_instance()
{ char mname[40];
  int mi;

  tok = yylex(); /* name */
  if ( tok != NEWIDENT_ )
     { kb_error(1708,"Need instance name.\n",DATAFILE_ERROR); return; }

  strncpy(mname,yytext,sizeof(mname));
  tok = yylex();
  if ( tok != METHOD_ )
     { kb_error(1709,"Missing METHOD keyword.\n",DATAFILE_ERROR); return; }

  tok = yylex();
  if ( tok != NEWIDENT_  && tok != MEAN_CURV_INT_ /* kludge */)
     { kb_error(1710,"Need method name.\n",DATAFILE_ERROR); return; }

  mi = new_method_instance(yytext,mname);
  tok = yylex();
  
  if ( mi >= 0 ) read_instance_attr(mi);

}

/*************************************************************************
*
* function: read_instance_attr()
*
* purpose: Read attributes of method instance, whether in method_instance
* definition or part of quantity definition.
*/

void read_instance_attr(mnum)
int mnum;  /* number of method instance */
{
  int bflag;
  int spec_flag = 0;
  int esize,i;
  struct gen_quant_method *gm;
  int comps;
  REAL val;

  gm = basic_gen_methods + METH_INST[mnum].gen_method;
  /* read further attributes of instance */
  for (bflag=0;bflag==0;)
     switch ( tok )
        { case GLOBAL_:
             if ( !(METH_INST[mnum].flags & GLOBAL_INST) )
                apply_method(NULLID,METH_INST[mnum].name); tok = yylex(); 
             break;
          case MODULUS_:
             if ( read_const(&METH_INST[mnum].modulus)  <= 0 )
                kb_error(1711,"Missing modulus.\n",WARNING);
             else tok = yylex();
             break;
          case IGNORE_FIXED_:
             METH_INST[mnum].flags |= IGNORE_FIXED;
             tok = yylex();
             break;
          case IGNORE_CONSTRAINTS_:
             METH_INST[mnum].flags |= IGNORE_CONSTR;
             sqcurve_ignore_constr = 1; /* kludge */
             tok = yylex();
             break;
          case K_VEC_ORDER_:
             if ( read_const(&val)  <= 0 )
             { kb_error(1712,"Missing k_vector_order value.\n",DATAFILE_ERROR);
                val = 1.;
             }
             else tok = yylex();
             METH_INST[mnum].vec_order = (int)val;
             break;
          case PARAMETER_1_:
             if ( read_const(&METH_INST[mnum].parameter_1)  <= 0 )
                kb_error(1713,"Missing parameter_1 value.\n",WARNING);
             else tok = yylex();
             METH_INST[mnum].flags |= METH_PARAMETER_1;
             break;
          case SCALAR_INTEGRAND_: 
              /* read and parse integrand function */
              if ( !(gm->spec_flags & SPEC_SCALAR) )
                 { kb_error(1714,"No scalar integrand for this method.\n",DATAFILE_ERROR);
             /* read in anyway */
                 }
              spec_flag |= SPEC_SCALAR;
              METH_INST[mnum].expr[0] =
                  (struct expnode *)mycalloc(1,sizeof(struct expnode));
              if ( gm->spec_flags & SPEC_EXTRADIM )
              { esize = exparse(2*SDIM,METH_INST[mnum].expr[0],USERCOPY);
                 spec_flag |= SPEC_EXTRADIM;
              }
              else esize = exparse(SDIM,METH_INST[mnum].expr[0],USERCOPY);
              tok = yylex();
              if ( esize <= 0 )
                      { sprintf(errmsg, "Bad integrand definition.\n");
                         kb_error(1715,errmsg,DATAFILE_ERROR);
                      }
              break;

          case VECTOR_INTEGRAND_:
              /* read and parse integrand vector */
              if ( !(gm->spec_flags & (SPEC_VECTOR|SPEC_KVECTOR)) )
                 { kb_error(1716,"No vector integrand for this method.\n",DATAFILE_ERROR);
                    comps = SDIM;
                    /* read vector integrand anyway to get past it */
                 }
              if ( gm->spec_flags & SPEC_VECTOR )
              { spec_flag |= SPEC_VECTOR;
                 comps = SDIM;
              }
              else /* SPEC_KVECTOR  */
              { spec_flag |= SPEC_KVECTOR;
                 comps = METH_INST[mnum].vec_order*SDIM;
                 if ( comps <= 0 ) 
                    kb_error(1717,"Need k_vector_order.\n",DATAFILE_ERROR);
              }
              if ( comps > MAXMEXPR )
                 kb_error(1718,"Total components exceeds MAXMEXPR.\n",DATAFILE_ERROR);
              METH_INST[mnum].expr[0]=(struct expnode *)mycalloc(comps,
                 sizeof(struct expnode));
              tok = yylex();
              for ( i = 0 ; i < comps ; i++ )
                 { METH_INST[mnum].expr[i] = METH_INST[mnum].expr[0] + i;
                    if ( (tok != QVECT_) || (int_val != 1 + i ))
                      { sprintf(errmsg,
                            "Expected component %d definition.\n",i+1);
                         kb_error(1719,errmsg,DATAFILE_ERROR);
                      }
                    esize = exparse(SDIM,METH_INST[mnum].expr[i],USERCOPY);
                    tok = yylex();
                    if ( esize <= 0 )
                      { sprintf(errmsg, "Bad component %d definition.\n",i+1);
                         kb_error(1720,errmsg,DATAFILE_ERROR);
                      }
                 }
              break;

          case FORM_INTEGRAND_:  /* 2-form */
              /* read and parse integrand 2-form */
              if ( !(gm->spec_flags & SPEC_2FORM) )
                 { kb_error(1721,"No 2-form integrand for this method.\n",DATAFILE_ERROR);
                    /* read in anyway to get past it */
                 }
              spec_flag |= SPEC_2FORM;
              comps = (SDIM*(SDIM-1))/2;
              if ( comps > MAXMEXPR )
                 kb_error(1722,"Total components exceeds MAXMEXPR.\n",DATAFILE_ERROR);

              METH_INST[mnum].expr[0]=(struct expnode *)mycalloc(comps,
                 sizeof(struct expnode));
              tok = yylex();
              for ( i = 0 ; i < comps ; i++ )
                 {
                    METH_INST[mnum].expr[i] = METH_INST[mnum].expr[0] + i;
                    if ( (tok != QVECT_) || (int_val != 1 + i ))
                      { sprintf(errmsg,
                            "Expected component %d definition.\n",i+1);
                         kb_error(1723,errmsg,DATAFILE_ERROR);
                      }
                    esize = exparse(SDIM,METH_INST[mnum].expr[i],USERCOPY);
                    tok = yylex();
                    if ( esize <= 0 )
                      { sprintf(errmsg, "Bad component %d definition.\n",i+1);
                         kb_error(1724,errmsg,DATAFILE_ERROR);
                      }
                 }
              break;

          default: bflag = 1; break;
        }
    if ( (gm->spec_flags & SPEC_SCALAR) & ~spec_flag )
      kb_error(1725,"Need scalar integrand for this method.\n",DATAFILE_ERROR);

    if ( (gm->spec_flags & SPEC_VECTOR) & ~spec_flag )
      kb_error(1726,"Need vector integrand for this method.\n",DATAFILE_ERROR);

 }

/*************************************************************************
*
* Reads and parses information for one quantity specification, 
* in particular, facet integrands and fixed value.
* Current line starts with QUANTITY keyword.
* Optional GLOBAL keyword.
*/

void read_quantity()
{
  int snum;  /* spec number */
  int i;
  int esize;
  int more_attr;
  struct quantity *quan;

  tok = gettok(INTEGER_);
  if ( tok != INTEGER_ ) 
     { read_named_quantity();
        return;
     }
  kb_error(1727,"Numbered quantities soon obsolete. Please use named quantity.\n",
      WARNING);
  pressure_set_flag = 0;

  snum = int_val;     /* spec number */
  if ( snum >= QUANTMAX )
     { sprintf(errmsg,
          "Quantity number exceeds maximum (%d): %d.\n",QUANTMAX-1,snum);
        kb_error(1728,errmsg,DATAFILE_ERROR);
        tok = yylex();
        return;
     }
  if  ( snum >= web.quantity_count ) web.quantity_count = snum+1;
  quan = web.quants + snum;
  if ( quan->quanvect[0] )
     { sprintf(errmsg,"Quantity number %d already defined.\n",snum);
        kb_error(1729,errmsg,PARSE_ERROR);
     }
  quan->attr |= IN_USE;
  if ( snum >= web.quantity_count ) web.quantity_count = snum+1;

  tok = yylex();
  for ( more_attr = 1 ; more_attr ; )
    switch ( tok )
    {
      case GLOBAL_:
          quan->attr |= GLOBAL;
          web.quant_global_map |= (1 << snum);
          tok = yylex();
          break;

      case FIXED_:
          quan->attr |= QFIXED;
          tok = yylex();
          if ( (tok != '=') || (read_const(&quan->target) <= 0) )
              kb_error(1730,"Missing quantity target value after FIXED.\n",DATAFILE_ERROR);
          else tok = yylex();
          break;

      default: more_attr = 0;
    }

  /* read and parse energy integrand vector */
  quan->quanvect[0]=(struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
  for ( i = 0 ; i < SDIM ; i++ )
     {
        quan->quanvect[i] = quan->quanvect[0] + i;
        if ( (tok != QVECT_) || (int_val != 1 + i ))
          { sprintf(errmsg,
                "Expected component %d definition for quantity %d.\n",i+1,snum);
             kb_error(1731,errmsg,DATAFILE_ERROR);

          }
        esize = exparse(SDIM,quan->quanvect[i],USERCOPY);
        tok = yylex();
        if ( esize <= 0 )
          { sprintf(errmsg,
                "Bad component %d definition for quantity %d.\n",i+1,snum);
             kb_error(1732,errmsg,DATAFILE_ERROR);

          }
     }

}

/***********************************************************************
*
*  function: read_named_quantity()
*
*  purpose: parse named quantity definition.
*/

void read_named_quantity()
{
  int n;
  int gnum; /* quantity number */
  char qname[100];
  int meth;
  struct method_instance *mi;
  char inst_name[40];
  int namecount; /* number of uninstantiated methods */
  int globality;
  int esize;
  int lagmulflag = 0;

  if ( (tok != NEWIDENT_) && (tok != IDENT_) )
    { kb_error(1733,"Need quantity number or name.\n",DATAFILE_ERROR);
       return;
    }
  strncpy(qname,yytext,sizeof(qname));
  /* check not already defined and add to list */
  tok = yylex();
  if ( tok == FIXED_ )
    { gnum = new_quantity(qname,Q_FIXED);
       tok = yylex(); /* eat '=' */
       if ( tok != '=' )
          kb_error(1734,"Missing '='\n",DATAFILE_ERROR);
       if (read_const(&(GEN_QUANTS[gnum].target)) <= 0)
        kb_error(1735,"Missing quantity target value. \n",DATAFILE_ERROR);
       else tok = yylex();
    }
  else if ( tok == ENERGY_ )
    { gnum = new_quantity(qname,Q_ENERGY);
       tok = yylex();
    }
  else if ( tok == INFO_QUANTITY_ )
    { gnum = new_quantity(qname,Q_INFO);
       tok = yylex();
    }
  else { kb_error(1736,"Need type of quantity: energy, fixed, or info_only.\n",

           DATAFILE_ERROR);
           gnum = new_quantity(qname,Q_INFO);
       }
  namecount = 0;
  for (;;) /* further attributes */
    switch ( tok )
    { case GLOBAL_METHOD_ :
      case METHOD_ :
          if ( GEN_QUANTS[gnum].flags & Q_COMPOUND )
            kb_error(1737,
              "Can't list separate methods with function of methods.\n",
                         DATAFILE_ERROR);
          globality = tok;
          tok = yylex();
          /* see if instance or method */
          for ( n=0,mi = METH_INST ; n < meth_inst_count ; mi++,n++ )
             if ( stricmp(mi->name,yytext) == 0 ) break;
          if ( n >= meth_inst_count )
             { /* need to instantiate method */
                strncpy(inst_name,qname,25);
                sprintf(inst_name+strlen(inst_name),"%d_",++namecount);
                strncat(inst_name,yytext,31-strlen(inst_name));
                meth = new_method_instance(yytext,inst_name);
                if ( meth >= 0 )
                { METH_INST[meth].flags |= IMPLICIT_INSTANCE;
                  attach_method(gnum,inst_name);
                  if ( globality == GLOBAL_METHOD_) 
                      apply_method(NULLID,inst_name);
                  tok = yylex();
                  read_instance_attr(meth);
                }
            }
          else /* predefined instance */
            { int mnum = attach_method(gnum,yytext);
              if ( (globality  == GLOBAL_METHOD_) && (mnum>=0) &&
                 !(METH_INST[mnum].flags & GLOBAL_INST) )
                  apply_method(NULLID,yytext);
              tok = yylex();
            }
          break;
      case LAGRANGE_MULTIPLIER_: 
         if ( read_const(&(GEN_QUANTS[gnum].pressure)) <= 0 ) 
            kb_error(2730,"Missing lagrange_multiplier value.\n",DATAFILE_ERROR);  
         else { GEN_QUANTS[gnum].flags |= Q_PRESSURE_SET; 
                  lagmulflag = 1; tok = yylex(); }
         break;

      case VOLCONST_:
          if (read_const(&(GEN_QUANTS[gnum].volconst)) <= 0)
             kb_error(1738,"Missing quantity volconst value. \n",DATAFILE_ERROR);
          else tok = yylex();
          break;
      case TOLERANCE_:
         if ( read_const(&GEN_QUANTS[gnum].tolerance) <= 0 )
            kb_error(2331,"Missing tolerance value.\n",DATAFILE_ERROR);
         else tok=yylex();
         if ( GEN_QUANTS[gnum].tolerance <= 0.0 ) 
          kb_error(2332,"Tolerance must be positive.\n",DATAFILE_ERROR);
         break;

      case MODULUS_:
          if (read_const(&(GEN_QUANTS[gnum].modulus)) <= 0)
             kb_error(1739,"Missing quantity modulus value. \n",DATAFILE_ERROR);
          else tok = yylex();
          break;
      case FUNCTION_:  /* compound function of instances */
          cur_quant = gnum;
          if ( GEN_QUANTS[cur_quant].method_count > 0 )
            kb_error(1740,
              "Can't list separate methods with function of methods.\n",
                         DATAFILE_ERROR);
          reading_comp_quant_flag = 1;
          esize = exparse(0,&(GEN_QUANTS[gnum].expr),USERCOPY);
          reading_comp_quant_flag = 0;
          cur_quant = -1 ;
          tok = yylex();
          if ( esize <= 0 )
              { sprintf(errmsg, "Bad function definition.\n");
                 kb_error(1741,errmsg,DATAFILE_ERROR);
              }
          GEN_QUANTS[gnum].flags |= Q_COMPOUND;
          break;
      default:
          goto named_exit; /* done with quantity */
          
  }

named_exit:
  if ( !lagmulflag ) pressure_set_flag = 0;
  return;
}

/*************************************************************
*
*  Function: add_outside()
*
*  Purpose:  Adds body outside all other bodies.  Used in
*                dynamic pressure calculations.
*/

void add_outside()
{

return; /* temporary turn off */
#ifdef OUTSIDEBODY     
    body_id b_id;    /* both models */


        b_id = new_body();
        web.outside_body = b_id;
        set_attr(b_id,PRESSURE); /* since has ambient pressure */

        if ( web.representation == STRING )
         { edge_id e_id;
            facet_id f_id;

            f_id = new_facet();    /* outside facet */
            set_facet_body(f_id,b_id);

            /* add to outside of every edge bordering just ONE cell */
            FOR_ALL_EDGES(e_id)
              {
                 facetedge_id new_fe;
                 facetedge_id fe_id;
                 
                 fe_id = get_edge_fe(e_id);
                 if ( !valid_id(fe_id) ) continue;
                 if ( !equal_id(fe_id,get_next_facet(fe_id)) ) continue;
                 new_fe = new_facetedge(inverse_id(f_id),e_id);
                 set_next_facet(fe_id,new_fe);
                 set_prev_facet(fe_id,new_fe);
                 set_next_facet(new_fe,fe_id);
                 set_prev_facet(new_fe,fe_id);
                 set_facet_fe(f_id,new_fe);

              }
         }
      else /* SOAPFILM */
         {
            facet_id f_id;

            /* add to outside of every facet bordering just ONE cell */
            FOR_ALL_FACETS(f_id)
              {
                 body_id b1_id,b2_id;

                 b1_id = get_facet_body(f_id);
                 b2_id = get_facet_body(inverse_id(f_id));
                 if ( valid_id(b1_id) == valid_id(b2_id) ) continue;

                 if ( valid_id(b1_id) )
                    set_facet_body(inverse_id(f_id),b_id);
                 if ( valid_id(b2_id) )
                    set_facet_body(f_id,b_id);
              }
        }
#endif
}

/*********************************************************************
*
*  Function: fix_volconst()
*
*  Purpose:  For torus, figure out constant volume adjustments, 
*                figuring that given volumes are within 1/2 of a 
*                torus volume of their true value
*/

void fix_volconst()
{
  REAL adjust;
  body_id b_id;

  /* get volume of piece of unit cell */
  if ( SDIM == 2 )
   {
      adjust = web.torusv  /* /2 */;
   }
  else /* web.representation == SOAPFILM */
   {
      adjust = web.torusv  /* /6 */;
   }

  /* adjust volconsts */
  FOR_ALL_BODIES(b_id)
    if ( get_battr(b_id) & FIXEDVOL )
    { REAL vol = get_body_volume(b_id);
       REAL fix = get_body_fixvol(b_id);
       REAL vc = get_body_volconst(b_id);
       REAL calcvol = vol-vc;
       REAL newvc = fix - calcvol;
       newvc = adjust*floor(0.5+newvc/adjust);
       set_body_volconst(b_id,newvc);
       set_body_volume(b_id,calcvol+newvc);
       if ( everything_quantities_flag )
       { struct gen_quant *q = GEN_QUANTS + get_body_volquant(b_id);
          q->volconst = newvc; q->value = calcvol+newvc;
       }
    }
}

/*******************************************************************
*
*  Function: fe_reorder()
*
*  Purpose:  Order facets properly around edge.
*
*/

struct fsort { facetedge_id fe;
                    REAL angle;
                 };

static int fcompare(a,b)
struct fsort *a,*b;
{
  if ( a->angle < b->angle ) return -1;
  if ( a->angle > b->angle ) return 1;
  return 0;
}

void fe_reorder(e_id)
edge_id e_id;
{ int fcount = 0;
  facetedge_id fe;
#define FSORTMAX 100
  struct fsort fe_list[FSORTMAX];
  REAL side[MAXCOORD],norm_a[MAXCOORD],side_a[MAXCOORD];
  REAL side_b[MAXCOORD],norm_b[MAXCOORD],a_norm,b_norm;
  REAL angle;
  int i,j,k;

  /* see if we have 3 or more facets */
  generate_edge_fe_init();
  while ( generate_edge_fe(e_id,&fe) ) fcount++;
  if ( fcount <= 2 ) return;

  if ( fcount > FSORTMAX )
  { kb_error(1742,"More than 100 facets on an edge; not sorted.\n",WARNING); 
    return;
  }
  /* use first facet as reference facet */
  get_edge_side(e_id,side);     
  generate_edge_fe_init();
  generate_edge_fe(e_id,&fe);
  fe_list[0].fe = fe;
  get_fe_side(get_next_edge(fe),side_a);
  cross_prod(side,side_a,norm_a);
  a_norm = sqrt(SDIM_dot(norm_a,norm_a));
  if ( a_norm == 0.0 )
  { sprintf(errmsg,"Zero area facet %d in initial configuration.\n",
       ordinal(get_fe_facet(fe))+1);
    kb_error(1743,errmsg,WARNING);
    return;
  }
  
  /* now get angles to rest of facets */
  for ( i = 1 ; i < fcount ; i++ )
    {
       generate_edge_fe(e_id,&fe);
       fe_list[i].fe = fe;
       get_fe_side(get_next_edge(fe),side_b);
       cross_prod(side,side_b,norm_b);
       b_norm = sqrt(SDIM_dot(norm_b,norm_b));
       if ( b_norm == 0.0 )
       { sprintf(errmsg,"Zero area facet %d in initial configuration.\n",
           ordinal(get_fe_facet(fe))+1);
         kb_error(1744,errmsg,WARNING);
            continue;
       }
       angle = acos(0.999999999*SDIM_dot(norm_a,norm_b)/a_norm/b_norm);
       if ( SDIM_dot(norm_a,side_b) < 0.0 )
          angle = 2*M_PI - angle;
       fe_list[i].angle = angle;
    }

  /* sort by angle */
/* Bombed in IRIX 6 long double mode
  qsort((char *)&fe_list[1],fcount-1,sizeof(struct fsort),FCAST fcompare);
*/
  for ( j = 1 ; j < fcount-1 ; j++ )
    for ( k = j+1 ; k < fcount ; k++ )
       if ( fe_list[k].angle < fe_list[j].angle )
       { struct fsort ftemp;
          ftemp = fe_list[k];
          fe_list[k] = fe_list[j];
          fe_list[j] = ftemp;
       } 

  /* check consistency for facets on bodies */
  for ( i = 0 ; i < fcount-1 ; i++ )
  { facet_id f1,f2;

    f1 = facet_inverse(get_fe_facet(fe_list[i].fe));
    f2 = get_fe_facet(fe_list[i+1].fe);
    if ( !equal_id(get_facet_body(f1),get_facet_body(f2)) )
      /* search for one with proper body */
      { for ( j = i+2 ; j < fcount ; j++ )
        { f2 = get_fe_facet(fe_list[j].fe);
          if ( equal_id(get_facet_body(f1),get_facet_body(f2)) )
          { /* swap */
            struct fsort tmp;
            tmp = fe_list[j];
            fe_list[j] = fe_list[i+1];
            fe_list[i+1] = tmp;
            break;
          }
        }
      }
    }


  /* relink in proper order */
  for ( i = 0 ; i < fcount ; i++ )
    { set_next_facet(fe_list[i].fe,fe_list[(i+1)%fcount].fe);
       set_prev_facet(fe_list[i].fe,fe_list[(i+fcount-1)%fcount].fe);
    }
}



/**************************************************************
*
*  Function: gettok()
*
*  Purpose:  get next integer or real value, possibly
*                with '-' preceding.
*/

int gettok(kind)
int kind;
{
  int sign = 1;

  tok = yylex();
  if ( tok == ',' ) tok = yylex(); /* skip separating comma */
  if ( tok == '-' ) { sign = -1; tok = yylex(); }
  if ( tok == UMINUS_ ) { sign = -1; tok = yylex(); }
  if ( tok != kind )
  { if ( !((tok == INTEGER_) && (kind == REAL_)) )
      return tok;
    /* caller should check for error, and if error leave tok as lookahead */
  }


  int_val *= sign;
  real_val *= sign;
  tok = kind;
  return tok;
}


/*******************************************************************
*
*  function: read_const()
* 
*  purpose: read constant expression from datafile
*              and return value.
*
*  input:    REAL *value  - address for value
*
*  return:  < 0 error
*                 0 no expression
*              > 0 valid expression
*/

int read_const(value)
REAL *value;
{
  int retval;
  struct expnode node;  /* for getting constant expression */

  node.root = NULL;
  const_expr_flag = 1;
  if ( (retval = exparse(0,&node,NOUSERCOPY)) <= 0 )
     { const_expr_flag = 0;
        return retval;
     }
  *value = eval(&node,NULL,NULLID);
  const_expr_flag = 0;
  return 1;
}

/*********************************************************************
*
* function: const_expr()
*
* purpose: get numerical value of string expresssion
*
*  input:    char *str - string with expression
*              REAL *value  - address for value
*
*  return:  < 0 error
*                 0 no expression
*              > 0 valid expression
*/

int const_expr(str,value)
char *str;
REAL *value;
{ char *old_cmdptr;  /* in case in middle of another parse */
  int ret;    /* return value */

  if ( str == NULL ) return 0;
  old_cmdptr = cmdptr;
  cmdptr = str;
  yylex_init();
  ret = read_const(value);
  cmdptr = old_cmdptr;
  return ret;
}


/*********************************************************************
*
*  function: string_fixup()
*
*  purpose: put minimal facet-edges on string net, so can
*           do edge changes.  all facet-edges given the null facet.
*
*/

void string_fixup()
{
  edge_id e_id;
  facetedge_id fe;

  /* do all bare edges */
  FOR_ALL_EDGES(e_id)
    {
      fe = get_edge_fe(e_id);
      if ( valid_id(fe) ) continue;

      fe = new_facetedge(NULLFACET,e_id);
      set_next_facet(fe,fe);
      set_prev_facet(fe,fe);
      set_next_edge(fe,inverse_id(fe));
      set_prev_edge(fe,inverse_id(fe));
      set_edge_fe(e_id,fe);

    }
}


/*******************************************************
*
*  phase_initialize()
*
*  Purpose: Read in phase boundary energies from file  
*
*  Input:    Name of file with phase boundary energies
*              First line has number of phases
*              Succeeding lines have pair of phase numbers
*                      and boundary energy.
*
*/

void phase_initialize(phasename)
char *phasename;
{
  FILE *pfd;
  int i,j;         /* which phases */
  REAL value;
  char line[200];

  phase_flag = 1;

  /* save  name */
  strncpy(phase_file_name,phasename,sizeof(phase_file_name));

  pfd = path_open(phasename);
  if ( pfd == NULL )
     { sprintf(errmsg, "Cannot open phase boundary energy file %s.\n",phasename);
        kb_error(1746,errmsg, DATAFILE_ERROR);
        return;
     }

  fscanf(pfd,"%d",&phasemax);
  phase_data = dmatrix(0,phasemax,0,phasemax);
  for ( i = 0 ; i <= phasemax ; i++ )
     for ( j = 0 ; j <= phasemax ; j++ ) 
        phase_data[i][j] = 1.0;
  while ( fgets(line,sizeof(line),pfd) )
     {
#ifdef LONGDOUBLE
        if ( sscanf(line,"%d %d %Lf",&i,&j,&value) == 3 ) 
#else
        if ( sscanf(line,"%d %d %lf",&i,&j,&value) == 3 ) 
#endif
         { if ( (i < 0) || (i > phasemax) || (j < 0) || (j > phasemax) )
              { sprintf(errmsg,"Bad phase numbers: %d %d\n",i,j);
                 kb_error(1747,errmsg,DATAFILE_ERROR);
              }
            phase_data[i][j] = phase_data[j][i] = value;
         }
     }
  fclose(pfd);
}

/****************************************************************
*
*  Function: read_transforms()
*
*  Purpose:  Reads additional view transforms.
*                Works both from datafile and commandline
*/

void read_transforms(count)
int count; /* number, if known from command */
{
  int i,j,n;
  REAL value;
  MAT2D(temp_mat,MAXCOORD+1,MAXCOORD+1);

  lists_flag = 1;
  if ( count > 0 ) transform_count = count;
  else
  {
     /* find how many transforms */
     if ( (tok = gettok(INTEGER_)) != INTEGER_ )
        { kb_error(1748,"Missing number of transforms.\n",DATAFILE_ERROR);
          transform_count = 1;
        }
     else transform_count = int_val;
  }

  if ( view_transforms ) 
     { free_matrix3(view_transforms); view_transforms = NULL;}

  if ( transform_count <= 0 ) return;
  view_transforms = dmatrix3(transform_count,SDIM+1,SDIM+1);
  transform_colors = (int*)mycalloc(transform_count,sizeof(int));
  view_transform_det = (int*)mycalloc(transform_count,sizeof(int));

  /* read in transform matrices, in homogeneous coords */
  for ( n = 0 ; n < transform_count ; n++ )
     {
        tok = yylex();
        if ( tok == COLOR_ )
          { if ( (tok = gettok(INTEGER_)) != INTEGER_ )
                 kb_error(1749,"Missing transform color.\n",DATAFILE_ERROR);
             transform_colors[n] = int_val;
             transform_colors_flag = 1;
          }
        else if ( tok == SWAP_COLORS_ )
          { transform_colors[n] = SWAP_COLORS;
             transform_colors_flag = 1;
          }
        else
          { transform_colors[n] = SAME_COLOR;
             unput_tok();
          }
        for ( i = 0 ; i <= SDIM ; i++ )
         { 
            for ( j = 0 ; j <= SDIM ; j++ ) 
              { 
                 if ( read_const(&value) <= 0 )
                     { kb_error(1750,"Not enough values for transform matrix.\n",DATAFILE_ERROR);
                        return;
                      }
                 view_transforms[n][i][j] = value;
              }     
             matcopy(temp_mat,view_transforms[n],SDIM+1,SDIM+1);
             if ( determinant(temp_mat,SDIM+1) > 0.0 )
                view_transform_det[n] = 1;
             else  view_transform_det[n] = -1;
          }
    }
  transforms_flag = 1; /* default is to show */
  lists_flag = 0;
  if ( n == transform_count ) tok = yylex(); /* lookahead */
}

/****************************************************************
*
*  Function: read_transform_generators()
*
*  Purpose:  Reads view transform generators.
*                Works both from datafile and commandline
*/

void read_transform_generators(count)
int count; /* number, if known from command */
{
  int i,j,n;
  REAL value;

  lists_flag = 1;
  if ( count > 0 ) transform_count = count;
  else
  {
     /* find how many transforms */
     if ( (tok = gettok(INTEGER_)) != INTEGER_ )
        { kb_error(1751,"Missing number of transforms.\n",DATAFILE_ERROR);
          transform_gen_count = 1;
        }
     else transform_gen_count = int_val;
  }

  if ( transform_gen_swap ) myfree((char*)transform_gen_swap);
  transform_gen_swap = (int*)mycalloc(transform_gen_count+SDIM,sizeof(int));
  view_transform_gens = dmatrix3(web.torus_flag?(transform_gen_count+SDIM):
            transform_gen_count,SDIM+1,SDIM+1);

  /* read in transform matrices, in homogeneous coords */
  for ( n = 0 ; n < transform_gen_count ; n++ )
     {
        tok = yylex();
        if ( tok == SWAP_COLORS_ )
          { transform_gen_swap[n] = 1;
          }
        else unput_tok();
        for ( i = 0 ; i <= SDIM ; i++ )
         { 
            for ( j = 0 ; j <= SDIM ; j++ ) 
              { 
                 if ( read_const(&value) <= 0 )
                     { kb_error(1752,"Not enough values for transform matrix.\n",DATAFILE_ERROR);
                        return;
                      }
                 view_transform_gens[n][i][j] = value;
              }     
          }
    }
  if ( n == transform_gen_count ) tok = yylex(); /* lookahead */
  if ( web.torus_flag ) transform_gen_count += SDIM;

  transforms_flag = 0; /* default is not to show */
  lists_flag = 0;
}

/***************************************************************************
*
* function: convert_to_quantities()
*
* purpose: convert numbered quantities, surface energies, and 
*             constraint integrals to named quantities. 
*             Used in everything_quantities mode.
*/

void convert_to_quantities()
{
  int gq;
  int meth;
  char inst_name[40];
  char qname[100];
  int i,j;
  body_id b_id;
  edge_id e_id; facet_id f_id;
  int q;
  char formula[1000];
  char *gformula;

  if ( everything_quantities_flag ) return;

  outstring("Converting to all named quantities...");

  /* set up default length or area quantities */
  if ( web.representation == STRING )
  {  q = new_quantity("default_length",Q_ENERGY);
     default_area_quant_num = q;
     GEN_QUANTS[q].flags |= DEFAULT_QUANTITY; 
     if ( web.wulff_flag ) 
        kb_error(1754,"Can't do wulff energies for strings yet.\n",RECOVERABLE);

     if ( klein_metric_flag )
     {
        meth = new_method_instance("klein_length","default_length_inst");
     }
     else if ( web.conformal_flag )
     { 
        meth = new_method_instance("edge_scalar_integral","default_length_inst");
        strcpy(formula,"density*sqrt(");
        gformula = print_express(&web.metric[0][0],'X');
        if ( strlen(gformula) > sizeof(formula)-20 )
          kb_error(1755,"Conformal metric formula too long.\n",RECOVERABLE);

        strcat(formula,gformula);
        strcat(formula,")");
        METH_INST[meth].expr[0] = 
             (struct expnode *)mycalloc(1,sizeof(struct expnode));
        cmdptr = formula;
        exparse(SDIM,METH_INST[meth].expr[0],USERCOPY);
        cmdptr = NULL;
     }
     else if ( web.metric_flag )
     {
        meth = new_method_instance("edge_general_integral","default_length_inst");
        strcpy(formula,"density*sqrt(");
        for ( i = 1 ; i <= SDIM ; i++ )
          for ( j = 1 ; j <= i  ; j++ )
          { if ( i > 1 ) strcat(formula,"+");
             if ( i==j ) sprintf(formula+strlen(formula),"X%d^2*(",i);
             else sprintf(formula+strlen(formula),"2*X%dX%d*(",i,j);
             gformula = print_express(&web.metric[i-1][j-1],'X');
             if ( strlen(gformula) > sizeof(formula)-20 )
                     kb_error(1756,"Conformal metric formula too long.\n",RECOVERABLE);
             strcat(formula,gformula);
          }
        strcat(formula,"))");
        METH_INST[meth].expr[0] = 
            (struct expnode *)mycalloc(1,sizeof(struct expnode));
        cmdptr = formula;
        exparse(2*SDIM,METH_INST[meth].expr[0],USERCOPY);
        cmdptr = NULL;
     }
     else
        meth = new_method_instance("density_edge_length","default_length_inst");
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(q,"default_length_inst");
     apply_method(NULLID,"default_length_inst"); /* global method */
  }
  else
  {  q = new_quantity("default_area",Q_ENERGY);
     default_area_quant_num = q;
     GEN_QUANTS[q].flags |= DEFAULT_QUANTITY;
     if ( web.wulff_flag )
     {
        meth = new_method_instance("wulff_energy","default_area_inst");
     }
     else if ( klein_metric_flag )
     {
        meth = new_method_instance("klein_area","default_area_inst");
     }
     else if ( web.conformal_flag )
     { 
        meth = new_method_instance("facet_scalar_integral","default_area_inst");
        strcpy(formula,"density*(");
        gformula = print_express(&web.metric[0][0],'X');
        if ( strlen(gformula) > sizeof(formula)-20 )
          kb_error(1757,"Conformal metric formula too long.\n",RECOVERABLE);

        strcat(formula,gformula);
        strcat(formula,")");
        METH_INST[meth].expr[0] = 
            (struct expnode *)mycalloc(1,sizeof(struct expnode));
        cmdptr = formula;
        exparse(SDIM,METH_INST[meth].expr[0],USERCOPY);
        cmdptr = NULL;
     }
     else if ( web.metric_flag )
     {
        meth = new_method_instance("metric_facet_area","default_area_inst");
     }
     else
        meth = new_method_instance("density_facet_area","default_area_inst");
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(q,"default_area_inst");
     apply_method(NULLID,"default_area_inst");  /* global method */
  }

  /* numbered quantities */
  for ( i = 0 ; i < web.quantity_count ; i++ )
  { struct quantity *quan = web.quants + i;
     if ( quan->quanvect[0] == NULL ) continue;
     sprintf(qname,"quantity__%d",i);
     sprintf(inst_name,"quantity__%d_inst",i);
     if ( quan->attr & QFIXED )
     { gq = new_quantity(qname,Q_FIXED);
        GEN_QUANTS[gq].target = quan->target;
        quan->attr &= ~QFIXED;  /* so won't confuse things */
     }
     else gq = new_quantity(qname,Q_ENERGY);
     if ( web.representation ==  STRING )
        meth = new_method_instance("edge_vector_integral",inst_name);
     else
        meth = new_method_instance("facet_vector_integral",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     if ( quan->attr & GLOBAL ) 
         apply_method(NULLID,inst_name);
     else
     { 
        int bit = 1 << i;
        if ( web.representation == STRING )
        { FOR_ALL_EDGES(e_id)
             if ( get_e_quant_map(e_id) & bit ) 
                apply_quantity(e_id,gq);
        }
        else 
        { FOR_ALL_FACETS(f_id)
             if ( get_f_quant_map(f_id) & bit ) 
                apply_quantity(f_id,gq);
        }
     }
     for ( j = 0 ; j < SDIM ; j++ ) 
        METH_INST[meth].expr[j] = quan->quanvect[j];
     quan->attr |= USURPED_BY_QUANTITY;  /* prevent dual expr deallocation */
  }

  /* surface integrals */
  for ( i = 0 ; i < web.surfen_count ; i++ )
  { struct surf_energy *surfen = web.surfen + i;
     if ( surfen->envect[0] == NULL ) continue;
     sprintf(qname,"surface_energy__%d",i);
     sprintf(inst_name,"surface_energy__%d_inst",i);
     gq = new_quantity(qname,Q_ENERGY);
     if ( web.representation ==  STRING )
        meth = new_method_instance("edge_vector_integral",inst_name);
     else
        meth = new_method_instance("facet_vector_integral",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     if ( surfen->attr & GLOBAL ) 
         apply_method(NULLID,inst_name);
     else
     {
        int bit = 1 << i;
        if ( web.representation == STRING )
        { FOR_ALL_EDGES(e_id)
             if ( get_e_surfen_map(e_id) & bit ) 
                apply_quantity(e_id,gq);
        }
        else 
        { FOR_ALL_FACETS(f_id)
             if ( get_f_surfen_map(f_id) & bit ) 
                apply_quantity(f_id,gq);
        }
     }
     for ( j = 0 ; j < SDIM ; j++ ) 
        METH_INST[meth].expr[j] = surfen->envect[j];
     surfen->attr |= USURPED_BY_QUANTITY;  /* prevent dual expr deallocation */
  }

  /* constraint energy integrals */
  for ( i = 0 ; i < web.concount ; i++ )
  { struct constraint *con = get_constraint(i);
     if ( !(con->attr & CON_ENERGY) ) continue;
     sprintf(qname,"constraint_%d_energy",i);
     sprintf(inst_name,"constraint_%d_energy_inst",i);
     gq = new_quantity(qname,Q_ENERGY);
     GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
     if ( web.representation ==  STRING )
        meth = new_method_instance("vertex_scalar_integral",inst_name);
     else
        meth = new_method_instance("edge_vector_integral",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     if ( con->attr & GLOBAL ) 
         apply_method(NULLID,inst_name);
     else
     { vertex_id v_id;
        if ( web.representation == STRING )
        { FOR_ALL_VERTICES(v_id)
             if ( v_on_constraint(v_id,i) ) 
                apply_quantity(v_id,gq);
        }
        else 
        { FOR_ALL_EDGES(e_id)
             if ( e_on_constraint(e_id,i) ) 
                apply_quantity(e_id,gq);
        }
     }
     for ( j = 0 ; j < SDIM ; j++ ) 
        METH_INST[meth].expr[j] = con->envect[j];
     con->attr |= USURPED_BY_QUANTITY;  /* prevent dual expr deallocation */
     con->energy_method = meth;
  }


  /* boundary energy integrals */
  for ( i = 0 ; i < BDRYMAX ; i++ )
  { struct boundary *bdry = web.boundaries + i;
     if ( !(bdry->attr & CON_ENERGY) ) continue;
     sprintf(qname,"boundary_%d_energy",i);
     sprintf(inst_name,"boundary_%d_energy_inst",i);
     gq = new_quantity(qname,Q_ENERGY);
     GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
     if ( web.representation ==  STRING )
        meth = new_method_instance("vertex_scalar_integral",inst_name);
     else
        meth = new_method_instance("edge_vector_integral",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     { vertex_id v_id;
        if ( web.representation == STRING )
        { FOR_ALL_VERTICES(v_id)
             if ( bdry == get_boundary(v_id) ) 
                apply_quantity(v_id,gq);
        }
        else 
        { FOR_ALL_EDGES(e_id)
             if ( bdry == get_edge_boundary(e_id) ) 
                apply_quantity(e_id,gq);
        }
     }
     for ( j = 0 ; j < SDIM ; j++ ) 
        METH_INST[meth].expr[j] = bdry->envect[j];
     bdry->attr |= USURPED_BY_QUANTITY;  /* prevent dual expr deallocation */
     bdry->energy_method = meth;
  }

  /* bodies */
  FOR_ALL_BODIES(b_id)
      convert_body_to_quantity(b_id);

  /* gravity */
  /* if ( web.gravflag ) do it always anyway! */
  {
     sprintf(qname,"gravity_quant");
     sprintf(inst_name,"gravity_inst");
     gq = new_quantity(qname,Q_ENERGY);
     GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
     GEN_QUANTS[gq].modulus = web.grav_const;
     gravity_quantity_num = gq;
     if ( web.representation ==  STRING )
        meth = new_method_instance("string_gravity",inst_name);
     else
        meth = new_method_instance("gravity_method",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     apply_method(NULLID,inst_name);
  }

  if ( mean_curv_int_flag )
     add_standard_quantity("mean_curvature_integral",1.0);

  if ( sqgauss_flag )
     add_standard_quantity("sq_gauss_curvature",1.0);

  /* square mean curvature */
  if ( square_curvature_flag )
  {
     sprintf(qname,"sq_mean_curvature_quant");
     sprintf(inst_name,"sq_mean_curvature_inst");
     gq = new_quantity(qname,Q_ENERGY);
     GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
     sq_mean_curv_quantity_num = gq;
     if ( web.representation ==  STRING )
        meth = new_method_instance("sqcurve_string",inst_name);
     else
        meth = new_method_instance("sq_mean_curvature",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     apply_method(NULLID,inst_name);
     GEN_QUANTS[gq].modulus = globals[square_curvature_param].value.real;
  }

  if ( web.convex_flag )
  {
     sprintf(qname,"gap_quant");
     sprintf(inst_name,"gap_energy");
     gq = new_quantity(qname,Q_ENERGY);
     GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
     gap_quantity_num = gq;
     meth = new_method_instance("gap_energy",inst_name);
     METH_INST[meth].flags |= IMPLICIT_INSTANCE;
     attach_method(gq,inst_name);
     apply_method(NULLID,inst_name);
  }

  quantities_only_flag = everything_quantities_flag = 1;

  outstring("Done.\n");
}


/****************************************************************************
*
* function: convert_body_to_quantity()
*
* purpose: Convert a body volume to a named quantity. Done for
*             individual bodies so can be called when new body created,
*             i.e. rebody().
*/

void convert_body_to_quantity(b_id)
body_id b_id;
{
  int gq;
  int meth1,meth2;
  char inst_name1[30];
  char inst_name2[30];
  char qname[30];
  int i,j,k;
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe;
  char formula[100];

  i = ordinal(b_id) + 1;
  sprintf(qname,"body_%d_vol",i);
  sprintf(inst_name1,"body_%d_vol_pos",i);
  sprintf(inst_name2,"body_%d_vol_neg",i);
  if ( get_battr(b_id) & FIXEDVOL )
  { 
    gq = find_quantity(qname);
    if ( gq < 0 ) gq = new_quantity(qname,Q_FIXED);
    GEN_QUANTS[gq].target = get_body_fixvol(b_id);
  }
  else if ( get_battr(b_id) & PRESSURE )
  { gq = find_quantity(qname);
    if ( gq < 0 ) gq = new_quantity(qname,Q_ENERGY);
    GEN_QUANTS[gq].modulus = -get_body_pressure(b_id);
  }
  else { gq = find_quantity(qname);
         if ( gq < 0 ) gq = new_quantity(qname,Q_INFO);
       }
  GEN_QUANTS[gq].flags |= DEFAULT_QUANTITY;
  GEN_QUANTS[gq].b_id = b_id;
  set_body_volquant(b_id,gq);
  GEN_QUANTS[gq].volconst = get_body_volconst(b_id);
  GEN_QUANTS[gq].pressure = get_body_pressure(b_id);
  GEN_QUANTS[gq].value = get_body_volume(b_id);
  GEN_QUANTS[gq].oldvalue = get_body_volume(b_id);

  if ( web.representation == STRING )
  {
     meth1 = find_method_instance(inst_name1);
     meth2 = find_method_instance(inst_name2);
     if ( meth1 < 0 )
     { /* have to create */
        if ( web.symmetric_content )
        { 
          meth1 = new_method_instance("edge_vector_integral",inst_name1);
          METH_INST[meth1].expr[0] = 
              (struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
          meth2 = new_method_instance("edge_vector_integral",inst_name2);
          METH_INST[meth2].expr[0] = 
            (struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
          for ( j = 0 ; j < 2 ; j++ )
          { if ( j==0 )sprintf(formula,"-y/2");
             if ( j==1 )sprintf(formula,"x/2");
             cmdptr = formula;
             METH_INST[meth1].expr[j] = METH_INST[meth1].expr[0] + j;
             datafile_flag = 1;
             exparse(SDIM,METH_INST[meth1].expr[j],USERCOPY);
             cmdptr = formula;
             METH_INST[meth2].expr[j] = METH_INST[meth2].expr[0] + j;
             exparse(SDIM,METH_INST[meth2].expr[j],USERCOPY);
             datafile_flag = 0;
             cmdptr = NULL;
          }
        }
        else
        {
          meth1 = new_method_instance("edge_area",inst_name1);
          meth2 = new_method_instance("edge_area",inst_name2);
        }
        METH_INST[meth1].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
        METH_INST[meth1].modulus = 1.0;
        attach_method(gq,inst_name1);
        set_body_volmethpos(b_id,meth1);
        METH_INST[meth2].modulus = -1.0;
        METH_INST[meth2].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
        attach_method(gq,inst_name2);
        set_body_volmethneg(b_id,meth2);
     }
     FOR_ALL_EDGES(e_id)
     { fe = get_edge_fe(e_id); 
        if ( !valid_id(fe)  ) continue;
        f_id = get_fe_facet(fe);
        if ( valid_id(f_id)) 
        { if ( equal_id(b_id,get_facet_body(f_id)) )
             apply_method(e_id,inst_name1);
          else if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
             apply_method(e_id,inst_name2);
        }
        fe = get_next_facet(fe);
        f_id = get_fe_facet(fe);
        if ( valid_id(f_id)) 
        { /* check both in case of torus wrapping */
          if ( equal_id(b_id,get_facet_body(f_id)) )
             apply_method(e_id,inst_name1);
          if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
             apply_method(e_id,inst_name2);
        }
     }
  }
  else  /* SOAPFILM */
  {
     meth1 = find_method_instance(inst_name1);
     meth2 = find_method_instance(inst_name2);
     if ( meth1 < 0 )
     { /* have to create */
        if ( web.symmetric_content )
        {
          meth1 = new_method_instance("facet_vector_integral",inst_name1);
          meth2 = new_method_instance("facet_vector_integral",inst_name2);
          METH_INST[meth1].expr[0] = 
            (struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
          METH_INST[meth2].expr[0] = 
            (struct expnode *)mycalloc(SDIM,sizeof(struct expnode));
          for ( j = 0 ; j < SDIM ; j++ )
          { sprintf(formula,"x%d/%d",j+1,SDIM);
             cmdptr = formula;
             METH_INST[meth1].expr[j] = METH_INST[meth1].expr[0] + j;
             datafile_flag = 1;
             exparse(SDIM,METH_INST[meth1].expr[j],USERCOPY);
             cmdptr = formula;
             METH_INST[meth2].expr[j] = METH_INST[meth2].expr[0] + j;
             exparse(SDIM,METH_INST[meth2].expr[j],USERCOPY);
             datafile_flag = 0;
             cmdptr = NULL;
          }
        }
        else
        {
          meth1 = new_method_instance("facet_volume",inst_name1);
          meth2 = new_method_instance("facet_volume",inst_name2);
        }
        METH_INST[meth1].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
        METH_INST[meth1].modulus = 1.0;
        attach_method(gq,inst_name1);
        set_body_volmethpos(b_id,meth1);
        METH_INST[meth2].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
        METH_INST[meth2].modulus = -1.0;
        attach_method(gq,inst_name2);
        set_body_volmethneg(b_id,meth2);
     }
     FOR_ALL_FACETS(f_id)
     { /* check both in case of torus wrapping */
        if ( equal_id(b_id,get_facet_body(f_id)) )
            apply_method(f_id,inst_name1);
        if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
            apply_method(f_id,inst_name2);
     }
  }

  /* constraint content integrals */
  for ( j = 0 ; j < web.concount ; j++ )
     { struct constraint *con = get_constraint(j);
        vertex_id v_id;

        if ( !(con->attr & CON_CONTENT) ) continue;
        sprintf(inst_name1,"body_%d_con_%d_pos",i,j);
        sprintf(inst_name2,"body_%d_con_%d_neg",i,j);
        meth1 = find_method_instance(inst_name1);
        meth2 = find_method_instance(inst_name2);
        if ( meth1 < 0 )
        { /* have to create */
          if ( web.representation ==  STRING )
          { meth1 = new_method_instance("vertex_scalar_integral",inst_name1);
            meth2 = new_method_instance("vertex_scalar_integral",inst_name2);
          }
          else
          { meth1 = new_method_instance("edge_vector_integral",inst_name1);
            meth2 = new_method_instance("edge_vector_integral",inst_name2);
          }
          METH_INST[meth2].modulus = -1.0;
          METH_INST[meth1].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
          METH_INST[meth2].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
          attach_method(gq,inst_name1);
          attach_method(gq,inst_name2);
          con = get_constraint(j); /* may have moved */
          con->content_method_1 = meth1;
          con->content_method_2 = meth2;
          if ( web.representation == STRING )
             METH_INST[meth1].expr[0] = METH_INST[meth2].expr[0] = con->convect[0];
          else
            for ( k = 0 ; k < SDIM ; k++ ) 
             METH_INST[meth1].expr[k] = METH_INST[meth2].expr[k] = con->convect[k];
        }
        if ( web.representation == STRING )
        { FOR_ALL_VERTICES(v_id)
             if ( v_on_constraint(v_id,j) ) 
             { edge_id first_e = e_id = get_vertex_edge(v_id);
               do
                { facetedge_id first_fe = fe = get_edge_fe(e_id);
                  do
                  { f_id = get_fe_facet(fe);
                     if ( equal_id(b_id,get_facet_body(f_id)) )
                         apply_method(v_id,inst_name2);
                     if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
                         apply_method(v_id,inst_name1);
                     fe = get_next_facet(fe);
                  } while ( !equal_id(fe,first_fe) );
                  e_id = get_next_tail_edge(e_id);
                } while ( !equal_id(first_e,e_id));
             }
        }
        else  /* SOAPFILM */
        { FOR_ALL_EDGES(e_id)
             if ( e_on_constraint(e_id,j) ) 
             { facetedge_id first_fe = fe = get_edge_fe(e_id);
                do
                { f_id = get_fe_facet(fe);
                  if ( equal_id(b_id,get_facet_body(f_id)) )
                  {  apply_method(e_id,inst_name1);
                  }
                  if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
                  {  apply_method(e_id,inst_name2);
                  }
                  fe = get_next_facet(fe);
                } while ( !equal_id(fe,first_fe) );
             }
        }
     }

  /* boundary content integrals */
  for ( j = 0 ; j < BDRYMAX ; j++ )
     { struct boundary *bdry = web.boundaries + j;
        vertex_id v_id;

        if ( !(bdry->attr & CON_CONTENT) ) continue;
        sprintf(inst_name1,"body_%d_bdry_%d_pos",i,j);
        sprintf(inst_name2,"body_%d_bdry_%d_neg",i,j);
        meth1 = find_method_instance(inst_name1);
		  meth2 = find_method_instance(inst_name2);
        if ( meth1 < 0 )
        { /* have to create */
          if ( web.representation ==  STRING )
          { meth1 = new_method_instance("vertex_scalar_integral",inst_name1);
            meth2 = new_method_instance("vertex_scalar_integral",inst_name2);
          }
          else
          { meth1 = new_method_instance("edge_vector_integral",inst_name1);
            meth2 = new_method_instance("edge_vector_integral",inst_name2);
          }
          METH_INST[meth2].modulus = -1.0;
          METH_INST[meth1].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
          METH_INST[meth2].flags |= DEFAULT_INSTANCE | BODY_INSTANCE;
          attach_method(gq,inst_name1);
          attach_method(gq,inst_name2);
          bdry->content_method_1 = meth1;
          bdry->content_method_2 = meth2;
          if ( web.representation == STRING )
             METH_INST[meth1].expr[0] = METH_INST[meth2].expr[0] = bdry->convect[0];
          else
            for ( k = 0 ; k < SDIM ; k++ ) 
             METH_INST[meth1].expr[k] = METH_INST[meth2].expr[k] = bdry->convect[k];
        }
        if ( web.representation == STRING )
        { FOR_ALL_VERTICES(v_id)
             if ( bdry == get_boundary(v_id) ) 
             { edge_id first_e = e_id = get_vertex_edge(v_id);
                do
                { facetedge_id first_fe = fe = get_edge_fe(e_id);
                  do
                  { f_id = get_fe_facet(fe);
                     if ( equal_id(b_id,get_facet_body(f_id)) )
                         apply_method(v_id,inst_name1);
                     if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
                         apply_method(v_id,inst_name2);
                     fe = get_next_facet(fe);
                  } while ( !equal_id(fe,first_fe) );
                  e_id = get_next_tail_edge(e_id);
                } while ( !equal_id(first_e,e_id));
             }
        }
        else  /* SOAPFILM */
        { FOR_ALL_EDGES(e_id)
             if ( bdry == get_edge_boundary(e_id)  ) 
             { facetedge_id first_fe = fe = get_edge_fe(e_id);
                do
                { f_id = get_fe_facet(fe);
                  if ( equal_id(b_id,get_facet_body(f_id)) )
                  {  apply_method(e_id,inst_name1);
                  }
                  if ( equal_id(b_id,get_facet_body(inverse_id(f_id))) )
                  {  apply_method(e_id,inst_name2);
                  }
                  fe = get_next_facet(fe);
                } while ( !equal_id(fe,first_fe) );
             }
        }
     }

  if ( web.pressure_flag && ( get_battr(b_id) & FIXEDVOL ) )
  { /* create formula and do compound quantity */
    int mc = GEN_QUANTS[gq].method_count;
    char *volsum = temp_calloc(40,mc);
    char *formula = temp_calloc(2,40*mc + 100);
    int q;
    sprintf(qname,"body_%d_ambient_energy",i);
    q = find_quantity(qname);
    if ( q < 0 ) q = new_quantity(qname,Q_INFO);
    GEN_QUANTS[q].flags |= DEFAULT_QUANTITY;
    sprintf(volsum,"body[%d].volconst",i);
    for ( j = 0 ; j < mc ; j++ )
    { int m = GEN_QUANTS[gq].meth_inst[j];
      struct method_instance *mi = METH_INST+m;
      strcat(volsum,((mi->modulus > 0.0) ? "+" : "+"));
      strcat(volsum,mi->name);
      strcat(volsum,".value");
      mi->flags |= Q_COMPOUND;
    }
    GEN_QUANTS[q].method_count = 0;
    sprintf(formula,"-ambient_pressure_value*(body[%d].target*(log(%s)-log(body[%d].target))-(%s-body[%d].target))",
       i,volsum,i,volsum,i);
    cmdptr = formula;
    exparse(0,&(GEN_QUANTS[q].expr),USERCOPY);
    cmdptr = NULL;
    GEN_QUANTS[q].flags |= Q_COMPOUND;
    GEN_QUANTS[q].flags |= Q_ENERGY;
    GEN_QUANTS[gq].flags &= ~Q_FIXED;
    GEN_QUANTS[gq].flags |= Q_INFO;
    temp_free(volsum);
    temp_free(formula);
  }  
}
