/*
 * ruleset.cpp
 *
 * Part of ezbounce
 * 
 * (C) 1998, 1999 Murat Deligonul
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 * --- 
 * * [11-8-98] made some member functions const
 * * Using linkedlist class for lists 11-7-98
 * * Ruleset-related code moved here 08-30-98
 * 
 */

#include "autoconf.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "ruleset.h"
#include "general.h"
#include "config.h"

bool smart_match(const char *hostname, const char *pattern); 
bool port_in_set(u_short port, const char *set);

rule_set * rule_set::pFirst = NULL;
const char rule_set::FROM      = 70;
const char rule_set::TO        = 69;
const int  rule_set::UNLIMITED = -1;


/* 
 *  Checks if port is in set 'set'. 
 */
bool port_in_set(u_short port, const char *set)
{
    int cur = 1;
    if (strcasecmp(set, "all") == 0)
        return 1;
    /* 
     * Find tokens seperated by commas.
     * Will also get the first and only one if there are no commas.
     */
    do {
        char buff[15];
        if (!gettok(set, buff, 15, ',', cur++))
            break;
        if (strchr(buff, '-'))
        {
            char min[6], max[6];
            unsigned int _min, _max;
            /* x1-x2; get them both and compare */
            gettok(buff, min, sizeof(min), '-', 1);
            gettok(buff, max, sizeof(max), '-', 2);
            /* we are kind enough to trim down min/max if they 
               are too big */
            _min = (unsigned) atoi(min); _max = (unsigned) atoi(max);
            _min = (_min > 65536) ? 65536 : _min;
            _max = (_max > 65536) ? 65536 : _max;
            if (port >= (u_short)_min && port <= (u_short) _max)
                return 1;
        }
        else {
            if (port == u_short(atoi(buff)))
                return 1;
        }
    } while (69);
    return 0;
}

/* 
 * Checks if address works as an ip address..
 * Works if wildcards are in it too.
 */
static bool is_ip_address(const char *address)
{
    bool legal = 0;
    do {
        if (*address >= '0' && *address <= '9')
            legal = 1;
        else if (*address == '?' || *address == '*' || *address == '.')
            continue;
        else
            return 0;
    } while (*++address);
    return legal;
}

/*
 * smart_[and slow_]match
 * Hostname can be either an ip address or its resolved form.
 * Will try to convert hostname to type of pattern and then wild card match
 * them. It won't do that w/ the pattern however.
 */
bool smart_match(const char *hostname, const char *pattern)
{
    if (!hostname || !pattern)
    {
        fprintf(stderr, "%s NULL pointer(s) given to smart_match()!!\n", timestamp());
        return 0;
    }
    if (strcmp(pattern,"*") == 0)
        return 1;
    bool is_pattern_ip = is_ip_address(pattern),
                         is_host_ip = is_ip_address(hostname);

    /* pattern and host are same type */
    if (is_pattern_ip == is_host_ip)
        return wild_match(pattern, hostname);
    else if (!is_pattern_ip && is_host_ip && !(config.flags & NO_REVERSE_LOOKUPS))
    {  
        /* host is an ip address, but pattern is not */
        char resolved[512];
        if (reverse_lookup(hostname, resolved, 512))
            return wild_match(pattern, resolved);
        return 0;
    }
    else if (is_pattern_ip && !is_host_ip && !(config.flags & NO_REVERSE_LOOKUPS))
    {  
        /* host is not an ip address, but pattern is */
        struct sockaddr_in sin;
        if (resolve_address(hostname, &sin))
            return wild_match(pattern, inet_ntoa(sin.sin_addr));
    }
    return 0;
}



/* functions for rule_set class */
rule_set::rule_set(void)
{
    if (pFirst) {
        rule_set *r;
        for (r = pFirst; r->pNext; r = r->pNext)
            ;
        r->pNext = this;
    }
    else
        pFirst = this;
    pNext = NULL;     
    obsolete = 0; 
    num_registered_from = num_registered_to = 0;
    num_froms = 0; num_tos = 0;
#ifdef __DEBUG__
    fprintf(stderr, "CONSTRUCTOR: rule_set: %p\n", this);
#endif 
}

rule_set::~rule_set(void)
{
    if (pFirst == this)
        pFirst = pNext;
    else {
        for (rule_set *r = pFirst; r; r = r->pNext)
        {
            if (r->pNext == this)
            {
                r->pNext = pNext;
                break;
            }
        }
    }  
    /* clean up rs_hosts */
    list_iterator<rs_host> i(&hosts);
    do {
        delete i.get();
    } while (++i);
    
#ifdef __DEBUG__
    fprintf(stderr, "DESTRUCTOR: rule_set: %p\n", this);
#endif 
}

/*
 * Loops through rule_set objects and fills prs with a list of them
 * that contain the hostname & port given.
 */

bool rule_set::find_matching_sets(const char *host, unsigned short port, 
                                  obj_set<rule_set> *prs) 
{
    int numfound = 0;
    for (rule_set *r = pFirst; r; r = r->pNext)
    {
        if (!r->obsolete && r->does_match(host, port, rule_set::FROM))
        {
            prs->add(r);
            numfound++;
        }
    }
    return (numfound != 0);
}


/*
 *   Code for allowed_rule_set class:
 *   a config file block can be :
 *  allow {
 *     [num] from (address) [ports] : Num is optional.
 *                                     Max # of clients
 *                                     allowed to connect from that address.
 *                                     Defaults to -1, which means unlimited.
 *                              address:
 *                                    Address to permit clients from
 *                              ports: 
 *                                    What ports should it allow/deny to:
 *                                    "all"
 *                                    "6667"
 *                                    "6660-7000"
 *                                    "7000,4000,40355,29421,500-2403,6660-6666"
 *           
 *     [num] to (address) [ports]  : blah blah
 *   }
 *   There must be at least one `from' and one 'to' entry.
 *   `Reason' arguments to below functions are ignored.
 *              
 */

bool allowed_rule_set::add_host_to(const char *address_to, const char *ports, 
                                   const char *, int max_num)
{
    rs_host * r = new rs_host(TO, address_to, ports, NULL, max_num);
    if (r)
    {
        num_tos++;
        hosts.add(r);
    }
    return (r != 0);
}


bool allowed_rule_set::add_host_from(const char *address_from, const char *ports, 
                                     const char *, int max_num)
{
    rs_host * r = new rs_host(FROM, address_from, ports, NULL, max_num);
    if (r) 
    {
        num_froms++;
        hosts.add(r);
    }
    return (r != 0);
}

bool allowed_rule_set::does_match(const char *from, unsigned short port, char t) 
const
{
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p; (p = i.get()); i++)
    {
        if (p->type == t && port_in_set(port, p->ports) && 
            smart_match(from, p->address))
            return 1;
    }
    return 0;
}

/*
 *  Checks if this host is allowed entry by this rule list.
 *  return values:  
 *    1 - User is allowed
 *    0 - No matching hosts found
 *   -1 - User is banned - 'buff' is filled w/ the reason.. 
 *        for the case of the allowed_rule_set class, it will only be returned
 *        in case user limit for rule set was exceeded
 *                    
 *  buff = buffer to hold reason for banning, if any.
 *  
 */
int allowed_rule_set::is_allowed(const char *address, unsigned short port, 
                                 char *buff, size_t bufflen) const
{
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p; (p = i.get()); i++)
    {
        if (p->type == FROM && port_in_set(port, p->ports)
            && smart_match(address, p->address))
        {
            /* user is allowed, but are there anymore ppl permitted.. ? */
            if (p->num >= p->max && p->max != UNLIMITED)
            {
                safe_strcpy(buff, "No more connections allowed from your host.", bufflen);
                return -1;
            }
            else if (p->num < p->max || p->max == UNLIMITED)
                /* 
                  Enough room for this guy. 
                   Only one match is needed
                */
                return 1;
        }
    }
    return 0;
}

/* 
 * Like above but checks if it 'from' may connect to 'to' on port.
 * Even though there is a 'from' argument, we assume that the caller 
 * belongs to this rule set (for now)..
 */
int allowed_rule_set::is_allowed_to(const char *, const char *to, 
                                    unsigned short port, char *buff, size_t bufflen) 
const
{
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p; (p = i.get()); i++)
    {
        if (p->type == TO && smart_match(to, p->address) && 
            port_in_set(port, p->ports))
        {
            /* again, check for max */
            if (p->num >= p->max && p->max != UNLIMITED)
            {
                safe_strcpy(buff, "No more users from your host are allowed to connect there.", 
                            bufflen);
                return -1;
            }
            else if (p->num < p->max || p->max == UNLIMITED)
                return 1;
        }
    }
    return 0; 
}   

/* 
 *  Return values by register_connection and friends:
 *  1 - Connection [un]registered successfully
 *  0 - ''          ''            unsuccessfully
 * -1 - full ?
 */
int allowed_rule_set::register_connection(char t, const char *from, const char *to,
                                          u_short port)
{
    /* first find the matching rs_hosts */
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p; (p = i.get()); i++)
    {
        if (p->type == t && smart_match((t == FROM ? from : to),p->address) && port_in_set(port, p->ports))
        {
            /* check again if max has been exceeded */
            if (p->num >= p->max && p->max != UNLIMITED)
                return -1;
            else {
                p->num++;
                return rule_set::register_connection(t, from, to, port);
            }
        }
    }
    return 0;
}

int allowed_rule_set::unregister_connection(char t, 
        const char *from, const char *to, u_short port)
{
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p;( p = i.get()); i++)
    {
        if (p->type == t && smart_match((t == FROM ? from : to),p->address) 
            && port_in_set(port, p->ports))
        {
            p->num--;
            rule_set::unregister_connection(t, from, to, port);
            return 1;
        }
    }
    return 0;
}

/*
 *  Stuff for denied_rule_set.
 *  Syntax is basically same but:
 *
 *     * A from w/o a to or vice versa is permitted
 *     * In a rule set w/ no from entries is_allowed will always
 *        return 1. is_allowed_to will check as usual. does_match() will
 *        also always return 1 in such case.
 *     * max # nubmers are ignored here
 *     * If a rule set has both from and tos, the tos are only
 *        applied to the from addresses. The addresses in the form fields
 *        will be granted passage
 */

bool denied_rule_set::add_host_to(const char *address, const char *ports, 
                                  const char *reason4ban, int)          
{
    rs_host * r = new rs_host(TO, address, ports, reason4ban, 0);
    if (r)
    {   
        num_tos++;
        hosts.add(r);
    }
    return (r  != 0);  
}

bool denied_rule_set::add_host_from(const char *address, const char *ports,
                                    const char *reason4ban, int)
{
    rs_host * r = new rs_host(FROM, address, ports, reason4ban, 0);
    if (r)
    {
        num_froms++;
        hosts.add(r);
    }
    return (r != 0);
}

/* 
 *  NOTE: in the case that there are no entries of type 't',
 *        1 will be returned.
 */
bool denied_rule_set::does_match(const char *from, unsigned short port, char t) 
const
{
    unsigned num_these = 0;
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p; (p = i.get()); i++)
    {
        if (p->type == t)
        {
            if (port_in_set(port, p->ports) && smart_match(from, p->address))
                return 1;
            num_these++;
        }
    }
    if (!num_these)
        return 1;
    return 0;
}

/* 
 *  Returns:
 *    -1 - if banned
 *     1 - not found in list, so he must be allowed  
 *  notes:
 *     if there are 'to' entries as well as 'from' we auto let this one in
 */
int denied_rule_set::is_allowed(const char *address, u_short port, 
                                char *buff, size_t bufflen) const
{
    if (!num_tos && !num_froms)
        return 1;
    else 
    {
        list_iterator<rs_host> i(&hosts);
        for (rs_host *p; (p = i.get()); i++)
            if (p->type == FROM && smart_match(address, p->address) && port_in_set(port, p->ports))
                return (safe_strcpy(buff, p->reason, bufflen), -1);
    }
    return 0;

}

/*
 *  return: 
 *     -1: not allowed to connect
 *      0: not in list, does not necesssarily mean he may connect there tho
 */
int denied_rule_set::is_allowed_to(const char *, const char *address_to, u_short port,
                                   char *buff, size_t bufflen) const 

{
    list_iterator<rs_host> i(&hosts);
    for (rs_host *p = i.get(); (p = i.get()); i++)
        if (p->type == TO && smart_match(address_to, p->address) && port_in_set(port, p->ports))
            return (safe_strcpy(buff, p->reason, bufflen), -1);
    return 0;       
}


inline int denied_rule_set::register_connection(char t, const char*,const char *, u_short)
{
    return rule_set::register_connection(t, NULL, NULL, 0), 1;
}

inline int denied_rule_set::unregister_connection(char t, const char*,const char *, u_short)
{
    return rule_set::unregister_connection(t, NULL, NULL, 0), 1;
}

/* 
 * Compare two rule_sets
 * Ignores num_registered_xxx and obsolete and rs_host.num
 * Note! - Since this operates on the rule_set base class, there 
 *     is no way of knowing for sure if the *type* of the object we are being compared
 *     to is the same type as us. (Actually each rs_host of an Allowed_rule_set has 
 *     reason set to NULL, and each denined_rule_set should have it set to something
 *     other than that, because the parsing code will give 'No reason was given!'..)
 *     (ok, maybe you can make this virtual and do it that way somehow)
 *
 */
bool rule_set::operator == (rule_set& r2)
{
    list_iterator<rs_host> rs_it(&hosts);
    list_iterator<rs_host> rs2_it(&r2.hosts);
    rs_host * rs2, * rs;
    if (!(num_froms == r2.num_froms) ||
        !(num_tos   == r2.num_tos))
        return 0;
    rs = rs_it.get();
    for (; (rs2 = rs2_it.get()); rs_it++)
    {
        /* fixme - we check for reason first because it can be null for
         * allowed_rule_set types. However we assume the other pointers are all vaild */
        if ((!rs2) || (rs->reason && !rs2->reason) || (!rs->reason && rs->reason))
            return 0;
        if ((rs->type != rs2->type) ||
            (rs->max  != rs2->max)  ||
            /* (rs->num  != rs2->num)  || */
            strcasecmp(rs->address, rs2->address) ||
            strcasecmp(rs->ports, rs2->ports) || 
            (rs->reason && rs2->reason && strcasecmp(rs->reason, rs2->reason)))
            return 0;
        rs2 = ++rs2_it;
    }
    return 1;         
}

rule_set::rs_host::~rs_host()
{
    delete[] reason;
    delete[] address;
    delete[] ports;     
}

rule_set::rs_host::rs_host(char _type, const char *_addr, 
                           const char *_ports , const char *_reason, int _max)
{
    max = _max; 
    num = 0;
    type = _type;
    reason = my_strdup(_reason);
    address = my_strdup(_addr);
    ports   = my_strdup(_ports);
}
