/* main.h
 * 
 * Copyright (C) 1998-2001  Oskar Liljeblad
 *
 * This file is part of icoutils.
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file "LICENSE" for
 * details.
 */

#include "icotool.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

/*
 *
 * Variables
 *
 */

char *program_name;
FILE *verbose_file;
static ImageFormat image_format;
static char *output = NULL;
static char *output_base = NULL;
static IcoToolCommand command = COMMAND_LIST;
static int verbosity = 0;
int image_width = -1;
int image_height = -1;
int image_color_count = -1;
ResourceType resource_type = RESOURCE_UNSPECIFIED;
static int resource_index = -1;
int cursor_hotspot_x = -1;
int cursor_hotspot_y = -1;
ImageMode image_mode = IMAGEMODE_BOTH;

/*
 * 
 * Macros
 *
 */

/* VERSION_MESSAGE:
 * Message to display when the --version option is used.
 */
#define VERSION_MESSAGE \
PROGRAM " (" PACKAGE ") " VERSION "\n\
Written by Oskar Liljeblad.\n\
\n\
Copyright (C) 1998-2001 Oskar Liljeblad\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"

/* HELP_MESSAGE:
 * Message to display when the --help option is used.
 */
#define HELP_MESSAGE "\
Usage: %s [OPTION]... [FILE]...\n\
Convert or display information on Win32 icon (.ico) files.\n\
\n\
Commands:\n\
  -x, --extract           extract images from file\n\
  -l, --list              output list of images in file\n\
  -c, --create            create icon files from specified images\n\
\n\
Filters:\n\
  -i, --index=NUMBER      index of image to perform operation on\n\
  -w, --width=PIXELS      width of image to perform operation on\n\
  -h, --height=PIXELS     height of image to perform operation on\n\
  -C, --colors=COUNT      number of colors in image to perform operation on\n\
  -X, --hotspot-x=COORD   cursor hotspot x-coordinate to match\n\
  -Y, --hotspot-y=COORD   cursor hotspot y-coordinate to match\n\
      --icon              match icons only\n\
      --cursor            match cursors only\n\
  -a, --all               perform operation on all images (default)\n\
\n\
Command options:\n\
  -o, --output=PATH       where to place extracted files\n\
  -O, --output-base=BASE  basename of extracted files\n\
      --format=FORMAT     extraction format; `xpm' or `ppm'\n\
  -m, --image-mode=MODE   what to extract; `image' for image data only, \n\
                          `mask' for mask only, `both' for both (default)\n\
  -v, --verbose           explain what is being done\n\
      --help              display this help and exit\n\
      --version           output version information and exit\n\
\n\
Report bugs to <" BUG_EMAIL ">.\n"

static struct option long_options[] = {
    { "extract",	no_argument,       NULL, 'x' },
    { "list",	    	no_argument, 	   NULL, 'l' },
    { "create",     	no_argument,       NULL, 'c' },
    { "verbose",	no_argument, 	   NULL, 'v' },
    { "version", 	no_argument, 	   NULL, 150 },
    { "help", 	    	no_argument,	   NULL, 151 },
    { "format",     	required_argument, NULL, 152 },
    { "image-mode",	required_argument, NULL, 'm' },
    { "output", 	required_argument, NULL, 'o' },
    { "output-base",	required_argument, NULL, 'O' },
    { "index",	    	required_argument, NULL, 'i' },
    { "width",	    	required_argument, NULL, 'w' },
    { "height", 	required_argument, NULL, 'h' },
    { "colors", 	required_argument, NULL, 'C' },
    { "hotspot-x",      required_argument, NULL, 'X' },
    { "hotspot-y",      required_argument, NULL, 'Y' },
    { "icon",       	no_argument,       NULL, 153 },
    { "cursor",     	no_argument,       NULL, 154 },
    { "all",	    	no_argument, 	   NULL, 'a' },
    { 0, 0, 0, 0 }
};

/*
 *
 * Function declarations
 *
 */

static bool process_file(char *filename, bool single);
static int parse_format_option(char *format);
static char *get_image_format_extension(int format);
static bool is_icon_matched(IconFile *itf, int index, IconImage *icon);
static void *read_file(FILE *file, size_t *total_size);

/*
 *
 * Functions
 *
 */

int
main(int argc, char **argv)
{
    int c;

    program_name = basename(argv[0]);
    if (program_name == NULL)
	program_name = "icotool";

    if (!strncasecmp(program_name, "ico2", 4)
    	    || !strncasecmp(program_name, "icoto", 5)) {
    	char *extension = &program_name[strlen(program_name)-3];

	image_format = parse_format_option(extension);
	if (image_format == FORMAT_INVALID) {
	    command = COMMAND_LIST;
	    image_format = FORMAT_XPM;
	    program_name = "icotool";
	} else {
    	    command = COMMAND_EXTRACT;
	}
    }

    while (TRUE) {
	int option_index = 0;
	c = getopt_long(argc, argv, "xlcvm:o:O:i:w:h:C:X:Y:a",
	    	    	long_options, &option_index);
	if (c == EOF)
	    break;

	switch (c) {
	case 'o': /* --output */
	    output = optarg;
	    output_base = NULL;
	    break;

	case 'O': /* --output-base */
	    output_base = optarg;
	    output = NULL;
	    break;

	case 'v': /* --verbose */
	    verbosity++;
	    break;

	case 'l': /* --list */
	    command = COMMAND_LIST;
	    break;

	case 'x': /* --extract */
	    command = COMMAND_EXTRACT;
	    break;

	case 'c': /* --create */
	    command = COMMAND_CREATE;
	    break;

	case 'a': /* --all */
	    resource_index = -1;
	    image_width = -1;
	    image_height = -1;
	    image_color_count = -1;
	    resource_type = -1;
	    cursor_hotspot_x = -1;
	    cursor_hotspot_y = -1;
	    break;

	case 153: /* --icon */
	    resource_type = RESOURCE_ICON;
	    break;

	case 154: /* --cursor */
	    resource_type = RESOURCE_CURSOR;
	    break;

	case 'i': /* --index */
	    if (!xstrtoi(optarg, &resource_index)) {
		put_error("invalid index: %s\n", optarg);
		exit(1);
	    }
	    break;

	case 'w': /* --width */
	    if (!xstrtoi(optarg, &image_width)) {
		put_error("invalid width: %s\n", optarg);
		exit(1);
	    }
	    break;

	case 'h': /* --height */
	    if (!xstrtoi(optarg, &image_height)) {
		put_error("invalid height: %s\n", optarg);
		exit(1);
	    }
	    break;

	case 'C': /* --colors */
	    if (!xstrtoi(optarg, &image_color_count)) {
		put_error("invalid color count: %s\n", optarg);
		exit(1);
	    }
	    break;

    	case 'X': /* --hotspot-x */
	    if (!xstrtoi(optarg, &cursor_hotspot_x)) {
		put_error("invalid hotspot coordinate: %s\n", optarg);
		exit(1);
	    }
	    break;

    	case 'Y': /* --hotspot-y */
	    if (!xstrtoi(optarg, &cursor_hotspot_y)) {
		put_error("invalid hotspot coordinate: %s\n", optarg);
		exit(1);
	    }
	    break;

	case 'm': /* --image-mode */
	    if (!strcasecmp(optarg, "mask") || !strcasecmp(optarg, "m")) {
		image_mode = IMAGEMODE_MASK;
	    } else if (!strcasecmp(optarg, "image") || !strcasecmp(optarg, "i")) {
		image_mode = IMAGEMODE_IMAGE;
	    } else if (!strcasecmp(optarg, "both") || !strcasecmp(optarg, "b")) {
		image_mode= IMAGEMODE_BOTH;
	    } else {
		put_error("invalid image mode: %s\n", optarg);
		exit(1);
	    }
	    break;

	case 150: /* --version */
	    printf(VERSION_MESSAGE);
	    exit(0);

	case 151: /* --help */
	    printf(HELP_MESSAGE, program_name);
	    exit(0);

	case 152: /* --format */
	    image_format = parse_format_option(optarg);
	    if (image_format == FORMAT_INVALID) {
		put_error("invalid image format: %s\n", optarg);
		exit(1);
	    }
	    break;

	case '?':
	default:
	    exit(1);
    	}
    }

    verbose_file = (output == NULL && output_base == NULL ? stderr : stdin);

    /* At least one file argument is required. */
    if (optind >= argc) {
	put_error("missing file argument\n");
	fprintf(stderr, "Try `%s --help' for more information.\n", program_name);
	exit(1);
    }

    /* --create not implemented yet. */
    if (command == COMMAND_CREATE) {
	put_error("--create option not implemented yet\n");
	/* make_ico_file(argc - optind, &argv[optind]); */
	exit(1);
    }

    for (c = optind ; c < argc ; c++)
    	process_file(argv[c], (optind+1 == argc));

    exit(0);
}
    
/* process_file:
 * Perform the by user specified command on the file - list or extract.
 */
static bool
process_file(char *filename, bool single)
{
    IconFile iconfile;
    FILE *file;
    int c;

    iconfile.name = filename;
    if (!strcmp(filename, "-")) {
    	file = stdin;
	filename = "(standard input)";
    } else {
    	file = fopen(filename, "rb");
	if (file == NULL) {
	    put_error("%s: %s\n", filename, strerror(errno));
	    return FALSE;
	}
    }

    iconfile.contents = read_file(file, &iconfile.size);
    fclose(file);
    if (iconfile.contents == NULL) {
	put_error("%s: %s\n", filename, strerror(errno));
	return FALSE;
    }

    if (!initialize_icotool_file(&iconfile)) {
    	free(iconfile.contents);
	return FALSE;
    }

    if (command == COMMAND_LIST && !single)
	put_message(0, "%s:\n", iconfile.name);

    for (c = 0; c < iconfile.ci_dir->count; c++) {
	IconImage icon;

    	if (!initialize_icotool_icon(&iconfile, c, &icon)) {
            free(iconfile.contents);
	    return FALSE;
	}

    	if (is_icon_matched(&iconfile, c, &icon)) {
	    if (command == COMMAND_LIST) {
		print_icotool_icon(&iconfile, c, &icon);
	    } else if (command == COMMAND_EXTRACT) {
		make_xpm_image(&iconfile, &icon);
	    }
    	}
    }

    if (command == COMMAND_LIST && !single)
	put_message(0, "\n");

    free(iconfile.contents);
    return TRUE;
}

/* parse_format_option:
 * Return a FORMAT_-constant for the named image format.
 */
static int
parse_format_option(char *format)
{
    if (!strcasecmp(format, "xpm"))
	return FORMAT_XPM;
    if (!strcasecmp(format, "ppm"))
	return FORMAT_PPM;
    return FORMAT_INVALID;
}

/* get_image_format_extension:
 * Return the extension for files of the specified image format.
 */
static char *
get_image_format_extension(int format)
{
    switch (format) {
    case FORMAT_XPM:
    	return ".xpm";
    case FORMAT_PPM:
    	return ".ppm";
    default:
    	return "";
    }
}

/* get_destination_name:
 * Make a filename for an image that is to be extracted.
 */
char *
get_destination_name(IconFile *fi, int width, int height, int color_count)
{
    char *filename;

    /* If --output[-base] not specified, write to standard out. */
    if (output == NULL && output_base == NULL)
	return NULL;

    if (output_base != NULL) {
	xasprintf(&filename, "%s_%d_%dx%d%s",
		output_base, color_count, width, height,
		get_image_format_extension(image_format));
	return filename;
    }

    if (is_directory(output) || lastchar(output) == '/') {
	xasprintf(&filename, "%s%s%s_%d_%dx%d%s",
	    	output, (lastchar(output) != '/' ? "/" : ""),
		basename(fi->name), color_count, width, height,
		get_image_format_extension(image_format));
	return filename;
    }

    /* Otherwise, just return the --output argument */
    return xstrdup(output);
}

/* is_icon_matched
 * Return TRUE if the icon/cursor at specific offset matches given
 * arguments.
 */
static bool
is_icon_matched(IconFile *itf, int index, IconImage *icon)
{
    if (resource_type != RESOURCE_UNSPECIFIED && resource_type == itf->resource_type)
	return FALSE;
    if (cursor_hotspot_x != -1 && icon->entry->hotspot_x != cursor_hotspot_x)
    	return FALSE;
    if (cursor_hotspot_y != -1 && icon->entry->hotspot_y != cursor_hotspot_y)
    	return FALSE;
    if (resource_index != -1 && resource_index != index)
	return FALSE;
    if (image_width != -1 && icon->entry->width != image_width)
	return FALSE;
    if (image_height != -1 && icon->entry->height != image_height)
	return FALSE;
    if (image_color_count != -1 && icon->color_count != image_color_count)
	return FALSE;
    return TRUE;
}

#define READ_MEMORY_CHUNK 32768

/* read_file:
 * Read all of file into memory, returning the allocated memory.
 * NULL is returned in case of an I/O error.
 */
static void *
read_file(FILE *file, size_t *total_size)
{
    uint8_t *memory = NULL;
    size_t mem_size = 0;
    size_t file_size = 0;

    while (TRUE) {
    	size_t read;

	if (file_size >= mem_size) {
	    mem_size += READ_MEMORY_CHUNK;
	    memory = xrealloc(memory, mem_size);
	}

	read = fread(memory+file_size, 1, mem_size-file_size, file);
    	if (read <= 0) {
	    if (!feof(file)) {
	    	printf("!!! %d %d %d\n", read, mem_size, file_size);
	    	free(memory);
		return NULL;
	    }
	    *total_size = file_size;
	    return memory;
	}

	file_size += read;
    }
}

/* put_message:
 * Print a message if verbosity high enough.
 */
void
put_message(int min_verbosity_level, char *format, ...)
{
    va_list args;

    va_start(args, format);
    if (min_verbosity_level <= verbosity)
	vfprintf(verbose_file, format, args);
    va_end(args);
}
