/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IANIMATOR_H
#define IANIMATOR_H


#include "iobject.h"
#include "iviewmodulecomponent.h"
#include "ierrorstatusholder.h"
#include "ideletable.h"


#include "iarray.h"
#include "imath.h"
#include "iimage.h"


class iAnimatorEventObserver;
class iAnimatorScript;
class iCameraPath;
class iStereoImageArray;


struct iAnimatorState 
{
	int mode;
	int nframes;
	int nBlendedFrames;
	int nTransitionFrames;
	float dphi;
	float dtheta;
	float dscale;
	float droll;
	float slideSpeed;
	float flybySpeed;
	double xsecPos;
	bool ifBoundingBox;
	bool ifColorBars;
	bool ifTimeLabel;
	int currec;
	int cameraProjection;
	double cameraPosition[3];
	double cameraFocalPoint[3];
	double cameraViewUp[3];
	double cameraParallelScale;
	double cameraClippingRange[2];
	iString titlePageFile;
	int titlePageNumBlendedFrames;
	int titlePageNumFrames;
	iString logoFile;
	float logoOpacity;
	int logoPosition;
	bool stopOnPath;
	int posOnPath;
};


class vtkPolyData;
class iCameraPath;
class vtkImageData;


class iAnimator : public iObject, public iViewModuleComponent, public iErrorStatusHolder, public iDeletable
{
	
	friend class iAnimatorScript;

public:
	
	IOBJECT_DECLARE_SELF(iAnimator,iObject);

	static iAnimator* New(iViewModule *vm);
	static const iObjectType& Type();
		
	IOBJECT_DECLARE_GETSET(Mode,mState.mode,int);
	//virtual void SetMode(int ma);
	//inline int GetMode() const { return mState.mode; }

	IOBJECT_DECLARE_GETSET(NumberOfFrames,mState.nframes,int);
	//virtual void SetNumberOfFrames(int na);
	//inline int GetNumberOfFrames() const { return mState.nframes; }

	IOBJECT_DECLARE_GETSET(Phi,mState.dphi,float);
	//virtual void SetDPhi(float va);
	//inline float GetDPhi() const { return mState.dphi; }

	IOBJECT_DECLARE_GETSET(Theta,mState.dtheta,float);
	//virtual void SetDTheta(float va);
	//inline float GetDTheta() const { return mState.dtheta; }

	IOBJECT_DECLARE_GETSET(Zoom,mState.dscale,float);
	//virtual void SetDZoom(float va);
	//inline float GetDZoom() const { return mState.dscale; }

	IOBJECT_DECLARE_GETSET(Roll,mState.droll,float);
	//virtual void SetDRoll(float va);
	//inline float GetDRoll() const { return mState.droll; }

	IOBJECT_DECLARE_GETSET(FlybySpeed,mState.flybySpeed,float);
	//virtual void SetFlybySpeed(float va);
	//inline float GetFlybySpeed() const { return mState.flybySpeed; }

	IOBJECT_DECLARE_GETSET(NumberOfBlendedFrames,mState.nBlendedFrames,int);
	//virtual void SetNumberOfBlendedFrames(int na);
	//inline int GetNumberOfBlendedFrames() const { return mState.nBlendedFrames; }

	IOBJECT_DECLARE_GETSET(NumberOfTransitionFrames,mState.nTransitionFrames,int);
	//virtual void SetNumberOfTransitionFrames(int na);
	//inline int GetNumberOfTransitionFrames() const { return mState.nTransitionFrames; }

	IOBJECT_DECLARE_GETSET(NumberOfTitlePageFrames,mState.titlePageNumFrames,int);
	//virtual void SetNumberOfTitlePageFrames(int n);
	//inline int GetNumberOfTitlePageFrames() const { return mState.titlePageNumFrames; }
	
	IOBJECT_DECLARE_GETSET(NumberOfTitlePageBlendedFrames,mState.titlePageNumBlendedFrames,int);
	//virtual void SetNumberOfTitlePageBlendedFrames(int n);
	//inline int GetNumberOfTitlePageBlendedFrames() const { return mState.titlePageNumBlendedFrames; }

	IOBJECT_DECLARE_GETSET(LogoPosition,mState.logoPosition,int);
	//virtual void SetLogoPosition(int n);
	//inline int GetLogoPosition() const { return mState.logoPosition; }

	IOBJECT_DECLARE_GETSET(StopOnPath,mState.stopOnPath,bool);
	//virtual void SetStopOnPath(bool n);
	//inline bool GetStopOnPath() const { return mState.stopOnPath; }

	IOBJECT_DECLARE_GETSET(PositionOnPath,mState.posOnPath,int);
	//virtual void SetPositionOnPath(int n);
	//inline int GetPositionOnPath() const { return mState.posOnPath; }

	IOBJECT_DECLARE_GETSET(LogoOpacity,mState.logoOpacity,float);
	//virtual void SetLogoOpacity(float v);
	//inline float GetLogoOpacity() const { return mState.logoOpacity; }

	IOBJECT_DECLARE_GETSET1(UseScript,bool);
	//virtual void SetUseScript(bool s);
	//inline bool GetUseScript() const { return mUseScript; }

	IOBJECT_DECLARE_GETSET1(RestoreCamera,bool);
	//virtual void SetRestoreCamera(bool s);
	//inline bool GetRestoreCamera() const { return mRestoreCamera; }

	IOBJECT_DECLARE_GETSET1(InheritSettings,bool);
	//virtual void SetInheritSettings(bool s);
	//inline bool GetInheritSettings() const { return mInheritSettings; }

	IOBJECT_DECLARE_GETSET1(ScriptFileName,const iString&);
	//virtual void SetScriptFileName(const iString &s);
	//inline const String& GetScriptFileName() const { return mScriptFileName; }

	IOBJECT_DECLARE_GETSET1(DebugFlag,int);
	//virtual void SetDebugFlag(int s);
	//inline int GetDebugFlag() const { retun mDebugFlag; }
	static const iObjectKey& KeyDebugging();

	IOBJECT_DECLARE_GETSET(NumberOfCameraPathSteps,mNumPathSteps,int);
	//virtual void SetNumberOfCameraPathSteps(int n);
	//inline int GetNumberOfCameraPathSteps() const { return mNumPathSteps; }
	
	IOBJECT_DECLARE_GETSET2(NumberOfCameraPathHandles,int);
	//virtual void SetNumberOfCameraPathHandles(int n);
	//int GetNumberOfCameraPathHandles() const;

	IOBJECT_DECLARE_GETSET2(CameraPathClosed,bool);
	//virtual void SetCameraPathClosed(bool s);
	//bool GetCameraPathClosed() const;
	
	IOBJECT_DECLARE_GETSET2(FocalPathEnabled,bool);
	//virtual void SetFocalPathEnabled(bool s);
	//bool GetFocalPathEnabled() const;
	
	IOBJECT_DECLARE_GETSET2(FocalPathToPoint,bool);
	//virtual void SetFocalPathToPoint(bool s);
	//bool GetFocalPathToPoint() const;

	bool SetTitlePageFile(const iString &s);
	IOBJECT_DECLARE_GET(TitlePageFile,mState.titlePageFile,const iString&);
	//inline const iString& GetTitlePageFile() const { return mState.titlePageFile; }

	bool SetLogoFile(const iString &s);
	IOBJECT_DECLARE_GET(LogoFile,mState.logoFile,const iString&);
	//inline const iString& GetLogoFile() const { return mState.logoFile; }

	inline const iImage& GetTitlePageImage() const { return mTitlePageImage; }
	static const iObjectKey& KeyTitlePageImage();
	inline const iImage& GetLogoImage() const { return mLogoImage; }
	static const iObjectKey& KeyLogoImage();

	virtual void SetSlideSpeed(float va);
	inline float GetSlideSpeed() const { return (float)fabs(mState.slideSpeed); }
	static const iObjectKey& KeySlideSpeed();

	static const iObjectKey& KeyCameraPathX();
	static const iObjectKey& KeyCameraPathY();
	static const iObjectKey& KeyCameraPathZ();
	static const iObjectKey& KeyFocalPathX();
	static const iObjectKey& KeyFocalPathY();
	static const iObjectKey& KeyFocalPathZ();
	static const iObjectKey& KeyCameraPathDemo();

	//
	//  Animator controls
	//
	void Animate();
	virtual void Start();
	virtual void Finish();
	virtual bool Continue();

	inline iAnimatorEventObserver* GetObserver() const { return mAnimatorObserver; }
	inline iAnimatorScript* GetScript() const { return mAnimatorScript; }
	inline void GetInfo(bool &nr, int &cr, int &cf) const { nr = mNewRec; cr = mCurRec; cf = mCurFrame; }
	virtual void CopyState(iAnimator *);

	//
	//  Camera path functions
	//
	void SetCameraPathColor(iColor &c);
	bool IsCameraPathValid() const;
	void PlayCameraPathDemo();

protected:

	iAnimator(iViewModule *vm);
	virtual ~iAnimator();

	void PositionCameraOnThePath();

	bool RenderImages(bool dumpImage = true);
	void Reset();
	void ResetCurrentFile();
	void ResetState();
	void SaveState();
	void RestoreState();

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

private:

	bool mStarted, mStartedRender, mUseScript, mRestoreCamera, mInheritSettings;
	long mSeed;
	int mDebugFlag;

	iString mScriptFileName, mScriptText;

	iAnimatorState mState;
	iAnimatorState mState2;

	float mRandStep;
	bool mNewRec;
	int mPrevRec, mCurRec, mTotFrame, mCurFrame, mNumPathSteps;

	iArray<iStereoImageArray*> mBlenderBase;

	iImage mTitlePageImage, mLogoImage;
	bool mDoingTitlePage;

	iAnimatorScript *mAnimatorScript;
	iAnimatorEventObserver *mAnimatorObserver;

	iCameraPath *mCameraPath;

	//
	//  Internal (work) variables
	//
	struct AnimatorData
	{
		double pos[3];
		float r, dphl0, dthl0, ramp;
		double xc1[3], xc2[3], x[3];
		float v[3], t, dt0, dt;
	} wData;
};


inline void iAnimator::SetNumberOfFrames(int na)
{ 
	mState.nframes = na;
	this->ClearCache();
}


inline void iAnimator::SetPhi(float va)
{ 
	mState.dphi = va; 
	this->ClearCache();
}


inline void iAnimator::SetTheta(float va)
{ 
	mState.dtheta = va; 
	this->ClearCache();
}


inline void iAnimator::SetZoom(float va)
{ 
	mState.dscale = va; 
	this->ClearCache();
}


inline void iAnimator::SetRoll(float va)
{ 
	mState.droll = va; 
	this->ClearCache();
}


inline void iAnimator::SetSlideSpeed(float va)
{ 
	mState.slideSpeed = va; 
	this->ClearCache();
}


inline void iAnimator::SetFlybySpeed(float va)
{ 
	mState.flybySpeed = va; 
	this->ClearCache();
}


inline void iAnimator::SetNumberOfBlendedFrames(int na)
{ 
	mState.nBlendedFrames = na; 
	this->ClearCache();
}


inline void iAnimator::SetNumberOfTransitionFrames(int na)
{ 
	mState.nTransitionFrames = na; 
	this->ClearCache();
}


inline void iAnimator::SetNumberOfTitlePageFrames(int n)
{ 
	mState.titlePageNumFrames = n; 
	this->ClearCache();
}


inline void iAnimator::SetNumberOfTitlePageBlendedFrames(int n)
{ 
	mState.titlePageNumBlendedFrames = n; 
	this->ClearCache();
}


inline void iAnimator::SetLogoPosition(int n)
{ 
	mState.logoPosition = n; 
	this->ClearCache();
}


inline void iAnimator::SetLogoOpacity(float v)
{ 
	mState.logoOpacity = v; 
	this->ClearCache();
}


inline void iAnimator::SetStopOnPath(bool s)
{ 
	mState.stopOnPath = s; 
	this->ClearCache();
}


inline void iAnimator::SetPositionOnPath(int s)
{ 
	if(s>=0 && s<mNumPathSteps)
	{
		mState.posOnPath = s;
		if(mState.mode == 4) this->PositionCameraOnThePath();
		this->ClearCache();
	}
}


inline void iAnimator::SetUseScript(bool s)
{ 
	mUseScript = s; 
	this->ClearCache();
}


inline void iAnimator::SetRestoreCamera(bool s)
{ 
	mRestoreCamera = s; 
	this->ClearCache();
}


inline void iAnimator::SetInheritSettings(bool s)
{ 
	mInheritSettings = s; 
	this->ClearCache();
}


inline void iAnimator::SetScriptFileName(const iString &s)
{ 
	mScriptFileName = s; 
	this->ClearCache();
}

#endif // IANIMATOR_H

