/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icolorbars.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatastretch.h"
#include "ierror.h"
#include "imath.h"
#include "ipalette.h"
#include "itextactor.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkLookupTable.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkRenderer.h>
#include <vtkScalarBarActor.h>

using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"


IOBJECT_DEFINE_TYPE(iColorBars,ColorBars,cb,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iColorBars,Automatic,a,Bool,1);
IOBJECT_DEFINE_KEY(iColorBars,BarLeft,bl,Int,3);
IOBJECT_DEFINE_KEY(iColorBars,BarRight,br,Int,3);
IOBJECT_DEFINE_KEY(iColorBars,Color,c,Color,1);
IOBJECT_DEFINE_KEY(iColorBars,SideOffset,so,Float,1);


const struct iColorBars::QueueItem iColorBars::NullQueueItem = { 0, 0, 0, iMath::_IntMax, 0 };


//
//  helper struct
//
class iColorBarsLabel
{
	
public:

	iTextActor *Base[2];
	iTextActor *Power[2];

	iColorBarsLabel(const iRenderTool *parent)
	{
		int j;
		for(j=0; j<2; j++)
		{
			Base[j] = iTextActor::New(parent); IERROR_ASSERT(Base[j]);
			Power[j] = iTextActor::New(parent); IERROR_ASSERT(Power[j]);
			Base[j]->SetBold(true);
			Power[j]->SetBold(true);
			Power[j]->SetHJustification(_TextJustLeft);
		}
		Base[0]->SetHJustification(_TextJustLeft);
		Base[1]->SetHJustification(_TextJustRight);
	}

	~iColorBarsLabel()
	{
		int j;
		for(j=0; j<2; j++)
		{
			Base[j]->Delete();
			Power[j]->Delete();
		}
	}
};


//
//  Main class
//
iColorBars* iColorBars::New(iViewModule *vm)
{
	return new iColorBars(vm);
}


iColorBars::iColorBars(iViewModule *vm) : iOverlayBase(vm->GetRenderTool()), iViewModuleComponent(vm), mAutomaticBars(NullQueueItem), mManualBars(NullQueueItem), mLabels(0)
{
	int i;

	mAutomatic = true;
	mFontType = _TextTypeBitmap;
	mStarted = false;
	mScale = 1.0;

	mJmax = 0;
	mWidth = 0.03;
	mHeight = 0.7;
	mYOff = 0.2;

	//
	//  Create manual bars
	//
	mManualBars.Add(NullQueueItem);
	mManualBars.Add(NullQueueItem);
	mManualBars[0].Priority = -2;
	mManualBars[1].Priority = -1;
	mManualBars[0].DataTypeId = mManualBars[1].DataTypeId = 1;
	mManualBars[0].Count = mManualBars[1].Count = 1;

	for(i=0; i<2; i++)
	{
		mBarActor[i] = vtkScalarBarActor::New(); IERROR_ASSERT(mBarActor[i]);
		mBarActor[i]->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
		mBarActor[i]->SetOrientationToVertical();
		mBarActor[i]->SetMaximumNumberOfColors(256);
		mBarActor[i]->SetWidth(mWidth);
		mBarActor[i]->SetHeight(mHeight);
		mBarActor[i]->SetNumberOfLabels(0);
		mBarActor[i]->SetTitle("");
		mBarActor[i]->PickableOff();
	}

	for(i=0; i<2; i++)
	{
		mTitleActor[i] = iTextActor::New(this->GetRenderTool()); IERROR_ASSERT(mTitleActor[i]);
		mTitleActor[i]->Rotate(90.0);
		mTitleActor[i]->SetVJustification(_TextJustCenter);
		mTitleActor[i]->SetBold(true);
	}
	mTitleActor[0]->SetHJustification(_TextJustRight);
	mTitleActor[1]->SetHJustification(_TextJustLeft);

	iColorBarsLabel *l;
	for(i=0; i<10; i++)
	{
		l = new iColorBarsLabel(this->GetRenderTool()); IERROR_ASSERT(l);
		mLabels.Add(l);
	}

	this->SetSideOffset(0.08);
	this->SetColor(iColor());
	this->SetFontScale(mFontScale);
	this->SetFontType(mFontType);
}


iColorBars::~iColorBars()
{
	int i;

	for(i=0; i<2; i++) mBarActor[i]->Delete();

	for(i=0; i<2; i++)
	{
		mTitleActor[i]->Delete();
	}

	for(i=0; i<mLabels.Size(); i++) delete mLabels[i];
}


void iColorBars::SetSideOffset(float v)
{
	if(v>0.01 && v<0.2)
	{
		mXOff[1] = v; 
		mXOff[0] = 1.0 - mWidth - mXOff[1]; 
		mTextOff[0] = mXOff[0] + mWidth + 0.0; mTextOff[1] = mXOff[1] - 0.01;
		mTitleOff[0] = mXOff[0] - 0.3*mWidth; mTitleOff[1] = mXOff[1] + mWidth;
		mTitleActor[0]->SetPosition(mTitleOff[0],0.5);
		mTitleActor[1]->SetPosition(mTitleOff[1],0.5);
		mBarActor[0]->SetPosition(mXOff[0],mYOff);
		mBarActor[1]->SetPosition(mXOff[1],mYOff);
		this->ClearCache();
		this->Modified();
	}
}


void iColorBars::SetColor(iColor c)
{
	int i, j;
	for(i=0; i<2; i++)
	{
		mTitleActor[i]->SetColor(c);
		for(j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->SetColor(c);
			mLabels[j]->Power[i]->SetColor(c);
		}
	}
	mColor = c;
	this->ClearCache();
}

	
void iColorBars::SetFontScale(int s)
{
	int i, j;
	for(i=0; i<2; i++)
	{
		mTitleActor[i]->SetFontScale(s+1);
		for(j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->SetFontScale(s);
			mLabels[j]->Power[i]->SetFontScale(s);
		}
	}
	mFontScale = mLabels[0]->Base[0]->GetFontScale();
	this->ClearCache();
}

	
void iColorBars::SetFontType(int s)
{
	int i, j;
	for(i=0; i<2; i++)
	{
		mTitleActor[i]->SetFontType(_TextTypeVector);
		for(j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->SetFontType(s);
			mLabels[j]->Power[i]->SetFontType(s);
		}
	}
	mFontType = mLabels[0]->Base[0]->GetFontType();
	this->ClearCache();
}

	
void iColorBars::ReleaseGraphicsResources(vtkWindow *win)
{
	int i, j;
	for(i=0; i<2; i++) 
	{
		mBarActor[i]->ReleaseGraphicsResources(win);
		mTitleActor[i]->ReleaseGraphicsResources(win);
		for(j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->ReleaseGraphicsResources(win);
			mLabels[j]->Power[i]->ReleaseGraphicsResources(win);
		}
	}
	this->vtkProp::ReleaseGraphicsResources(win);
}


int iColorBars::RenderOverlay(vtkViewport* viewport)
{
	if(!mStarted) return 0;

	int i;
	for(i=0; i<this->Queue().Size(); i++)  if(this->Queue()[i].Count>0 && this->Queue()[i].Palette!=0) 
	{
		mBarActor[i]->RenderOverlay(viewport);
		mTitleActor[i]->RenderOverlay(viewport);

		for(int j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->RenderOverlay(viewport);
			mLabels[j]->Power[i]->RenderOverlay(viewport);
		}
	}
	
	return 1;
}


int iColorBars::RenderOpaqueGeometry(vtkViewport* viewport)
{
	int i, j;
	float w1 = 0.0, h1 = 0.0, w2 = 0.0, h2 = 0.0;

	if(!mStarted) mStarted = true;

	if(this->GetRenderingMagnification() == 1)
	{
		for(i=0; i<2; i++) 
		{
			mBarActor[i]->SetPosition(mXOff[i],mYOff);
			mBarActor[i]->SetWidth(mWidth);
			mBarActor[i]->SetHeight(mHeight);
		}

		for(i=0; i<this->Queue().Size(); i++) if(this->Queue()[i].Count>0 && this->Queue()[i].Palette!=0) 
		{
			int nstep;

			iDataLimits *lim = this->GetViewModule()->GetReader()->GetLimits(iDataType::FindTypeById(this->Queue()[i].DataTypeId));
			if(lim==0 || this->Queue()[i].Var>=lim->GetNumVars()) continue;

			float dv, voff, vstep;
			float vl = iDataStretch::ApplyStretch(lim->GetLowerLimit(this->Queue()[i].Var),lim->GetStretch(this->Queue()[i].Var),false);
			float vu = iDataStretch::ApplyStretch(lim->GetUpperLimit(this->Queue()[i].Var),lim->GetStretch(this->Queue()[i].Var),true);
			switch(lim->GetStretch(this->Queue()[i].Var))
			{
			case _StretchLog:
				{
					//
					//  Log table
					//
					int ivl = 1 + (int)floor(vl-0.01);
					int ivu = 0 + (int)floor(vu+0.01);
					dv = vu - vl;

					if(dv<=0.0 || ivl>vu) continue;

					voff = ivl - vl;
					int ndex = ivu - ivl;
					if(ndex > 1) nstep = 1 + ndex/mLabels.Size(); else nstep = 1;

					mJmax = 1 + ndex/nstep;
					vstep = nstep;

					for(j=0; j<mJmax; j++)
					{
						int pow = ivl + nstep*j;
						float s1[2], s2[2];
						char s[10];
						sprintf(s,"%d",pow);

						if(pow == 0)
						{
							mLabels[j]->Base[i]->SetText(" 1"); 
							mLabels[j]->Power[i]->SetText(""); 
						} 
						else 
						{
							mLabels[j]->Base[i]->SetText("10"); 
							mLabels[j]->Power[i]->SetText(s);
						}

						mLabels[j]->Base[i]->GetSize(viewport,s1);
						mLabels[j]->Power[i]->GetSize(viewport,s2);
						if(w1 < s1[0]) w1 = s1[0];
						if(h1 < s1[1]) h1 = s1[1];
						if(w2 < s2[0]) w2 = s2[0];
						if(h2 < s2[1]) h2 = s2[1];	
					}
					break;
				}
			default:
				{
					//
					//  Lin table
					//
					mJmax = 5;
					voff = 0.0;
					if(fabs(vl) < 1.0e-3*fabs(vu)) vl = 0.0;
					if(fabs(vu) < 1.0e-3*fabs(vl)) vu = 0.0;
					dv = vu - vl;
					vstep = dv/(mJmax-1);

					for(j=0; j<mJmax; j++)
					{
						float v = vl + vstep*j;
						float s1[2], s2[2];
						char s[100];
						sprintf(s,"%g",v);

						mLabels[j]->Base[i]->SetText(s); 
						mLabels[j]->Power[i]->SetText(""); 

						mLabels[j]->Base[i]->GetSize(viewport,s1);
						mLabels[j]->Power[i]->GetSize(viewport,s2);
						if(w1 < s1[0]) w1 = s1[0];
						if(h1 < s1[1]) h1 = s1[1];
						if(w2 < s2[0]) w2 = s2[0];
						if(h2 < s2[1]) h2 = s2[1];
					}
				}
			}

			w1 *= 0.8;
			w2 *= 0.8;
			h1 *= 0.8;
			
			float xpos, ypos;
			xpos = mTextOff[i];
			for(j=0; j<mJmax; j++)
			{
				ypos = mYOff + 0.86*mHeight*(voff+vstep*j)/dv;
				
				if(i == 0) 
				{
					mLabels[j]->Base[i]->SetPosition(xpos,ypos-0.5*h1);
					mLabels[j]->Power[i]->SetPosition(w1+xpos,ypos+0.5*h1);
				} 
				else 
				{
					mLabels[j]->Base[i]->SetPosition(xpos-w2,ypos-0.5*h1);
					mLabels[j]->Power[i]->SetPosition(xpos-w2,ypos+0.5*h1);
				}
			}
			
			for(j=mJmax; j<mLabels.Size(); j++)
			{
				mLabels[j]->Base[i]->SetText(""); 
				mLabels[j]->Power[i]->SetText(""); 
			}

			mTitleActor[i]->SetText(lim->GetName(this->Queue()[i].Var).ToCharPointer());
			
			mBarActor[i]->SetLookupTable(this->GetViewModule()->GetControlModule()->GetPalette(iMath::Abs(this->Queue()[i].Palette)-1)->GetLookupTable(this->Queue()[i].Palette < 0));
		}
	}
	else
	{
		int mag = this->GetRenderingMagnification();
		for(i=0; i<2; i++)
		{
			mBarActor[i]->SetWidth(mag*mWidth);
			mBarActor[i]->SetHeight(mag*mHeight);
		}
		//
		//  Shift positions if under magnification - 
		//
		int winij[2];
		this->ComputePositionShiftsUnderMagnification(winij);
		
		for(i=0; i<2; i++) mBarActor[i]->SetPosition(mag*mXOff[i]-winij[0],mag*mYOff-winij[1]);
	}
	//
	//  Render everything
	//
	for(i=0; i<this->Queue().Size(); i++) if(this->Queue()[i].Count>0 && this->Queue()[i].Palette!=0) 
	{
		mBarActor[i]->RenderOpaqueGeometry(viewport);
		mTitleActor[i]->RenderOpaqueGeometry(viewport);
		for(j=0; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base[i]->RenderOpaqueGeometry(viewport);
			mLabels[j]->Power[i]->RenderOpaqueGeometry(viewport);
		}
	}	
		
	return 1;
}


void iColorBars::ShowBar(int priority, int v, const iDataType &dt, int p, bool show)
{
	int i;
	
	if((!show && this->Queue().Size()<1) || v<0) return;

	iDataLimits *lim = this->GetViewModule()->GetReader()->GetLimits(dt);
	if(lim==0 || (show && v>=lim->GetNumVars())) return;

	QueueItem tmp;
	tmp.Var = v;
	tmp.Palette = p;
	tmp.DataTypeId = dt.GetId();
	tmp.Priority = priority;
	i = this->Queue().Find(tmp);

	if(i >= 0) 
	{
		if(show) this->Queue()[i].Count++; else 
		{
			this->Queue()[i].Count--;
			if(this->Queue()[i].Count == 0) // delete the entry
			{
				this->Queue().iArray<QueueItem>::Remove(i);
			}
		}
	}
	else 
	{
		if(show)
		{
			//
			//  Insert a new one keeping the list ordered by priority
			//
			tmp.Count = 1;
			this->Queue().Add(tmp);
		}
		else
		{
			// cannot delete non-existing entry
		}
	}
	this->ClearCache();
	this->Modified();
}


void iColorBars::SetAutomatic(bool automatic)
{
	mAutomatic = automatic;
	this->ClearCache();
	this->Modified();
}


void iColorBars::SetBar(int bar, int v, int id, int p)
{
	if(bar>=0 && bar<2)
	{
		mManualBars[bar].Var = v;
		mManualBars[bar].Palette = p;
		mManualBars[bar].DataTypeId = id;
		this->ClearCache();
	}
}


void iColorBars::GetBar(int bar, int &v, int &id, int &p) const
{
	if(bar<0 || bar>1)
	{
		v = id = p = 0;
	}
	else
	{
		v = mManualBars[bar].Var;
		p = mManualBars[bar].Palette;
		id = mManualBars[bar].DataTypeId;
	}
}

//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iColorBars::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyColor(),mColor);
	this->PackValue(s,KeySideOffset(),this->GetSideOffset());
	this->PackValue(s,KeyAutomatic(),this->GetAutomatic());

	int iv[3];
	this->GetBar(0,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	this->PackValue(s,KeyBarRight(),iv,3);
	this->GetBar(1,iv[0],iv[1],iv[2]);
	iv[0]++;
	iv[2]++;
	this->PackValue(s,KeyBarLeft(),iv,3);
}


void iColorBars::UnPackStateBody(const iString &s)
{
	int iv[3]; bool a; float f; iColor c;

	if(this->UnPackValue(s,KeyColor(),c)) this->SetColor(c);
	if(this->UnPackValue(s,KeySideOffset(),f)) this->SetSideOffset(f);

	a = this->GetAutomatic();
	this->UnPackValue(s,KeyAutomatic(),a);
	this->SetAutomatic(false);

	this->GetBar(0,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	if(this->UnPackValue(s,KeyBarRight(),iv,3)) this->SetBar(0,iv[0]-1,iv[1],iv[2]-1);
	this->GetBar(1,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	if(this->UnPackValue(s,KeyBarLeft(),iv,3)) this->SetBar(1,iv[0]-1,iv[1],iv[2]-1);

	this->SetAutomatic(a);
}


bool iColorBars::QueueItem::operator<(const QueueItem &item) const
{
	return Priority < item.Priority;
}


bool iColorBars::QueueItem::operator==(const QueueItem &item) const
{
	return (Var==item.Var && DataTypeId==item.DataTypeId && Palette==item.Palette); 
}

