/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imeasuringboxinteractorstyle.h"


#include "ierror.h"
#include "imeasuringbox.h"
#include "itransform.h"

#include <vtkCamera.h>
#include <vtkCommand.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkRenderer.h>
#include <vtkRenderWindowInteractor.h>


iMeasuringBoxInteractorStyle* iMeasuringBoxInteractorStyle::New()
{
	return new iMeasuringBoxInteractorStyle;
}


void iMeasuringBoxInteractorStyle::FindPickedActor(int x, int y)
{ 
	this->InteractionProp = mBox; x = y = 0; 
}


//
//  Panning in XYZ - a copy of vtkInteractorStyleTrackballActor::Pan() with 
//  one change
//
void iMeasuringBoxInteractorStyle::UniformScale()
{
	this->InteractionProp = mBox;
	if (this->CurrentRenderer == 0 || this->InteractionProp == 0)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// Use initial center as the origin from which to pan
	
	double *obj_center = this->InteractionProp->GetCenter();
	
	double disp_obj_center[3], new_pick_point[4], old_pick_point[4], motion_vector[3];
	
	this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
		disp_obj_center);
	
	this->ComputeDisplayToWorld((double)rwi->GetEventPosition()[0], 
		(double)rwi->GetEventPosition()[1], 
		disp_obj_center[2],
		new_pick_point);
	
	this->ComputeDisplayToWorld(double(rwi->GetLastEventPosition()[0]), 
		double(rwi->GetLastEventPosition()[1]), 
		disp_obj_center[2],
		old_pick_point);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	float x = rwi->GetEventPosition()[0];
	float y = rwi->GetEventPosition()[1];
	float oldX = rwi->GetLastEventPosition()[0];
	float oldY = rwi->GetLastEventPosition()[1];
	
	float d = sqrt(vtkMath::Distance2BetweenPoints(new_pick_point,old_pick_point));
	
	double wc[4], wx[4], wy[4], wz[4];
	double ex[3], ey[3], ez[3];
	double dx[2];
	int i;
	
	dx[0] = x - oldX;
	dx[1] = y - oldY;
	vtkMath::Normalize2D(dx);
	
	this->ComputeWorldToDisplay(0.0,0.0,0.0,wc);
	this->ComputeWorldToDisplay(1.0,0.0,0.0,wx);
	this->ComputeWorldToDisplay(0.0,1.0,0.0,wy);
	this->ComputeWorldToDisplay(0.0,0.0,1.0,wz);
	
	for(i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
		ez[i] = wz[i] - wc[i];
	}
	
	vtkMath::Normalize2D(ex);
	vtkMath::Normalize2D(ey);
	vtkMath::Normalize2D(ez);
	
	double crossProduct[3];
	crossProduct[0] = dx[0]*ex[0] + dx[1]*ex[1];
	crossProduct[1] = dx[0]*ey[0] + dx[1]*ey[1];
	crossProduct[2] = dx[0]*ez[0] + dx[1]*ez[1];
	
	for(i=0; i<3; i++) crossProduct[i] = pow((double)crossProduct[i],9.0);
	
	vtkMath::Normalize(crossProduct);
	
	motion_vector[0] = d*crossProduct[0];
	motion_vector[1] = d*crossProduct[1];
	motion_vector[2] = d*crossProduct[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != 0)
    {
		iTransform *t;
		t = iTransform::New(); IERROR_ASSERT(t);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(motion_vector[0], motion_vector[1], motion_vector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(motion_vector);
    }
	
	if (this->AutoAdjustCameraClippingRange)
    {
		this->CurrentRenderer->ResetCameraClippingRange();
    }
	
	rwi->Render();
}
//
//  Panning in screen - a copy of vtkInteractorStyleTrackballActor::PanXY() with 
//  one change
//
void iMeasuringBoxInteractorStyle::Pan()
{
	this->InteractionProp = mBox;
	if (this->CurrentRenderer == 0 || this->InteractionProp == 0)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	// Use initial center as the origin from which to pan
	
	double *obj_center = this->InteractionProp->GetCenter();
	
	double disp_obj_center[3], new_pick_point[4], old_pick_point[4], motion_vector[3];
	
	this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
		disp_obj_center);
	
	this->ComputeDisplayToWorld((double)rwi->GetEventPosition()[0], 
		(double)rwi->GetEventPosition()[1], 
		disp_obj_center[2],
		new_pick_point);
	
	this->ComputeDisplayToWorld(double(rwi->GetLastEventPosition()[0]), 
		double(rwi->GetLastEventPosition()[1]), 
		disp_obj_center[2],
		old_pick_point);
	//
	//  Below is the only difference from vtkInteractorStyleTrackballActor::PanXY()
	//
	// float x = rwi->GetEventPosition()[0];
	float y = rwi->GetEventPosition()[1];
	// float oldX = rwi->GetLastEventPosition()[0];
	float oldY = rwi->GetLastEventPosition()[1];
	
	float d = sqrt(vtkMath::Distance2BetweenPoints(new_pick_point,old_pick_point));
	if(y > oldY) d = -d;
	
	double wc[4], wx[4], wy[4];
	double ex[3], ey[3], ez[3];
	
	this->ComputeDisplayToWorld(0.0,0.0,disp_obj_center[2],wc);
	this->ComputeDisplayToWorld(1.0,0.0,disp_obj_center[2],wx);
	this->ComputeDisplayToWorld(0.0,1.0,disp_obj_center[2],wy);
	
	for(int i=0; i<3; i++)
	{
		ex[i] = wx[i] - wc[i];
		ey[i] = wy[i] - wc[i];
	}
	
	vtkMath::Cross(ex,ey,ez);
	vtkMath::Normalize(ez);
	
	motion_vector[0] = d*ez[0];
	motion_vector[1] = d*ez[1];
	motion_vector[2] = d*ez[2];
	//
	//  end of change
	//
	if (this->InteractionProp->GetUserMatrix() != 0)
    {
		iTransform *t;
		t = iTransform::New(); IERROR_ASSERT(t);
		t->PostMultiply();
		t->SetMatrix(this->InteractionProp->GetUserMatrix());
		t->Translate(motion_vector[0], motion_vector[1], motion_vector[2]);
		this->InteractionProp->GetUserMatrix()->DeepCopy(t->GetMatrix());
		t->Delete();
    }
	else
    {
		this->InteractionProp->AddPosition(motion_vector);
    }
	
	if (this->AutoAdjustCameraClippingRange)
    {
		this->CurrentRenderer->ResetCameraClippingRange();
    }
	
	rwi->Render();
}
//
//  Rotating - a copy of vtkInteractorStyleTrackballActor::Rotate() with 
//  no changes
//
void iMeasuringBoxInteractorStyle::Rotate()
{
  this->InteractionProp = mBox;
  if (this->CurrentRenderer == 0 || this->InteractionProp == 0)
    {
    return;
    }
  
  vtkRenderWindowInteractor *rwi = this->Interactor;
  vtkCamera *cam = this->CurrentRenderer->GetActiveCamera();

  // First Get the origin of the assembly

  double *obj_center = this->InteractionProp->GetCenter();
  
  // GetLength Gets the length of the diagonal of the bounding box

  double boundRadius = mBox->GetLength() * 0.5;
//  double boundRadius = this->InteractionProp->GetLength() * 0.5;

  // Get the view up and view right vectors

  double view_up[3], view_look[3], view_right[3];

  cam->OrthogonalizeViewUp();
  cam->ComputeViewPlaneNormal();
  cam->GetViewUp(view_up);
  vtkMath::Normalize(view_up);
  cam->GetViewPlaneNormal(view_look);
  vtkMath::Cross(view_up, view_look, view_right);
  vtkMath::Normalize(view_right);
  
  // Get the furtherest point from object position+origin

  double outsidept[3];

  outsidept[0] = obj_center[0] + view_right[0] * boundRadius;
  outsidept[1] = obj_center[1] + view_right[1] * boundRadius;
  outsidept[2] = obj_center[2] + view_right[2] * boundRadius;
  
  // Convert them to display coord

  double disp_obj_center[3];

  this->ComputeWorldToDisplay(obj_center[0], obj_center[1], obj_center[2], 
                              disp_obj_center);

  this->ComputeWorldToDisplay(outsidept[0], outsidept[1], outsidept[2], 
                              outsidept);
  
  double radius = sqrt(vtkMath::Distance2BetweenPoints(disp_obj_center,
                                                       outsidept));
  double nxf = 
    ((double)rwi->GetEventPosition()[0] - (double)disp_obj_center[0]) / radius;

  double nyf = 
    ((double)rwi->GetEventPosition()[1] - (double)disp_obj_center[1]) / radius;

  double oxf = 
    ((double)rwi->GetLastEventPosition()[0] - (double)disp_obj_center[0]) / radius;

  double oyf = 
    ((double)rwi->GetLastEventPosition()[1] - (double)disp_obj_center[1]) / radius;

  if (((nxf * nxf + nyf * nyf) <= 1.0) &&
      ((oxf * oxf + oyf * oyf) <= 1.0))
    {
    double newXAngle = asin(nxf) * vtkMath::RadiansToDegrees();
    double newYAngle = asin(nyf) * vtkMath::RadiansToDegrees();
    double oldXAngle = asin(oxf) * vtkMath::RadiansToDegrees();
    double oldYAngle = asin(oyf) * vtkMath::RadiansToDegrees();
    
    double scale[3];
    scale[0] = scale[1] = scale[2] = 1.0;

    double **rotate = new double*[2]; IERROR_ASSERT(rotate);

    rotate[0] = new double[4]; IERROR_ASSERT(rotate[0]);
    rotate[1] = new double[4]; IERROR_ASSERT(rotate[1]);
    
    rotate[0][0] = newXAngle - oldXAngle;
    rotate[0][1] = view_up[0];
    rotate[0][2] = view_up[1];
    rotate[0][3] = view_up[2];
    
    rotate[1][0] = oldYAngle - newYAngle;
    rotate[1][1] = view_right[0];
    rotate[1][2] = view_right[1];
    rotate[1][3] = view_right[2];
    
    
    this->Prop3DTransform(this->InteractionProp,
                          obj_center,
                          2, 
                          rotate, 
                          scale);
    
    delete [] rotate[0];
    delete [] rotate[1];
    delete [] rotate;
    
    if (this->AutoAdjustCameraClippingRange)
      {
      this->CurrentRenderer->ResetCameraClippingRange();
      }

    rwi->Render();
    }
}
  

void iMeasuringBoxInteractorStyle::OnChar() 
{
	this->FindPokedRenderer(0, 0);
	this->FindPickedActor(0, 0);

	if (this->CurrentRenderer == 0 || this->InteractionProp == 0)
    {
		return;
    }
	
	vtkRenderWindowInteractor *rwi = this->Interactor;
	
	float s = mBox->GetSize();
	float o = mBox->GetOpacity();

	// catch keycodes
	switch (rwi->GetKeyCode())
	{
    case 'a':
    case 'A': { mBox->SetSize(s*0.95); break; }
    case 'z':
    case 'Z': { mBox->SetSize(s/0.95); break; }
    case 's':
    case 'S': { mBox->SetOpacity(o*0.8); break; }
    case 'x':
    case 'X': { mBox->SetOpacity(o/0.8); break; }
	case 'c':
	case 'C': { this->CurrentRenderer->GetActiveCamera()->SetFocalPoint(this->mBox->GetPosition()); break; }
    case 'u':
    case 'U': { rwi->UserCallback(); break; }
	}

	rwi->Render();
}

