/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// A small set of common functionality for all IFrIT viewization objects
//

#ifndef IOBJECT_H
#define IOBJECT_H


#include "icolor.h"
#include "iposition.h"
#include "istring.h"


template<class T> class iArray;


class iObjectKeyHelp;
class iObjectKeyPointer;
class iObjectTypeHelp;
class iObjectTypePointer;
class iPiecewiseFunction;
class iRangeCollection;


class iObjectType
{

	friend class iObject;
	friend class iObjectTypeRegistry;

public:

	enum ClassId
	{
		_Helper,
		_Visual,
		_Module,
		_Any
	};

	const iString& FullName() const;
	const iString& ShortName() const;
	inline ClassId Class() const { return mClass; }

	bool IsMatch(const iString &p) const;
	bool IsMatch(const iObjectType &p) const;

	inline const iObjectTypeHelp* GetHelp() const { return mHelp; }

    bool operator==(const iObjectType &s) const;
    bool operator<(const iObjectType &s) const;

	bool IsHidden() const;

private:

	iObjectType(){}
	iObjectType(ClassId c, const char *fullname, const char *shortname);
	~iObjectType();

	ClassId mClass;
	iString mFullName, mShortName;
	iObjectTypeHelp* mHelp;
};


class iObjectTypeRegistry
{

public:

	static const iObjectType* CreateType(iObjectType::ClassId c, const char *fp, const char *sp);

	static const iObjectType* FindType(const iString &str, bool hidden = false);

	static void InitTraversal(iObjectType::ClassId c = iObjectType::_Any);
	static const iObjectType* GetNextType(bool hidden = false);

private:

	//
	//  List cannot be a static variable because it must be created when it is first time
	//  requested.
	//
	static iArray<iObjectTypePointer>& List();

	static int mTraversalLocation;
	static iObjectType::ClassId mTraversalClass;
};


class iObjectKey
{

	friend class iObject;
	friend class iObjectKeyRegistry;
	friend class iShell;

public:

	enum Arg
	{
		_Any, _OffsetInt, _Int, _Bool, _Float, _Double, _Color, _String
	};

	const iString PrefixedFullName(int index = -1) const;
	const iString PrefixedShortName(int index = -1) const;
	const iString UnprefixedFullName(int index = -1) const;
	const iString UnprefixedShortName(int index = -1) const;

	const iString PrefixedKey(int index = -1) const;
	const iString UnprefixedKey(int index = -1) const;

	const iString HelpEntryKey() const;
	const iString PackingKey(int index = -1, bool prefixed = false) const;
	const iString RetypedKey(const iObjectType &type, int index = -1) const;

	const iObjectType& Type() const;

	const Arg Argument() const;
	const int Dimension() const;

	bool IsArgumentCompatible(Arg a) const;
	bool IndexHelper(Arg a, int n, int index, int &imin, int &imax) const;
 
	inline bool IsIndexed() const { return (mDim > 1); }

	inline const iObjectKeyHelp* GetHelp() const { return mHelp; }

	static const iString& PrefixSeparator();
	static const iString& FieldSeparator();
	static const iString& SubSeparator();
	static const iString& SubSubSeparator();
	static const iString& Delimiter();
	static const iString& LeftBracket();
	static const iString& RightBracket();
	static const iString& SpecialSeparator();

    bool operator==(const iObjectKey &s) const;
    bool operator<(const iObjectKey &s) const;

	bool IsHidden() const;

private:

	//
	//  The constructor only calls Define function
	//
	iObjectKey(const iObjectType *type, const char *fk, const char *sk, Arg a, int d, const iObjectType* parent = 0);
	~iObjectKey();

	//
	//  Allow the key to be redefined by the registry - this is a true constructor
	//
	void Define(const iObjectType *type, const char *fk, const char *sk, Arg a, int d, const iObjectType* parent = 0);

	Arg mArg;
	int mDim;
	const iObjectType *mType, *mParent;
	iString mShortKey, mFullKey; // formal keys
	iString mSearchableShortKey, mSearchableFullKey; // searchable keys
	iString mSearchableIndexedShortKey, mSearchableIndexedFullKey; // searchable keys with index
	iString mPrefixedShortKey, mPrefixedFullKey; // prefixed keys

	iObjectKeyHelp* mHelp;

	static const char mLastLetterOfDelimiter, mLastLetterOfPrefixSeparator;
	static bool mUseShortKeys;
};


class iObjectKeyRegistry
{

public:

	static const iObjectKey* CreateKey(const iObjectType &type, const char *fk, const char *sk, iObjectKey::Arg a, int d, const iObjectType* parent = 0, const iObjectKey* helpOwner = 0);

	static const iObjectKey& GetTempKey(const iObjectType &type, const iString &fk, int id = -1);
	static void ReleaseTempKey(const iObjectKey &key);

	static const iObjectKey* FindKey(const iString &str, bool hidden = false);

	static void InitTraversal(const iObjectType *type = 0);
	static const iObjectKey* GetNextKey(bool hidden = false);

private:

	//
	//  List cannot be a static variable because it must be created when it is first time
	//  requested.
	//
	static iArray<iObjectKeyPointer>& List();
	static iArray<iObjectKey*>& TempKeys();

	static int mTraversalLocation;
	static const iObjectType *mTraversalType;
};

#define IOBJECT_DECLARE_SELF(_self_,_parent_) \
	static _self_* SafeDownCast(iObject *self) { if(self!=0 && self->IsOfType(#_self_)) return (_self_*)self; else; return 0; } \
	static _self_* RequiredCast(iObject *self) { if(self==0 || !self->IsOfType(#_self_)) iObject::ReportBadCast(#_self_,__FILE__,__LINE__); return (_self_*)self; } \
	virtual bool IsOfType(const iString &name) const { static const iString thisName(#_self_); if(name == thisName) return true; else return this->_parent_::IsOfType(name); } \
	virtual const iString& GetThisClassName() const { static const iString thisName(#_self_); return thisName; }

#define IOBJECT_DECLARE_PACKUNPACK(_type1_,_type2_) \
	void PackValue(iString &s, const iObjectKey &key, const _type1_* val, int num, int index = -1, bool prefixed = false) const; \
	inline void PackValue(iString &s, const iObjectKey &key, _type2_ val, int index = -1, bool prefixed = false) const { if(index >= 0) PackValue(s,key,&val-index,1,index,prefixed); else PackValue(s,key,&val,1,index,prefixed); } \
	bool UnPackValue(const iString &s, const iObjectKey &key, _type1_* val, int num); \
	inline bool UnPackValue(const iString &s, const iObjectKey &key, _type1_ &val){ return !key.IsIndexed() && UnPackValue(s,key,&val,1); }  

class iObject
{
	
public:

	virtual const iString& GetThisClassName() const = 0;
	virtual bool IsOfType(const iString &name) const;

	virtual void PackCompleteState(iString &s) const;
	virtual void UnPackCompleteState(const iString &s);

	void PackState(iString &s) const;
	void UnPackState(const iString &s);
	virtual void CopyState(const iObject *p);

	IOBJECT_DECLARE_PACKUNPACK(int,int);
	IOBJECT_DECLARE_PACKUNPACK(bool,bool);
	IOBJECT_DECLARE_PACKUNPACK(float,float);
	IOBJECT_DECLARE_PACKUNPACK(double,double);
	IOBJECT_DECLARE_PACKUNPACK(iColor,iColor);
	IOBJECT_DECLARE_PACKUNPACK(iString,const iString&);

	//
	//  Packing helpers
	//
	void PackValuePiecewiseFunction(iString &s, const iObjectKey &key, const iPiecewiseFunction **f, int num, int index = -1, bool prefixed = false) const;
	bool UnPackValuePiecewiseFunction(const iString &s, const iObjectKey &key, iPiecewiseFunction **f, int num);
	inline void PackValuePiecewiseFunction(iString &s, const iObjectKey &key, const iPiecewiseFunction* val, int index = -1, bool prefixed = false) const { if(index >= 0) PackValuePiecewiseFunction(s,key,&val-index,1,index,prefixed); else PackValuePiecewiseFunction(s,key,&val,1,index,prefixed); }
	inline bool UnPackValuePiecewiseFunction(const iString &s, const iObjectKey &key, iPiecewiseFunction* &val){ return !key.IsIndexed() && UnPackValuePiecewiseFunction(s,key,&val,1); } 

	void PackValueRangeCollection(iString &s, const iObjectKey &key, const iRangeCollection **f, int num, int index = -1, bool prefixed = false) const;
	bool UnPackValueRangeCollection(const iString &s, const iObjectKey &key, iRangeCollection **f, int num);
	inline void PackValueRangeCollection(iString &s, const iObjectKey &key, const iRangeCollection* val, int index = -1, bool prefixed = false) const { if(index >= 0) PackValueRangeCollection(s,key,&val-index,1,index,prefixed); else PackValueRangeCollection(s,key,&val,1,index,prefixed); }
	inline bool UnPackValueRangeCollection(const iString &s, const iObjectKey &key, iRangeCollection* &val){ return !key.IsIndexed() && UnPackValueRangeCollection(s,key,&val,1); } 

	void PackValuePosition(iString &s, const iObjectKey &key, const iPosition &val, int index = -1, bool prefixed = false) const;
	bool UnPackValuePosition(const iString &s, const iObjectKey &key, iPosition &val); 
	void PackValueDistance(iString &s, const iObjectKey &key, const iDistance &val, int index = -1, bool prefixed = false) const;
	bool UnPackValueDistance(const iString &s, const iObjectKey &key, iDistance &val); 

	int FindUnPackKey(const iString &str, const iObjectKey &key, int &index) const;

	inline bool UnPackedSomething() const { return mUnPackedSomething; }
	inline void ClearUnPackedSomethingFlag() { mUnPackedSomething = false; }

	virtual void ClearCache();

	static void ReportMissingKeys(bool s);

protected:
	
	iObject();
	virtual ~iObject(); 

	virtual void PackStateBody(iString &s) const = 0;
	virtual void UnPackStateBody(const iString &s) = 0;

	int FindFullMatch(const iString &str, const iString &pat) const;

	static const iString mSlash;
	bool mUnPackedSomething;
	mutable bool mCacheInvalid;
	mutable iString mCache;

	static void ReportBadCast(const char *self, const char *file, int line);
	static void ReportAMissingKey(const iObjectKey &key);

private:

	static int mMissingKeysStack;
};


//
//  Helper macros: pointers to keys have to be global so that they exit
//  when the application starts. If they are declared as static members inside 
//  functions, then they are created "on-demand", and key querying by strings
//  would not work. However, the opposite is true for types, since they are used
//  in key creation.
//
#define IOBJECT_DEFINE_TYPE(_object_,_fname_,_sname_,_class_) \
	const iObjectType& _object_::Type() \
	{ \
		static const iObjectType *pval = iObjectTypeRegistry::CreateType(_class_,#_fname_,#_sname_); \
		return *pval; \
	}

#define IOBJECT_DEFINE_KEY(_object_,_fkey_,_skey_,_arg_,_dim_) \
	static const iObjectKey *__p##_object_##_fkey_##_arg_ = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_,#_skey_,iObjectKey::_##_arg_,_dim_); \
	const iObjectKey& _object_::Key##_fkey_() \
	{ \
		return *__p##_object_##_fkey_##_arg_; \
	}

#define IOBJECT_DEFINE_EXTRINSIC_KEY(_owner_,_object_,_fkey_,_skey_,_arg_,_dim_) \
	static const iObjectKey *__p##_object_##_fkey_##_arg_ = iObjectKeyRegistry::CreateKey(_owner_::Type(),#_fkey_,#_skey_,iObjectKey::_##_arg_,_dim_); \
	const iObjectKey& _object_::Key##_fkey_() \
	{ \
		return *__p##_object_##_fkey_##_arg_; \
	}

#define IOBJECT_DEFINE_INHERITED_KEY(_parent_,_object_,_fkey_,_skey_,_arg_,_dim_) \
	static const iObjectKey *__p##_object_##_fkey_##_arg_ = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_,#_skey_,iObjectKey::_##_arg_,_dim_,&_parent_::Type()); \
	const iObjectKey& _object_::Key##_fkey_() \
	{ \
		return *__p##_object_##_fkey_##_arg_; \
	}

#define IOBJECT_DECLARE_GETSET(_fun_,_var_,_type_) \
	virtual void Set##_fun_(_type_); \
    inline _type_ Get##_fun_() const { return _var_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_GETSET1(_fun_,_type_) \
	virtual void Set##_fun_(_type_); \
	inline _type_ Get##_fun_() const { return m##_fun_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_GETSET2(_fun_,_type_) \
	virtual void Set##_fun_(_type_); \
    _type_ Get##_fun_() const; \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_GET(_fun_,_var_,_type_) \
    inline _type_ Get##_fun_() const { return _var_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_GET1(_fun_,_type_) \
	inline _type_ Get##_fun_() const { return m##_fun_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_GET2(_fun_,_type_) \
    _type_ Get##_fun_() const; \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_EXT_GETSET(_ext_,_fun_,_var_,_type_) \
	virtual void Set##_fun_(_type_); \
    inline _type_ Get##_fun_() const { return _ext_##_var_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_EXT_GETSET1(_ext_,_fun_,_type_) \
	virtual void Set##_fun_(_type_); \
    inline _type_ Get##_fun_() const { return _ext_##_fun_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_EXT_GET(_ext_,_fun_,_var_,_type_) \
    inline _type_ Get##_fun_() const { return _ext_##_var_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DECLARE_EXT_GET1(_ext_,_fun_,_type_) \
    inline _type_ Get##_fun_() const { return _ext_##_fun_; } \
	static const iObjectKey& Key##_fun_()

#define IOBJECT_DEFINE_DISTANCE_KEY(_object_,_fkey_,_skey_) \
	static const iObjectKey *__p##_object_##_fkey_##Double = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_,#_skey_,iObjectKey::_Double,1); \
	const iObjectKey& _object_::Key##_fkey_(bool opengl) \
	{ \
		static const iObjectKey *ogl = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_"GL","-"#_skey_,iObjectKey::_Double,1,0,__p##_object_##_fkey_##Double); \
		if(opengl) return *ogl; else return *__p##_object_##_fkey_##Double; \
	}

#define IOBJECT_DEFINE_INHERITED_DISTANCE_KEY(_parent_,_object_,_fkey_,_skey_) \
	static const iObjectKey *__p##_object_##_fkey_##Double = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_,#_skey_,iObjectKey::_Double,1,&_parent_::Type()); \
	const iObjectKey& _object_::Key##_fkey_(bool opengl) \
	{ \
		static const iObjectKey *ogl = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_"GL","-"#_skey_,iObjectKey::_Double,1,0,__p##_object_##_fkey_##Double); \
		if(opengl) return *ogl; else return *__p##_object_##_fkey_##Double; \
	}

#define IOBJECT_DEFINE_POSITION_KEY(_object_,_fkey_,_skey_) \
	static const iObjectKey *__p##_object_##_fkey_##Double = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_,#_skey_,iObjectKey::_Double,3); \
	const iObjectKey& _object_::Key##_fkey_(bool opengl) \
	{ \
		static const iObjectKey *ogl = iObjectKeyRegistry::CreateKey(_object_::Type(),#_fkey_"GL","-"#_skey_,iObjectKey::_Double,3,0,__p##_object_##_fkey_##Double); \
		if(opengl) return *ogl; else return *__p##_object_##_fkey_##Double; \
	}

#define IOBJECT_DECLARE_GETSET_POSITION(_fun_,_var_) \
	virtual void Set##_fun_(const iPosition&); \
    inline const iPosition& Get##_fun_() const { return _var_; } \
	static const iObjectKey& Key##_fun_(bool opengl = false)

#define IOBJECT_DECLARE_GETSET_DISTANCE(_fun_,_var_) \
	virtual void Set##_fun_(const iDistance&); \
    inline const iDistance& Get##_fun_() const { return _var_; } \
	static const iObjectKey& Key##_fun_(bool opengl = false)

#endif // IOBJECT_H

